/*
 * tangle-actor.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-actor.h"
#include "tangle-action.h"
#include "tangle-properties.h"
#include "tangle-misc.h"
#include "marshalers.h"
#include <gobject/gvaluecollector.h>
#include <string.h>

/**
 * SECTION:tangle-actor
 * @Short_description: A base class for actors that display content
 * @Title: TangleActor
 */
 
G_DEFINE_ABSTRACT_TYPE(TangleActor, tangle_actor, CLUTTER_TYPE_ACTOR);

struct _TangleActorPrivate {
	gfloat transition_move_x;
	gfloat transition_move_y;
	gfloat transition_scale_x;
	gfloat transition_scale_y;
	gulong transition_mode;
	guint transition_duration;

	gfloat picking_scale_x;
	gfloat picking_scale_y;
	gfloat picking_scale_center_x;
	gfloat picking_scale_center_y;
	ClutterGravity picking_scale_gravity;

	gulong paint_event_handler_id;
	gulong pick_event_handler_id;

	GHashTable* animations;

	TangleSpacing margin;
	gfloat max_width;
	gfloat max_height;
	gdouble alignment_x;
	gdouble alignment_y;

	ClutterActorBox aligned_allocation;
	gdouble interactive_size_multiplier;

	TangleAction* button_press_action;
	TangleAction* button_release_action;
	TangleAction* enter_action;
	TangleAction* leave_action;
	TangleAction* motion_action;
	TangleAction* scroll_action;
	
	GList* ancestor_events;
	GList* descendant_events;
	
	ClutterAnimation* hiding_animation;

	guint transition_scale_children : 1;
	guint margin_set : 1;
	guint max_width_set : 1;
	guint max_height_set : 1;
	guint interacting : 1;
	guint interactive_size_multiplier_set : 1;
	guint propagating_events : 1;
	guint hide_all : 1;
	guint destroy : 1;
};

enum {
	PROP_0,
	PROP_TRANSITION_MOVE_X,
	PROP_TRANSITION_MOVE_Y,
	PROP_TRANSITION_SCALE_X,
	PROP_TRANSITION_SCALE_Y,
	PROP_TRANSITION_MODE,
	PROP_TRANSITION_DURATION,
	PROP_PICKING_SCALE_X,
	PROP_PICKING_SCALE_Y,
	PROP_PICKING_SCALE_CENTER_X,
	PROP_PICKING_SCALE_CENTER_Y,
	PROP_PICKING_SCALE_GRAVITY,
	PROP_MARGIN,
	PROP_MAX_WIDTH,
	PROP_MAX_WIDTH_SET,
	PROP_MAX_HEIGHT,
	PROP_MAX_HEIGHT_SET,
	PROP_ALIGNMENT_X,
	PROP_ALIGNMENT_Y,
	PROP_ALIGNMENT_MODE,
	PROP_ALIGNED_ALLOCATION,
	PROP_INTERACTING,
	PROP_INTERACTIVE_SIZE_MULTIPLIER,
	PROP_INTERACTIVE_SIZE_MULTIPLIER_SET,
	PROP_STYLE,
	PROP_BUTTON_PRESS_ACTION,
	PROP_BUTTON_RELEASE_ACTION,
	PROP_ENTER_ACTION,
	PROP_LEAVE_ACTION,
	PROP_MOTION_ACTION,
	PROP_SCROLL_ACTION
};

enum {
	ANIMATE_TRANSITION,
	LAST_SIGNAL
};

static void on_animation_completed(ClutterAnimation* animation, gpointer user_data);
static void set_max_width(TangleActor* actor, gfloat max_width);
static void set_max_height(TangleActor* actor, gfloat max_height);
static void set_picking_scale_center_by_gravity(TangleActor* actor);
static void set_picking_scale_gravity(TangleActor* actor, ClutterGravity gravity);
static gboolean handle_queued_events(TangleActor* actor);
static void unset_transition_scale_children(TangleActor* actor);
static void on_hiding_animation_completed(TangleActor* actor);

static guint signals[LAST_SIGNAL] = { 0 };

ClutterAnimation* tangle_actor_animate(TangleActor* actor, gulong mode, guint duration, const gchar* first_property_name, ...) {
	ClutterAnimation* animation;
	GObjectClass *klass;
	va_list args;
	const gchar* property_name;
	
	animation = clutter_animation_new();
	clutter_animation_set_object(animation, G_OBJECT(actor));
	clutter_animation_set_mode(animation, mode);
	clutter_animation_set_duration(animation, duration);
	g_signal_connect(animation, "completed", G_CALLBACK(on_animation_completed), actor);

	klass = G_OBJECT_GET_CLASS(actor);

	va_start(args, first_property_name);
	for (property_name = first_property_name; property_name; property_name = va_arg(args, gchar*)) {
		GParamSpec *param_spec;
		GValue value = { 0 };
		gchar* error;
		
		if ((param_spec = g_object_class_find_property(klass, property_name))) {
			g_value_init(&value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
			error = NULL;
			G_VALUE_COLLECT(&value, args, 0, &error);
			if (!error) {
				ClutterAnimation* existing_animation;
				
				if ((existing_animation = g_hash_table_lookup(actor->priv->animations, property_name))) {
					clutter_animation_unbind_property(existing_animation, property_name);
					g_hash_table_remove(actor->priv->animations, property_name);
				}
				clutter_animation_bind(animation, property_name, &value);
				g_hash_table_insert(actor->priv->animations, g_strdup(property_name), g_object_ref(animation)); // gobject_ref_sink does not work with 0.9.2!
			} else {
				g_warning("%s: %s", G_STRLOC, error);
				g_free(error);
				break;
			}
			g_value_unset(&value);
		} else {
			g_warning ("Cannot bind property '%s'.", property_name);
			break;
		}
	}
	va_end(args);

	clutter_timeline_start(clutter_animation_get_timeline(animation));

	return animation;
}

gboolean tangle_actor_should_transition_scale_children(TangleActor* actor) {

	return actor->priv->transition_scale_children;
}

void tangle_actor_show(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterAnimation* animation = NULL;

	if (TANGLE_ACTOR_IS_HIDING(actor) && !actor->priv->destroy) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);

		clutter_animation_completed(actor->priv->hiding_animation);
		g_object_unref(actor->priv->hiding_animation);
		actor->priv->hiding_animation = NULL;

		g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
		actor->priv->transition_scale_children = TRUE;
		g_signal_emit(actor, signals[ANIMATE_TRANSITION], 0, NULL, current_allocation, &animation);
		if (animation) {
			g_signal_connect_swapped(animation, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		}
		
		/* TODO: if hiding_animation should have done hide_all... */
	}

	if (!actor->priv->destroy && !CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_show(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_hide_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterAnimation* animation = NULL;
	
	g_return_if_fail(TANGLE_IS_ACTOR(actor));
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_PAINTED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			g_signal_emit(actor, signals[ANIMATE_TRANSITION], 0, current_allocation, NULL, &animation);
		}
		if (animation) {
			actor->priv->hiding_animation = animation;
			g_object_ref(animation);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(animation, "completed", G_CALLBACK(on_hiding_animation_completed), actor);
			g_signal_connect_swapped(animation, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_hide(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	}
}

void tangle_actor_hide_all_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterAnimation* animation = NULL;
	
	actor->priv->hide_all = TRUE;
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_PAINTED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			g_signal_emit(actor, signals[ANIMATE_TRANSITION], 0, current_allocation, NULL, &animation);
		}
		if (animation) {
			actor->priv->hiding_animation = animation;
			g_object_ref(animation);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(animation, "completed", G_CALLBACK(on_hiding_animation_completed), actor);
			g_signal_connect_swapped(animation, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_hide_all(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	} else if (!CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_hide_all(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_destroy_animated(TangleActor* actor) {
	ClutterActorBox* current_allocation;
	ClutterAnimation* animation = NULL;
	
	actor->priv->destroy = TRUE;
	
	if (CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor)) && !TANGLE_ACTOR_IS_HIDING(actor)) {
		if (TANGLE_ACTOR_IS_PAINTED(actor)) {
			g_object_get(G_OBJECT(actor), "allocation", &current_allocation, NULL);
			actor->priv->transition_scale_children = TRUE;
			g_signal_emit(actor, signals[ANIMATE_TRANSITION], 0, current_allocation, NULL, &animation);
		}
		if (animation) {
			actor->priv->hiding_animation = animation;
			g_object_ref(animation);
			TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_HIDING);

			g_signal_connect_swapped(animation, "completed", G_CALLBACK(on_hiding_animation_completed), actor);
			g_signal_connect_swapped(animation, "completed", G_CALLBACK(unset_transition_scale_children), actor);
		} else {
			clutter_actor_destroy(CLUTTER_ACTOR(actor));
			actor->priv->transition_scale_children = FALSE;
		}
	} else if (!CLUTTER_ACTOR_IS_VISIBLE(CLUTTER_ACTOR(actor))) {
		clutter_actor_destroy(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_get_margin(TangleActor* actor, TangleSpacing* spacing) {
	*spacing = actor->priv->margin;
}
	
void tangle_actor_set_margin(TangleActor* actor, const TangleSpacing* spacing) {
	if (!actor->priv->margin_set || !tangle_spacing_equal(&actor->priv->margin, spacing)) {
		g_object_freeze_notify(G_OBJECT(actor));
		
		if (actor->priv->margin_set) {
			actor->priv->aligned_allocation.x1 -= actor->priv->margin.left;
			actor->priv->aligned_allocation.y1 -= actor->priv->margin.top;
			actor->priv->aligned_allocation.x2 += actor->priv->margin.right;
			actor->priv->aligned_allocation.y2 += actor->priv->margin.bottom;
		}
		actor->priv->margin = *spacing;
		actor->priv->aligned_allocation.x1 += actor->priv->margin.left;
		actor->priv->aligned_allocation.y1 += actor->priv->margin.top;
		actor->priv->aligned_allocation.x2 -= actor->priv->margin.right;
		actor->priv->aligned_allocation.y2 -= actor->priv->margin.bottom;
		g_object_notify(G_OBJECT(actor), "margin");
		g_object_notify(G_OBJECT(actor), "aligned-allocation");
		tangle_actor_set_margin_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

gboolean tangle_actor_get_margin_set(TangleActor* actor) {

	return actor->priv->margin_set;
}

void tangle_actor_set_margin_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->margin_set != is_set) {
		g_object_freeze_notify(G_OBJECT(actor));

		actor->priv->margin_set = is_set;
		g_object_notify(G_OBJECT(actor), "margin-set");

		/* TODO: Calculate aligned allocation */

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

void tangle_actor_get_max_size(TangleActor* actor, gfloat* max_width_p, gfloat* max_height_p) {
	if (actor->priv->max_width_set) {
		*max_width_p = actor->priv->max_width_set;
	} else {
		*max_width_p = 0.0;
	}
	if (actor->priv->max_height_set) {
		*max_height_p = actor->priv->max_height_set;
	} else {
		*max_height_p = 0.0;
	}
}

void tangle_actor_set_max_size(TangleActor* actor, gfloat max_width, gfloat max_height) {
	g_object_freeze_notify(G_OBJECT(actor));
	
	set_max_width(actor, max_width);
	set_max_height(actor, max_height);
	
	g_object_thaw_notify(G_OBJECT(actor));
}

gdouble tangle_actor_get_alignment_x(TangleActor* actor) {

	return actor->priv->alignment_x;
}

void tangle_actor_set_alignment_x(TangleActor* actor, gdouble alignment_x) {
	if (actor->priv->alignment_x != alignment_x) {
		actor->priv->alignment_x = alignment_x;
		g_object_notify(G_OBJECT(actor), "alignment-x");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

gdouble tangle_actor_get_alignment_y(TangleActor* actor) {

	return actor->priv->alignment_y;
}

void tangle_actor_set_alignment_y(TangleActor* actor, gdouble alignment_y) {
	if (actor->priv->alignment_y != alignment_y) {
		actor->priv->alignment_y = alignment_y;
		g_object_notify(G_OBJECT(actor), "alignment-y");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_get_aligned_allocation(TangleActor* actor, ClutterActorBox* aligned_allocation) {
	*aligned_allocation = actor->priv->aligned_allocation;
}


gboolean tangle_actor_get_interacting(TangleActor* actor) {

	return actor->priv->interacting;
}

void tangle_actor_set_interacting(TangleActor* actor, gboolean interacting) {
	if (actor->priv->interacting != interacting) {
		actor->priv->interacting = interacting;
		g_object_notify(G_OBJECT(actor), "interacting");
		clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
	}
}

void tangle_actor_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	gfloat min_width;
	gboolean min_width_set;
	gfloat natural_width;
	gboolean natural_width_set;
	TangleActorClass* actor_class;

	g_object_get(actor, "min-width", &min_width, "min-width-set", &min_width_set, "natural-width", &natural_width, "natural-width-set", &natural_width_set, NULL);
	if (min_width_set && min_width_p) {
		*min_width_p = min_width;
		min_width_p = NULL;
	
	}
	if (natural_width_set && natural_width_p) {
		*natural_width_p = natural_width;
		natural_width_p = NULL;	
	}
	
	if (min_width_p || natural_width_p || max_width_p) {
		actor_class = TANGLE_ACTOR_GET_CLASS(actor);
		if (actor_class->get_preferred_width) {
			actor_class->get_preferred_width(actor, for_height, interacting, min_width_p, natural_width_p, max_width_p);

			if (natural_width_p && min_width_p && *natural_width_p < *min_width_p) {
				*natural_width_p = *min_width_p;
			}
			if (max_width_p && natural_width_p && *max_width_p != 0 && *max_width_p < *natural_width_p) {
				*max_width_p = *natural_width_p;
			}
			if (actor->priv->interacting && actor->priv->interactive_size_multiplier_set) {
				if (min_width_p) {
					*min_width_p *= actor->priv->interactive_size_multiplier;
				}
				if (natural_width_p) {
					*natural_width_p *= actor->priv->interactive_size_multiplier;
				}
				if (max_width_p) {
					*max_width_p *= actor->priv->interactive_size_multiplier;
				}
			}
		} else {
			if (min_width_p) {
				*min_width_p = 0.0;
			}
			if (natural_width_p) {
				*natural_width_p = 0.0;
			}
			if (max_width_p) {
				*max_width_p = 0.0;
			}
		}

		if (actor->priv->margin_set) {
			if (*min_width_p) {
				*min_width_p += actor->priv->margin.left + actor->priv->margin.right;
			}
			if (*natural_width_p) {
				*natural_width_p += actor->priv->margin.left + actor->priv->margin.right;
			}
		}
	}
}

void tangle_actor_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	gfloat min_height;
	gboolean min_height_set;
	gfloat natural_height;
	gboolean natural_height_set;
	TangleActorClass* actor_class;

	g_object_get(actor, "min-height", &min_height, "min-height-set", &min_height_set, "natural-height", &natural_height, "natural-height-set", &natural_height_set, NULL);
	if (min_height_set && min_height_p) {
		*min_height_p = min_height;
		min_height_p = NULL;
	
	}
	if (natural_height_set && natural_height_p) {
		*natural_height_p = natural_height;
		natural_height_p = NULL;	
	}
	
	if (min_height_p || natural_height_p || max_height_p) {
		actor_class = TANGLE_ACTOR_GET_CLASS(actor);
		if (actor_class->get_preferred_height) {
			actor_class->get_preferred_height(actor, for_width, interacting, min_height_p, natural_height_p, max_height_p);

			if (natural_height_p && min_height_p && *natural_height_p < *min_height_p) {
				*natural_height_p = *min_height_p;
			}
			if (max_height_p && natural_height_p && *max_height_p != 0 && *max_height_p < *natural_height_p) {
				*max_height_p = *natural_height_p;
			}
			if (actor->priv->interacting && actor->priv->interactive_size_multiplier_set) {
				if (min_height_p) {
					*min_height_p *= actor->priv->interactive_size_multiplier;
				}
				if (natural_height_p) {
					*natural_height_p *= actor->priv->interactive_size_multiplier;
				}
				if (max_height_p) {
					*max_height_p *= actor->priv->interactive_size_multiplier;
				}
			}
		} else {
			if (min_height_p) {
				*min_height_p = 0.0;
			}
			if (natural_height_p) {
				*natural_height_p = 0.0;
			}
			if (max_height_p) {
				*max_height_p = 0.0;
			}
		}

		if (actor->priv->margin_set) {
			if (*min_height_p) {
				*min_height_p += actor->priv->margin.top + actor->priv->margin.bottom;
			}
			if (*natural_height_p) {
				*natural_height_p += actor->priv->margin.top + actor->priv->margin.bottom;
			}		
		}
	}
}

void tangle_actor_get_preferred_size(TangleActor* actor, gboolean interacting, gfloat* min_width_p, gfloat* min_height_p, gfloat* natural_width_p, gfloat* natural_height_p, gfloat* max_width_p, gfloat* max_height_p) {
	ClutterRequestMode request_mode;
	gfloat natural_size;
	
	g_object_get(actor, "request-mode", &request_mode, NULL);
	if (request_mode == CLUTTER_REQUEST_HEIGHT_FOR_WIDTH) {
		tangle_actor_get_preferred_width(actor, -1, interacting, min_width_p, &natural_size, max_width_p);
		tangle_actor_get_preferred_height(actor, natural_size, interacting, min_height_p, natural_height_p, max_height_p);
		if (natural_width_p) {
			*natural_width_p = natural_size;
		}
	} else {
		tangle_actor_get_preferred_height(actor, -1, interacting, min_height_p, &natural_size, max_height_p);
		tangle_actor_get_preferred_width(actor, natural_size, interacting, min_width_p, natural_width_p, max_width_p);
		if (natural_height_p) {
			*natural_height_p = natural_size;
		}
	}

}

gdouble tangle_actor_get_interactive_size_multiplier(TangleActor* actor) {

	return actor->priv->interactive_size_multiplier;
}

void tangle_actor_set_interactive_size_multiplier(TangleActor* actor, gdouble multiplier) {
	if (actor->priv->interactive_size_multiplier != multiplier) {
		g_object_freeze_notify(G_OBJECT(actor));
		
		actor->priv->interactive_size_multiplier = multiplier;
		g_object_notify(G_OBJECT(actor), "interactive-size-multiplier");
		tangle_actor_set_interactive_size_multiplier_set(actor, TRUE);
		if (actor->priv->interacting) {
			clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
		}
		
		g_object_thaw_notify(G_OBJECT(actor));
	}
}

gboolean tangle_actor_get_interactive_size_multiplier_set(TangleActor* actor) {

	return actor->priv->interactive_size_multiplier_set;
}

void tangle_actor_set_interactive_size_multiplier_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->interactive_size_multiplier_set != is_set) {
		actor->priv->interactive_size_multiplier_set = is_set;
		g_object_notify(G_OBJECT(actor), "interactive-size-multiplier-set");
		if (actor->priv->interacting) {
			clutter_actor_queue_relayout(CLUTTER_ACTOR(actor));
		}
	}
}

void tangle_actor_send_event(TangleActor* actor, TangleEvent* event) {
	tangle_actor_handle_ancestor_event(actor, event);
	tangle_actor_handle_descendant_event(actor, event);
}

void tangle_actor_handle_ancestor_event(TangleActor* actor, TangleEvent* event) {
	if (actor->priv->propagating_events) {
		actor->priv->ancestor_events = g_list_prepend(actor->priv->ancestor_events, tangle_event_copy(event));
	} else {
		actor->priv->propagating_events = TRUE;
		TANGLE_ACTOR_GET_CLASS(actor)->handle_ancestor_event(actor, event);
		while (!handle_queued_events(actor)) {
		}
		actor->priv->propagating_events = FALSE;
	}
}

void tangle_actor_handle_descendant_event(TangleActor* actor, TangleEvent* event) {
	if (actor->priv->propagating_events) {
		actor->priv->descendant_events = g_list_prepend(actor->priv->descendant_events, tangle_event_copy(event));
	} else {
		actor->priv->propagating_events = TRUE;
		TANGLE_ACTOR_GET_CLASS(actor)->handle_descendant_event(actor, event);
		while (!handle_queued_events(actor)) {
		}
		actor->priv->propagating_events = FALSE;
	}
}

TangleSpacing* tangle_spacing_new(gfloat top, gfloat right, gfloat bottom, gfloat left) {
	TangleSpacing* spacing;
	
	spacing = (TangleSpacing*)g_slice_alloc(sizeof(TangleSpacing));
	spacing->top = top;
	spacing->right = right;
	spacing->bottom = bottom;
	spacing->left = left;

	return spacing;
}

TangleSpacing* tangle_spacing_copy(const TangleSpacing* spacing) {
	TangleSpacing* copy_of_spacing;
	
	copy_of_spacing = (TangleSpacing*)g_slice_alloc(sizeof(TangleSpacing));
	*copy_of_spacing = *spacing;
	
	return copy_of_spacing;
}

void tangle_spacing_free(TangleSpacing* spacing) {
	g_slice_free1(sizeof(TangleSpacing), spacing);
}

gboolean tangle_spacing_equal(const TangleSpacing* spacing_a, const TangleSpacing* spacing_b) {
	
	return spacing_a->top == spacing_b->top &&
	       spacing_a->right == spacing_b->right &&
	       spacing_a->bottom == spacing_b->bottom &&
	       spacing_a->left == spacing_b->left;
}

static gboolean is_value_user_data(gpointer key, gpointer value, gpointer user_data) {

	return value == user_data;
}

static void on_animation_completed(ClutterAnimation* animation, gpointer user_data) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(user_data);

	g_hash_table_foreach_remove(actor->priv->animations, is_value_user_data, animation);
	g_object_unref(animation);
}

static void set_max_width_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->max_width_set != is_set) {
		actor->priv->max_width_set = is_set;
		g_object_notify(G_OBJECT(actor), "max-width-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

static void set_max_width(TangleActor* actor, gfloat max_width) {
	if (actor->priv->max_width) {
		g_object_freeze_notify(G_OBJECT(actor));
	
		actor->priv->max_width = max_width;
		g_object_notify(G_OBJECT(actor), "max-width");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		
		set_max_width_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

static void set_max_height_set(TangleActor* actor, gboolean is_set) {
	if (actor->priv->max_height_set != is_set) {
		actor->priv->max_height_set = is_set;
		g_object_notify(G_OBJECT(actor), "max-height-set");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
	}
}

static void set_max_height(TangleActor* actor, gfloat max_height) {
	if (actor->priv->max_height) {
		g_object_freeze_notify(G_OBJECT(actor));
	
		actor->priv->max_height = max_height;
		g_object_notify(G_OBJECT(actor), "max-height");
		clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
		
		set_max_height_set(actor, TRUE);

		g_object_thaw_notify(G_OBJECT(actor));
	}
}

static void tangle_actor_override_get_preferred_width(ClutterActor* actor, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	tangle_actor_get_preferred_width(TANGLE_ACTOR(actor), for_height, TANGLE_ACTOR(actor)->priv->interacting, min_width_p, natural_width_p, NULL);
}

static void tangle_actor_override_get_preferred_height(ClutterActor* actor, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	tangle_actor_get_preferred_height(TANGLE_ACTOR(actor), for_width, TANGLE_ACTOR(actor)->priv->interacting, min_height_p, natural_height_p, NULL);
}

static void tangle_actor_allocate(ClutterActor* self, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleActor* actor;
	ClutterActorBox* old_allocation;
	ClutterActorBox* new_allocation;
	gfloat length;
	gfloat difference;
	ClutterAnimation* animation;

	actor = TANGLE_ACTOR(self);
	if (TANGLE_ACTOR_IS_PAINTED(actor)) {
		g_object_get(G_OBJECT(self), "allocation", &old_allocation, NULL);
	} else {
		old_allocation = NULL;
	}
	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->allocate(self, box, flags);
	g_object_get(G_OBJECT(self), "allocation", &new_allocation, NULL);

	actor->priv->aligned_allocation.x1 = 0.0;
	actor->priv->aligned_allocation.y1 = 0.0;
	actor->priv->aligned_allocation.x2 = new_allocation->x2 - new_allocation->x1;
	actor->priv->aligned_allocation.y2 = new_allocation->y2 - new_allocation->y1;

	if (actor->priv->margin_set) {
		actor->priv->aligned_allocation.x1 += actor->priv->margin.left;
		actor->priv->aligned_allocation.y1 += actor->priv->margin.top;
		actor->priv->aligned_allocation.x2 -= actor->priv->margin.right;
		actor->priv->aligned_allocation.y2 -= actor->priv->margin.bottom;
		if (actor->priv->aligned_allocation.x2 < 0.0) {
			actor->priv->aligned_allocation.x2 = 0.0;
		}
		if (actor->priv->aligned_allocation.y2 < 0.0) {
			actor->priv->aligned_allocation.y2 = 0.0;
		}
		if (actor->priv->aligned_allocation.x1 > actor->priv->aligned_allocation.x2) {
			actor->priv->aligned_allocation.x1 = actor->priv->aligned_allocation.x2;
		}
		if (actor->priv->aligned_allocation.y1 > actor->priv->aligned_allocation.y2) {
			actor->priv->aligned_allocation.y1 = actor->priv->aligned_allocation.y2;
		}
	}

	if (actor->priv->max_width_set) {
		length = actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1;
		difference = length - actor->priv->max_width;
		if (difference > 0) {
			actor->priv->aligned_allocation.x1 += difference * actor->priv->alignment_x;
			actor->priv->aligned_allocation.x2 = actor->priv->aligned_allocation.x1 + actor->priv->max_width;
		}
	}
	if (actor->priv->max_height_set) {
		length = actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1;
		difference = length - actor->priv->max_height;
		if (difference > 0) {
			actor->priv->aligned_allocation.y1 += difference * actor->priv->alignment_y;
			actor->priv->aligned_allocation.y2 = actor->priv->aligned_allocation.y1 + actor->priv->max_height;
		}
	}
	
	set_picking_scale_center_by_gravity(actor);

	if (!old_allocation ||
	    old_allocation->x1 != new_allocation->x1 || old_allocation->x2 != new_allocation->x2 ||
	    old_allocation->y1 != new_allocation->y1 || old_allocation->y2 != new_allocation->y2) {
		if (!old_allocation) {
			actor->priv->transition_scale_children = TRUE;
		}
		g_signal_emit(self, signals[ANIMATE_TRANSITION], 0, old_allocation, new_allocation, &animation);
		if (animation && !old_allocation) {
			g_signal_connect_swapped(animation, "completed", G_CALLBACK(unset_transition_scale_children), self);
		}
	}
}

static void tangle_actor_paint(ClutterActor* clutter_actor) {
	TangleActor* actor;
	gfloat width, height;
	TangleActorClass* actor_class;
	
	actor = TANGLE_ACTOR(clutter_actor);
	width = actor->priv->aligned_allocation.x2 - actor->priv->aligned_allocation.x1;
	height = actor->priv->aligned_allocation.y2 - actor->priv->aligned_allocation.y1;
	
	cogl_translate(actor->priv->aligned_allocation.x1, actor->priv->aligned_allocation.y1, 0.0);
	
	actor_class = TANGLE_ACTOR_GET_CLASS(actor);
	actor_class->paint_aligned(actor, width, height);
}

static void tangle_actor_pick(ClutterActor* clutter_actor, const ClutterColor* color) {
	if (!TANGLE_ACTOR_IS_HIDING(TANGLE_ACTOR(clutter_actor))) {
		CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->pick(clutter_actor, color);
	}
}

static void tangle_actor_hide(ClutterActor* clutter_actor) {
	TangleActor* actor;
	
	actor = TANGLE_ACTOR(clutter_actor);

	TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_PAINTED);

	if (TANGLE_ACTOR_IS_HIDING(actor)) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);

		/* Unref also animation's internal reference to stop it. */
		g_object_unref(actor->priv->hiding_animation);
		g_object_unref(actor->priv->hiding_animation);
		actor->priv->hiding_animation = NULL;
	}
	
	CLUTTER_ACTOR_CLASS(tangle_actor_parent_class)->hide(clutter_actor);
}

static gboolean tangle_actor_button_press_event(ClutterActor* clutter_actor, ClutterButtonEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->button_press_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "button", G_TYPE_UINT /* TODO: G_TYPE_UINT32 */, event->button, NULL);
		tangle_action_execute_full(actor->priv->button_press_action, G_OBJECT(actor), "button-press", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static gboolean tangle_actor_button_release_event(ClutterActor* clutter_actor, ClutterButtonEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->button_release_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "button", G_TYPE_UINT /* TODO: G_TYPE_UINT32 */, event->button, NULL);
		tangle_action_execute_full(actor->priv->button_release_action, G_OBJECT(actor), "button-release", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static gboolean tangle_actor_enter_event(ClutterActor* clutter_actor, ClutterCrossingEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->enter_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "related", CLUTTER_TYPE_ACTOR, event->related, NULL);
		tangle_action_execute_full(actor->priv->enter_action, G_OBJECT(actor), "enter", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static gboolean tangle_actor_leave_event(ClutterActor* clutter_actor, ClutterCrossingEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->leave_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "related", CLUTTER_TYPE_ACTOR, event->related, NULL);
		tangle_action_execute_full(actor->priv->leave_action, G_OBJECT(actor), "leave", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static gboolean tangle_actor_motion_event(ClutterActor* clutter_actor, ClutterMotionEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->motion_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "x", G_TYPE_FLOAT, event->x, "y", G_TYPE_FLOAT, event->y, NULL);
		tangle_action_execute_full(actor->priv->motion_action, G_OBJECT(actor), "motion", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static gboolean tangle_actor_scroll_event(ClutterActor* clutter_actor, ClutterScrollEvent* event) {
	TangleActor* actor;
	TangleProperties* properties;
	
	actor = TANGLE_ACTOR(clutter_actor);
	if (actor->priv->scroll_action) {
		properties = tangle_properties_new();
		tangle_properties_set(properties, "direction", CLUTTER_TYPE_SCROLL_DIRECTION, event->direction, NULL);
		tangle_action_execute_full(actor->priv->scroll_action, G_OBJECT(actor), "scroll", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static ClutterAnimation* tangle_actor_animate_transition(TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterAnimation* animation = NULL;
	ClutterActor* parent;

	if (actor->priv->transition_duration && actor->priv->transition_mode) {
		if (current_box && new_box) {
			actor->priv->transition_move_x += current_box->x1 - new_box->x1;
			actor->priv->transition_move_y += current_box->y1 - new_box->y1;
			actor->priv->transition_scale_x *= (current_box->x2 - current_box->x1) /
                                			   (new_box->x2 - new_box->x1);
			actor->priv->transition_scale_y *= (current_box->y2 - current_box->y1) /
                                			   (new_box->y2 - new_box->y1);

			animation = tangle_actor_animate(actor, actor->priv->transition_mode, actor->priv->transition_duration, "transition-move-x", 0.0, "transition-move-y", 0.0, "transition-scale-x", 1.0, "transition-scale-y", 1.0, NULL);
		} else if (current_box) {
			animation = tangle_actor_animate(actor, actor->priv->transition_mode, actor->priv->transition_duration,
			                                 "transition-move-x", (current_box->x2 - current_box->x1) / 2,
							 "transition-move-y", (current_box->y2 - current_box->y1) / 2,
							 "transition-scale-x", 0.0,
							 "transition-scale-y", 0.0,
							 NULL);
		} else if ((parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor))) &&
		           (!TANGLE_IS_ACTOR(parent) || TANGLE_ACTOR_IS_PAINTED(parent))) {
			/* If the parent is not also a new TangleActor (and thus, doing transition animation),
			 * animate self.
			 */
			actor->priv->transition_move_x = (new_box->x2 - new_box->x1) / 2;
			actor->priv->transition_move_y = (new_box->y2 - new_box->y1) / 2;
			actor->priv->transition_scale_x = 0.0;
			actor->priv->transition_scale_y = 0.0;

			animation = tangle_actor_animate(actor, actor->priv->transition_mode, actor->priv->transition_duration, "transition-move-x", 0.0, "transition-move-y", 0.0, "transition-scale-x", 1.0, "transition-scale-y", 1.0, NULL);
		}
	}

	return animation;
}

static void tangle_actor_paint_event_handler(ClutterActor* self, gpointer user_data) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(self);
	TANGLE_ACTOR_SET_FLAGS(actor, TANGLE_ACTOR_PAINTED);
	cogl_translate(actor->priv->transition_move_x, actor->priv->transition_move_y, 0);
}

static void tangle_actor_pick_event_handler(ClutterActor* self, gpointer user_data) {
	TangleActor* actor;

	actor = TANGLE_ACTOR(self);
	cogl_translate(actor->priv->transition_move_x, actor->priv->transition_move_y, 0);
	if (actor->priv->picking_scale_x != 1.0 || actor->priv->picking_scale_y != 1.0) {
		cogl_translate(actor->priv->picking_scale_center_x, actor->priv->picking_scale_center_y, 0.0);
		cogl_scale(actor->priv->picking_scale_x, actor->priv->picking_scale_y, 1.0);
		cogl_translate(-actor->priv->picking_scale_center_x, -actor->priv->picking_scale_center_y, 0.0);
	}
}

static void tangle_actor_real_handle_ancestor_event(TangleActor* actor, TangleEvent* event) {
}

static void tangle_actor_real_handle_descendant_event(TangleActor* actor, TangleEvent* event) {
	ClutterActor* parent;

	parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor));
	if (parent && TANGLE_IS_ACTOR(parent)) {
		tangle_actor_handle_descendant_event(TANGLE_ACTOR(parent), event);
	}
	
}

#define SET_ACTION(a) \
	if (actor->priv->a) { \
		g_object_unref(actor->priv->a); \
	} \
	actor->priv->a = TANGLE_ACTION(g_value_get_object(value)); \
	g_object_ref(actor->priv->a);

static void tangle_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleActor* actor;
	
	actor = TANGLE_ACTOR(object);
	switch (prop_id) {
		case PROP_TRANSITION_MOVE_X:
			actor->priv->transition_move_x = g_value_get_double(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_MOVE_Y:
			actor->priv->transition_move_y = g_value_get_double(value);
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_SCALE_X:
			if ((actor->priv->transition_scale_x = g_value_get_double(value)) < 0) {
				actor->priv->transition_scale_x = 0.0;
			}
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_SCALE_Y:
			if ((actor->priv->transition_scale_y = g_value_get_double(value)) < 0) {
				actor->priv->transition_scale_y = 0.0;
			}
			clutter_actor_queue_redraw(CLUTTER_ACTOR(actor));
			break;
		case PROP_TRANSITION_MODE:
			actor->priv->transition_mode = g_value_get_ulong(value);
			break;
		case PROP_TRANSITION_DURATION:
			actor->priv->transition_duration = g_value_get_uint(value);
			break;
		case PROP_PICKING_SCALE_X:
			actor->priv->picking_scale_x = g_value_get_float(value);
			break;
		case PROP_PICKING_SCALE_Y:
			actor->priv->picking_scale_y = g_value_get_float(value);
			break;
		case PROP_PICKING_SCALE_CENTER_X:
			actor->priv->picking_scale_center_x = g_value_get_float(value);
			set_picking_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_PICKING_SCALE_CENTER_Y:
			actor->priv->picking_scale_center_y = g_value_get_float(value);
			set_picking_scale_gravity(actor, CLUTTER_GRAVITY_NONE);
			break;
		case PROP_PICKING_SCALE_GRAVITY:
			set_picking_scale_gravity(actor, g_value_get_enum(value));
			break;
		case PROP_INTERACTING:
			tangle_actor_set_interacting(actor, g_value_get_boolean(value));
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER:
			tangle_actor_set_interactive_size_multiplier(actor, g_value_get_double(value));
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER_SET:
			tangle_actor_set_interactive_size_multiplier_set(actor, g_value_get_boolean(value));
			break;
		case PROP_BUTTON_PRESS_ACTION:
			SET_ACTION(button_press_action);
			break;
		case PROP_BUTTON_RELEASE_ACTION:
			SET_ACTION(button_press_action);
			break;
		case PROP_ENTER_ACTION:
			SET_ACTION(enter_action);
			break;
		case PROP_LEAVE_ACTION:
			SET_ACTION(leave_action);
			break;
		case PROP_MOTION_ACTION:
			SET_ACTION(motion_action);
			break;
		case PROP_SCROLL_ACTION:
			SET_ACTION(scroll_action);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void tangle_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleActor* actor;

	actor = TANGLE_ACTOR(object);
        switch (prop_id) {
		case PROP_TRANSITION_MOVE_X:
			g_value_set_double(value, actor->priv->transition_move_x);
			break;
		case PROP_TRANSITION_MOVE_Y:
			g_value_set_double(value, actor->priv->transition_move_y);
			break;
		case PROP_TRANSITION_SCALE_X:
			g_value_set_double(value, actor->priv->transition_scale_x);
			break;
		case PROP_TRANSITION_SCALE_Y:
			g_value_set_double(value, actor->priv->transition_scale_y);
			break;
		case PROP_TRANSITION_MODE:
			g_value_set_ulong(value, actor->priv->transition_mode);
			break;
		case PROP_TRANSITION_DURATION:
			g_value_set_uint(value, actor->priv->transition_duration);
			break;
		case PROP_PICKING_SCALE_X:
			g_value_set_float(value, actor->priv->picking_scale_x);
			break;
		case PROP_PICKING_SCALE_Y:
			g_value_set_float(value, actor->priv->picking_scale_y);
			break;
		case PROP_PICKING_SCALE_CENTER_X:
			g_value_set_float(value, actor->priv->picking_scale_center_x);
			break;
		case PROP_PICKING_SCALE_CENTER_Y:
			g_value_set_float(value, actor->priv->picking_scale_center_y);
			break;
		case PROP_PICKING_SCALE_GRAVITY:
			g_value_set_enum(value, actor->priv->picking_scale_gravity);
			break;
		case PROP_INTERACTING:
			g_value_set_boolean(value, actor->priv->interacting);
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER:
			g_value_set_double(value, actor->priv->interactive_size_multiplier);
			break;
		case PROP_INTERACTIVE_SIZE_MULTIPLIER_SET:
			g_value_set_boolean(value, actor->priv->interactive_size_multiplier_set);
			break;
		case PROP_BUTTON_PRESS_ACTION:
			g_value_set_object(value, actor->priv->button_press_action);
			break;
		case PROP_BUTTON_RELEASE_ACTION:
			g_value_set_object(value, actor->priv->button_release_action);
			break;
		case PROP_ENTER_ACTION:
			g_value_set_object(value, actor->priv->enter_action);
			break;
		case PROP_LEAVE_ACTION:
			g_value_set_object(value, actor->priv->leave_action);
			break;
		case PROP_MOTION_ACTION:
			g_value_set_object(value, actor->priv->motion_action);
			break;
		case PROP_SCROLL_ACTION:
			g_value_set_object(value, actor->priv->scroll_action);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
        }
}

static void tangle_actor_finalize(GObject* object) {
	TangleActor* actor;
	
	actor = TANGLE_ACTOR(object);
	g_hash_table_destroy(actor->priv->animations);
	g_signal_handler_disconnect(actor, actor->priv->paint_event_handler_id);
	g_signal_handler_disconnect(actor, actor->priv->pick_event_handler_id);

	G_OBJECT_CLASS (tangle_actor_parent_class)->finalize (object);
}

static void tangle_actor_dispose(GObject* object) {
	G_OBJECT_CLASS (tangle_actor_parent_class)->dispose (object);
}

static void tangle_actor_class_init(TangleActorClass* actor_class) {
	GObjectClass* gobject_class;
	ClutterActorClass* clutter_actor_class;
	
	gobject_class = G_OBJECT_CLASS(actor_class);
	gobject_class->finalize = tangle_actor_finalize;
	gobject_class->dispose = tangle_actor_dispose;
	gobject_class->set_property = tangle_actor_set_property;
	gobject_class->get_property = tangle_actor_get_property;

	clutter_actor_class = CLUTTER_ACTOR_CLASS(actor_class);
	clutter_actor_class->get_preferred_width = tangle_actor_override_get_preferred_width;
	clutter_actor_class->get_preferred_height = tangle_actor_override_get_preferred_height;
	clutter_actor_class->allocate = tangle_actor_allocate;
	clutter_actor_class->paint = tangle_actor_paint;
	clutter_actor_class->pick = tangle_actor_pick;
	clutter_actor_class->hide = tangle_actor_hide;
	clutter_actor_class->button_press_event = tangle_actor_button_press_event;
	clutter_actor_class->button_release_event = tangle_actor_button_press_event;
	clutter_actor_class->enter_event = tangle_actor_enter_event;
	clutter_actor_class->leave_event = tangle_actor_leave_event;
	clutter_actor_class->motion_event = tangle_actor_motion_event;
	clutter_actor_class->scroll_event = tangle_actor_scroll_event;
	
	actor_class->animate_transition = tangle_actor_animate_transition;
	actor_class->handle_ancestor_event = tangle_actor_real_handle_ancestor_event;
	actor_class->handle_descendant_event = tangle_actor_real_handle_descendant_event;

	/**
	 * TangleActor:transition-move-x:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_MOVE_X,
	                                g_param_spec_double("transition-move-x",
	                                                   "Transition move x",
	                                                   "Moves actor along x axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-move-y:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_MOVE_Y,
	                                g_param_spec_double("transition-move-y",
	                                                   "Transition move y",
	                                                   "Moves actor along y axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-x:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_X,
	                                g_param_spec_double("transition-scale-x",
	                                                   "Transition scale x",
	                                                   "Scales actor along x axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-scale-y:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_SCALE_Y,
	                                g_param_spec_double("transition-scale-y",
	                                                   "Transition scale y",
	                                                   "Scales actor along y axis when transiting from an allocation to another",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-mode:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_MODE,
	                                g_param_spec_ulong("transition-mode",
	                                                   "Transition animation mode",
	                                                   "The animation mode used when transitioning between allocations",
	                                                   0, G_MAXULONG, 0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:transition-duration:
	 */
	g_object_class_install_property(gobject_class, PROP_TRANSITION_DURATION,
	                                g_param_spec_uint("transition-duration",
	                                                  "Transition animation duration",
	                                                  "The animation duration used when transitioning between allocations",
	                                                  0, G_MAXUINT, 0,
	                                                  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-x:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_X,
	                                g_param_spec_float("picking-scale-x",
	                                                   "Picking scale x",
	                                                   "Scales the picking area of the actor along x axis",
	                                                   0, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-y:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_Y,
	                                g_param_spec_float("picking-scale-y",
	                                                   "Picking scale y",
	                                                   "Scales the picking area of the actor along y axis",
	                                                   0, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-center-x:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_Y,
	                                g_param_spec_float("picking-scale-center-x",
	                                                   "Picking scale center x",
	                                                   "The horizontal center point for picking scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-center-y:
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_Y,
	                                g_param_spec_float("picking-scale-center-y",
	                                                   "Picking scale center y",
	                                                   "The vertical center point for picking scale",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:picking-scale-gravity
	 */
	g_object_class_install_property(gobject_class, PROP_PICKING_SCALE_Y,
	                                g_param_spec_enum("picking-scale-gravity",
	                                                   "Picking scale gravity",
	                                                   "The center point for picking as gravity",
	                                                   CLUTTER_TYPE_GRAVITY, CLUTTER_GRAVITY_NONE,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interacting:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTING,
	                                g_param_spec_boolean("interacting",
	                                "Interacting",
	                                "Whether the widget is interacting with an user or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interactive-multiplier:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_SIZE_MULTIPLIER,
	                                g_param_spec_double("interactive-size-multiplier",
	                                "Interactive size multiplier",
	                                "The multiplier that is used to multiply the preferred size of the widget when it is interactive",
	                                0.0, G_MAXDOUBLE, 1.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleWidget:interactive-multiplier-set:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_SIZE_MULTIPLIER_SET,
	                                g_param_spec_boolean("interactive-size-multiplier-set",
	                                "Interactive size multiplier set",
	                                "Whether the interactive size multiplier is set or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_BUTTON_PRESS_ACTION,
	                                g_param_spec_object("button-press-action",
	                                "Button press action",
	                                "The action that correspond to the button press event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_BUTTON_RELEASE_ACTION,
	                                g_param_spec_object("button-release-action",
	                                "Button release action",
	                                "The action that correspond to the button release event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_ENTER_ACTION,
	                                g_param_spec_object("enter-action",
	                                "Enter action",
	                                "The action that correspond to the enter event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_LEAVE_ACTION,
	                                g_param_spec_object("leave-action",
	                                "Button pressa ction",
	                                "The action that correspond to the leave event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_MOTION_ACTION,
	                                g_param_spec_object("motion-action",
	                                "Button pressa ction",
	                                "The action that correspond to the motion event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleActor:button-press-action:
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLL_ACTION,
	                                g_param_spec_object("scroll-action",
	                                "Button pressa ction",
	                                "The action that correspond to the scroll event",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));


	/**
	 * TangleActor::animate-transition:
	 * @actor: the object which received the signal
	 *
	 * The ::animate-transition signal is emitted when the allocation box of an actor has been
	 * changed.
	 */
	signals[ANIMATE_TRANSITION] = g_signal_new("animate-transition", G_TYPE_FROM_CLASS(gobject_class),
	                                           G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleActorClass, animate_transition),
						   tangle_signal_accumulator_non_null_handled, NULL,
						   tangle_marshal_OBJECT__BOXED_BOXED,
						   CLUTTER_TYPE_ANIMATION, 2,
						   CLUTTER_TYPE_ACTOR_BOX, CLUTTER_TYPE_ACTOR_BOX);

	g_type_class_add_private (gobject_class, sizeof (TangleActorPrivate));
}

static void tangle_actor_init(TangleActor* actor) {
	actor->priv = G_TYPE_INSTANCE_GET_PRIVATE (actor, TANGLE_TYPE_ACTOR, TangleActorPrivate);

	actor->priv->transition_scale_x = 1.0;
	actor->priv->transition_scale_y = 1.0;
	actor->priv->picking_scale_x = 1.0;
	actor->priv->picking_scale_y = 1.0;

	actor->priv->animations = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)g_free, (GDestroyNotify)g_object_unref);

	actor->priv->paint_event_handler_id = g_signal_connect(actor, "paint", G_CALLBACK(tangle_actor_paint_event_handler), NULL);
	actor->priv->pick_event_handler_id = g_signal_connect(actor, "pick", G_CALLBACK(tangle_actor_pick_event_handler), NULL);
}

static void set_coordinates_by_gravity(ClutterGravity gravity, gfloat width, gfloat height, gfloat* x_p, gfloat* y_p) {
	switch (gravity) {
		case CLUTTER_GRAVITY_NORTH_WEST:
		case CLUTTER_GRAVITY_WEST:
		case CLUTTER_GRAVITY_SOUTH_WEST:
			*x_p = 0.0;
			break;
		case CLUTTER_GRAVITY_NORTH_EAST:
		case CLUTTER_GRAVITY_EAST:
		case CLUTTER_GRAVITY_SOUTH_EAST:
			*x_p = width;
			break;
		case CLUTTER_GRAVITY_CENTER:
			*x_p = width / 2.0;
			break;
		default:
			break;
	}
	switch (gravity) {
		case CLUTTER_GRAVITY_NORTH_WEST:
		case CLUTTER_GRAVITY_NORTH:
		case CLUTTER_GRAVITY_NORTH_EAST:
			*y_p = 0.0;
			break;
		case CLUTTER_GRAVITY_SOUTH_WEST:
		case CLUTTER_GRAVITY_SOUTH:
		case CLUTTER_GRAVITY_SOUTH_EAST:
			*y_p = height;
			break;
		case CLUTTER_GRAVITY_CENTER:
			*y_p = height / 2.0;
			break;
		default:
			break;
	}
}

static void set_picking_scale_center_by_gravity(TangleActor* actor) {
	gfloat x, y;
	
	if (actor->priv->picking_scale_gravity != CLUTTER_GRAVITY_NONE) {
		set_coordinates_by_gravity(actor->priv->picking_scale_gravity,
		                           clutter_actor_get_width(CLUTTER_ACTOR(actor)), clutter_actor_get_height(CLUTTER_ACTOR(actor)),
					   &x, &y);
		if (actor->priv->picking_scale_center_x != x) {
			actor->priv->picking_scale_center_x = x;
			g_object_notify(G_OBJECT(actor), "picking-scale-center-x");
		}
		if (actor->priv->picking_scale_center_y != y) {
			actor->priv->picking_scale_center_y = y;
			g_object_notify(G_OBJECT(actor), "picking-scale-center-y");
		}
	}
}

static void set_picking_scale_gravity(TangleActor* actor, ClutterGravity gravity) {
	if (actor->priv->picking_scale_gravity != gravity) {
		actor->priv->picking_scale_gravity = gravity;
		set_picking_scale_center_by_gravity(actor);
		g_object_notify(G_OBJECT(actor), "picking-scale-gravity");
	}
}

static gboolean handle_queued_events(TangleActor* actor) {
	gboolean queue_empty = TRUE;
	GList* last;
	
	if (actor->priv->descendant_events) {
		last = g_list_last(actor->priv->descendant_events);
		TANGLE_ACTOR_GET_CLASS(actor)->handle_descendant_event(actor, TANGLE_EVENT(last->data));
		tangle_event_free(TANGLE_EVENT(last->data));
		actor->priv->descendant_events = g_list_delete_link(actor->priv->descendant_events, last);
		queue_empty = FALSE;
	}
	if (actor->priv->ancestor_events) {
		last = g_list_last(actor->priv->ancestor_events);
		TANGLE_ACTOR_GET_CLASS(actor)->handle_ancestor_event(actor, TANGLE_EVENT(last->data));
		tangle_event_free(TANGLE_EVENT(last->data));
		actor->priv->ancestor_events = g_list_delete_link(actor->priv->ancestor_events, last);
		queue_empty = FALSE;
	}
	
	return queue_empty;
}

static void unset_transition_scale_children(TangleActor* actor) {
	actor->priv->transition_scale_children = FALSE;
}

static void on_hiding_animation_completed(TangleActor* actor) {
	if (TANGLE_ACTOR_IS_HIDING(actor)) {
		TANGLE_ACTOR_UNSET_FLAGS(actor, TANGLE_ACTOR_HIDING);

		g_object_unref(actor->priv->hiding_animation);
		actor->priv->hiding_animation = NULL;

		if (actor->priv->destroy) {
			clutter_actor_destroy(CLUTTER_ACTOR(actor));
		} else if (actor->priv->hide_all) {
			clutter_actor_hide_all(CLUTTER_ACTOR(actor));
		} else {
			clutter_actor_hide(CLUTTER_ACTOR(actor));
		}
	}
	actor->priv->hide_all = FALSE;
}
