/*
 * tangle-style.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-vault.h"
#include <string.h>
#include <gobject/gvaluecollector.h>

/**
 * SECTION:tangle-vault
 * @Short_description: A data structure holding GValuea
 * @Title: TangleVault
 */

static void free_vault(TangleVault* vault, guint n_values_to_unset);

TangleVault* tangle_vault_new(guint n, GType first_type, ...) {
	TangleVault* vault;
	GType type;
	va_list args;
	guint i;
	gchar* error;
	
	vault = g_slice_new(TangleVault);
	vault->n = n;
	vault->values = (GValue*)g_slice_alloc0(n * sizeof(GValue));

	type = first_type;
	va_start(args, first_type);
	for (i = 0; i < n; i++, type = va_arg(args, GType)) {
		g_value_init(&vault->values[i], type);
		error = NULL;
		G_VALUE_COLLECT(&vault->values[i], args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
			
			free_vault(vault, i);
			vault = NULL;
			break;
		}
	}

	return vault;
}

void tangle_vault_free(TangleVault* vault) {
	free_vault(vault, vault->n);
}

void tangle_vault_get(TangleVault* vault, guint n, GType first_type, ...) {
	GType type;
	GValue value;
	va_list args;
	guint i;
	gchar* error;

	g_return_if_fail(vault->n == n);
	
	type = first_type;
	va_start(args, first_type);
	for (i = 0; i < n; i++, type = va_arg(args, GType)) {
		g_return_if_fail(G_VALUE_HOLDS(&vault->values[i], type));
		
		error = NULL;
		G_VALUE_LCOPY(&vault->values[i], args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		}
	}
	va_end(args);
}

static void free_vault(TangleVault* vault, guint n_values_to_unset) {
	guint i;
	
	for (i = 0; i < n_values_to_unset; i++) {
		g_value_unset(&vault->values[i]);
	}
	g_slice_free1(vault->n * sizeof(GValue), vault->values);
	g_slice_free(TangleVault, vault);
}
