/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2009,2010 LUT.
 *
 * @name IteratorWrapperC.cc
 * @memo Implementations of IteratorWrapper.h C API functions.
 *
 * @version 0.3
 * date     20.11.2009
 * change   18.02.2010
 */
 
#include "IteratorWrapper.h"

/**
 * @memo Create new service iterator.
 * @doc Create new service iterator, takes the first entry from the given list.
 *
 * @return pointer to the created service iterator
 */
ServiceIterator* ph_c_new_service_iterator(TServiceList* aServiceList)
{
	if(aServiceList == NULL) return NULL;
	else return new ServiceIterator(aServiceList->Begin());
}

/**
 * @memo Delete service iterator
 * @doc Delete the given iterator.
 */
void ph_c_delete_service_iterator(ServiceIterator* aWrapper)
{
	if(aWrapper) delete aWrapper;
}

/**
 * @memo Get the content pointed by iterator.
 * @doc Get the content pointed by the iterator.
 *
 * @returns Pointer to the CService
 */
CService* ph_c_service_iterator_get_service(ServiceIterator* aWrapper)
{
	if(aWrapper == NULL) return NULL;
	else return *(aWrapper->GetIterator());
}

/**
 * @memo Move the iterator to next.
 * @doc Increases the iterator with one step.
 */
void ph_c_service_iterator_next_iterator(ServiceIterator* aWrapper)
{
	if(aWrapper) aWrapper->NextIterator();
}

/**
 * @memo Check if the current iterator is the last one in the list.
 * @doc Checks if the iterator is the same as the last one in the list.
 *
 * @return 1 (TRUE) if is last, 0 (FALSE) if not the last
 */
int ph_c_service_iterator_is_last(ServiceIterator* aWrapper, TServiceList* aServiceList)
{
	if(aWrapper == NULL || aServiceList == NULL) return 0;
	else return (aWrapper->GetIterator() != aServiceList->End()) ? 0 : 1;
}

/**
 * @memo Create new device iterator.
 * @doc Create new device iterator, takes the first entry from the given list.
 *
 * @return pointer to the created service iterator
 */
DeviceIterator* ph_c_new_device_iterator(TDeviceList* aDeviceList)
{
	if(aDeviceList == NULL) return NULL;
	return new DeviceIterator(aDeviceList->Begin());
}

/**
 * @memo Delete device iterator
 * @doc Delete the given iterator.
 */
void ph_c_delete_device_iterator(DeviceIterator* aWrapper)
{
	if(aWrapper) delete aWrapper;
}

/**
 * @memo Get the content pointed by iterator.
 * @doc Get the content pointed by the iterator.
 *
 * @returns Pointer to the MAbstractDevice
 */
MAbstractDevice* ph_c_device_iterator_get_device(DeviceIterator* aWrapper)
{
	if(aWrapper == NULL) return NULL;
	return *(aWrapper->GetIterator());
}

/**
 * @memo Move the iterator to next.
 * @doc Increases the iterator with one step.
 */
void ph_c_device_iterator_next_iterator(DeviceIterator* aWrapper)
{
	if(aWrapper) aWrapper->NextIterator();
}

/**
 * @memo Check if the current iterator is the last one in the list.
 * @doc Checks if the iterator is the same as the last one in the list.
 *
 * @return 1 (TRUE) if is last, 0 (FALSE) if not the last
 */
int ph_c_device_iterator_is_last(DeviceIterator* aWrapper, TDeviceList* aDeviceList)
{
	if(aWrapper == NULL || aDeviceList == NULL) return 0;
	else return (aWrapper->GetIterator() != aDeviceList->End()) ? 0 : 1;
}
