/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name AbstractDevice.h
 * @memo Abstract base class shared by all PeerHood device classes.
 *
 * @version 0.43
 * date     09.04.2003
 * change   18.02.2003
 */

#ifndef __ABSTRACTDEVICE_H__
#define __ABSTRACTDEVICE_H__

#include "Service.h"
#include "PHList.h"

#ifdef __cplusplus

typedef CPHList<CService*> TServiceList;
typedef CPHIterator<CService*> TServiceIterator;

/**
 * @memo Abstract base class shared by all PeerHood device classes.
 * @doc Abstract base class shared by all PeerHood device classes. One class
 * holds various information about one device: list of all available services,
 * PeerHood capability etc.
 */
class MAbstractDevice
{
 public:
  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MAbstractDevice() {};

  /**
   * @memo Returns device's name.
   * @doc Returns the name of the device.
   *
   * @return the name of the device
   */
  virtual const std::string& GetName() = 0;
  
  /**
   * @memo Returns device's checksum.
   * @doc Returns the device checksum
   *
   * @return the name of the device
   */
  virtual unsigned int GetChecksum() = 0;

  /**
   * @memo Returns device's prototype id.
   * @doc Returns device's prototype id. This id is unique for each network
   * technology.
   *
   * @return the prototype id of the device
   */
  virtual const std::string& GetPrototype() = 0;

  /**
   * @memo Tells whether the device has PeerHood enabled or not.
   * @doc Tells whether the device has PeerHood enabled or not. If there's no
   * PeerHood then there are no services either. This function is somewhat
   * unnecessary since devices without PeerHood should be of no interest.
   * Anyway this function can be used to "sniff" the environment.
   *
   * @return true if the remote device has PeerHood enabled
   */
  virtual bool HasPeerHood() = 0;

  /**
   * @memo Used to query if the device has some particular service.
   * @doc Used to query if the device has some particular service present. The
   * service can be accessible through any networking technology; this function
   * doesn't make any difference between the technologies.
   *
   * @param aServiceName The name of the service to be queried.
   *
   * @return true if the device has the given service present
   */
  virtual bool HasService(const std::string& aServiceName) = 0;

  /**
   * @memo Returns all services available on the device.
   * @doc Returns all services available on the device. The returned variable
   * is a list that can be iterated via a <code>TServiceIterator</code> 
   * iterator. Note that caller is responsible of freeing the memory allocated
   * for the returned list.
   *
   * @return list of all available services on the device
   */
  virtual TServiceList* GetServiceListL() = 0;

  /**
   * @memo Returns the address used by the device.
   * @doc Returns the address used by the device. There's no formal shape for
   * the returned address, because the exact address format is depending on the
   * current transport medium.
   *
   * @return the address used by the device
   */
  virtual const std::string& GetAddress() = 0;


  
};

#else // C

typedef 
	struct MAbstractDevice
		MAbstractDevice;
typedef
	struct TServiceList
		TServiceList;
typedef
	struct TServiceIterator
		TServiceIterator;
#endif //__cplusplus

#ifdef __cplusplus
extern "C" {
#endif

// ------------ C-API definitions for AbstractDevice class ------------- //

const char* ph_c_device_get_name(MAbstractDevice* _aDevice);

unsigned int ph_c_device_get_checksum(MAbstractDevice* _aDevice);

const char* ph_c_device_get_prototype(MAbstractDevice* _aDevice);

int ph_c_device_has_peerhood(MAbstractDevice* _aDevice);

int ph_c_device_has_service(MAbstractDevice* _aDevice, const char* _aServiceName);

TServiceList* ph_c_device_get_servicelist(MAbstractDevice* _aDevice);

const char* ph_c_device_get_address(MAbstractDevice* _aDevice);

#ifdef __cplusplus
}
#endif

#endif // __ABSTRACTDEVICE_H__
