/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name MaemoWLANListener.cc
 * @memo Implementation of MaemoWLANListener.h, contains functionality for checking
 * and listening for WLAN adapter state changes in Maemo environment through D-Bus.
 * Uses private connection to D-Bus. This is meant to be used by WLAN plugin only, created
 * through CMaemoWLANCreator.
 *
 * @version 0.13
 * date     12.10.2008
 * change   7.1.2009
 */

#include <cstdlib>
#include <icd/osso-ic-dbus.h>
#include "MaemoWLANListener.h"

using namespace std;

/**
 * @memo Default constructor - hidden
 * @doc Not in use
 */
CMaemoWLANListener::CMaemoWLANListener()
{
	iConnection = NULL;
	iName = std::string("MaemoWLANListener");
	//iConverter = NULL;
}

/**
 * @memo Parametrizised constructor
 * @doc Constructor for creating a instance of this class, requires a reference to owner
 * component as parameter. The created instance of this class will be registered to that
 * given owner (WLAN plugin).
 * 
 * @param aConverter reference to owner
 */
CMaemoWLANListener::CMaemoWLANListener(MAbstractStateConverter* aConverter)
{
	if(aConverter == NULL) return;
	iConverter = aConverter;
	iConnection = NULL;
	iName = std::string("MaemoWLANListener");
	iConverter->RegisterListener(this);
}

/**
 * @memo Default destructor
 * @doc Destructor, closes connection to D-Bus if not already closed.
 */
CMaemoWLANListener::~CMaemoWLANListener()
{
	if(iConnection != NULL)
	{
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

/**
 * @memo Connects listener to source and sets up the connection to D-Bus
 * @doc Setups the connection to D-Bus (SetupConnection()) and registers
 * listener to certain signals (RegisterSignals()).
 * 
 * @return bool true if success
 * 
 */
bool CMaemoWLANListener::Connect()
{
	if(!SetupConnection()) return false;
	
	if(!RegisterSignals()) return false;
	
	return true;
}

/**
 * @memo Disconnects the listener
 * @doc Closes the connection to D-Bus.
 */
void CMaemoWLANListener::Disconnect()
{
	if(iConnection != NULL)
	{
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

/**
 * @memo Checks the initial state of the WLAN adapter
 * @doc Sends a pending method call (blocking) that requests current WLAN
 * state from ICD in Maemo through D-Bus. Based on the return value (that
 * is an unsigned 32-bit integer) the state of the WLAN adapter is changed 
 * (value=1 - WLAN on).
 * 
 */
void CMaemoWLANListener::CheckInitialState()
{
	DBusMessage* msg = NULL;
	DBusPendingCall* statecall = NULL;
	DBusMessageIter iter;
	dbus_uint32_t mdata; 
	int msgtype;

	// Connection lost
	if(iConnection == NULL) return;

	// Create a method call
	msg = dbus_message_new_method_call(ICD_DBUS_SERVICE, ICD_DBUS_PATH, ICD_DBUS_INTERFACE,ICD_GET_STATE_REQ);

	if(msg == NULL)
	{
		ERR("MaemoWLANListener::CheckInitialState: Cannot create message.");
		return;
	}

	// Reply cannot be sent?
	if(!dbus_connection_send_with_reply(iConnection,msg,&statecall,-1))
	{
		ERR("MaemoWLANListener::CheckInitialState: Cannot send state request.");
		return;
	}

	// Cannot do a pending call
	if(statecall == NULL)
	{
		ERR("MaemoWLANListener::CheckInitialState: Pending call failed.");
		return;
	}

	// Make sure that data is sent
	dbus_connection_flush(iConnection);

	// Don't need the message anymore - unref (free) it
	dbus_message_unref(msg);
	msg = NULL;

	// Block until reply received
	dbus_pending_call_block(statecall);

	// Check the reply - no reply, do nothing and return
	if((msg = dbus_pending_call_steal_reply(statecall)) == NULL)
	{
		ERR("MaemoWLANListener::CheckInitialState: no reply received.");
		return;
	}

	// Initialize iterator
	if(!dbus_message_iter_init(msg,&iter)) return;

	// Awaiting only an integer variable which indicates the state of the WLAN
	msgtype = dbus_message_iter_get_arg_type(&iter);

	// To make sure that variable is a 32bit integer (unsigned)
	if (msgtype == DBUS_TYPE_UINT32)
	{
		// Get the variable
		dbus_message_iter_get_basic(&iter, &mdata);

		if(mdata == 1)
		{
			DBG("MaemoWLANListener::CheckInitialState: WLAN ON.");
			iConverter->SetState(true);
		}
		else
		{
			DBG("MaemoWLANListener::CheckInitialState: WLAN OFF.");
			iConverter->SetState(false);
		}
	}

	// Free pending call
	dbus_pending_call_unref(statecall);
	statecall = NULL;

	// Free message
	dbus_message_unref(msg);
	msg = NULL;
}

/**
 * @memo Checks the current state of WLAN adapter
 * @doc Checks the first message from the D-Bus receive
 * queue, always takes the first one off from the queue.
 * Message is checked by HandleMessage() that executes
 * necessary operations related to message content. If 
 * there are no messages in the queue nothing will be done.
 * The connection to D-Bus is accessed in non-blocking way.
 * 
 */
void CMaemoWLANListener::CheckState()
{
	DBusMessage* message = NULL;
		
	// Check connection
	if(iConnection == NULL) return;

	// Allows messages to be read and marshalled from the wire on non-blocking connection
	// Second parameter is blocking time in milliseconds
	dbus_connection_read_write(iConnection,0);

	// Pop the first message
	if((message = dbus_connection_pop_message(iConnection)) != NULL)
	{
		HandleMessage(message);
		dbus_message_unref(message);
		message = NULL;
	}
	else return;
}

/**
 * @memo Returns listener name
 * @doc Returns the name of this listener (hardcoded atm.)
 * 
 * @return string Name of this listener
 */
const std::string& CMaemoWLANListener::GetName()
{
	return iName;
}

/**
 * @memo Sets up connection to D-Bus 
 * @doc Sets up the connection to D-Bus by using private connection.
 * 
 * @return bool true if no errors were detected
 */
bool CMaemoWLANListener::SetupConnection()
{
	DBusError error;

		// Initialize error
		dbus_error_init(&error);

		// Get system bus, private connection
		iConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM, &error);

		// Error
		if (iConnection == NULL)
		{
			if (dbus_error_is_set(&error))
			{
				ERR("MaemoWLANListener::SetupConnection: Connection error: %s", error.message);
				
				// Free error
				dbus_error_free(&error);
			}
			return false;
		}
		else return true;
}

/**
 * @memo Registers listener to certain D-Bus interface
 * @doc Registers to listen signals from interface ICD_DBUS_INTERFACE
 * (defined in osso-ic-dbus.h). 
 * 
 * @return bool true if no errors were detected
 */
bool CMaemoWLANListener::RegisterSignals()
{
	DBusError error;
	std::string signal_and_if;

	// Check connection
	if(iConnection == NULL) return false;

	// Create signal that is listened
	// form: type='signal',interface='com.nokia.wlancond.signal'
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += ICD_DBUS_INTERFACE;
	signal_and_if += "\'";

	// Initialize error variable
	dbus_error_init(&error);

	// Connect to com.nokia.mce.signal interface listening for 'signal'
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	// Make sure it is sent
	dbus_connection_flush(iConnection);

	// If cannot listen 
	if (dbus_error_is_set(&error))
	{
		ERR("MaemoWLANListener::RegisterSignals: Cannot add listening to signal, reason: %s",error.message);
		dbus_error_free(&error);
		return false;
	}
	return true;
}

/**
 * @memo Handles the given message and reacts based on the content of the msg
 * @doc Handles the message that is given as parameter. Allows only ICD_STATUS_CHANGED_SIG
 * signals that are emitted from ICD_DBUS_INTERFACE. With "CONNECTED" as 3rd parameter
 * WLAN plugin is set as active, with "DISCONNECTING" (about to disconnect) plugin is set to passive.
 * 
 * @param message Message to be handled.
 */
void CMaemoWLANListener::HandleMessage(DBusMessage* message)
{
	DBusMessageIter msg_iter;
	const char* msg_data = NULL;
	int msg_type;

	// Message iterator
	dbus_message_iter_init(message, &msg_iter);

	// Device state change - contains multiple parameters as strings
	if(dbus_message_is_signal(message,ICD_DBUS_INTERFACE,ICD_STATUS_CHANGED_SIG))
	{
		if((msg_type = dbus_message_iter_get_arg_type(&msg_iter)) == DBUS_TYPE_STRING)
		{
			// First string is the id of the access point that changed status, or [SCAN] -> skip
			// Id of AP could be utilized?
			dbus_message_iter_get_basic(&msg_iter,&msg_data);
			if(strcmp(msg_data,"[SCAN]") == 0) return;
		}

		// Next string
		dbus_message_iter_next(&msg_iter);

		if((msg_type = dbus_message_iter_get_arg_type(&msg_iter)) == DBUS_TYPE_STRING)
		{
			// Parameter contains information about network mode (infrastructure or ad hoc)
			//dbus_message_iter_get_basic(&msg_iter,&msg_data);
			//DBG("MaemoWLANListener::HandleMessage: Network mode: %s", msg_data);
		}

		// Next string
		dbus_message_iter_next(&msg_iter);

		if((msg_type = dbus_message_iter_get_arg_type(&msg_iter)) == DBUS_TYPE_STRING)
		{
			// Status message (CONNECTED, DISCONNECTING)
			dbus_message_iter_get_basic(&msg_iter,&msg_data);
			
			// Connected to AP
			if(strcmp(msg_data,"CONNECTED") == 0)
			{
				DBG("MaemoWLANListener::HandleMessage: Wlan connected");
				iConverter->SetState(true);
			}
			
			// Starting to disconnect, disable WLAN plugin at this point
			else if(strcmp(msg_data,"DISCONNECTING") == 0)
			{
				DBG("MaemoWLANListener::HandleMessage: Wlan disconnected");
				iConverter->SetState(false);
			}
		}
	} // if(message_is_signal)
}
