/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name Factory.cc
 * @memo Implementation of the abstract factory used to transparently create
 * network specified MAbstractConnection objects.
 *
 * @version 0.1
 * date     03.04.2003
 * change   03.04.2003
 */

#include <cassert>
#include <Factory.h>
#include <Logger.h>


std::list<MAbstractCreator *> Factory::iCreators;

/** 
 * @memo Registers a new MAbstractCreator implementation.
 * @doc Registers a new MAbstractCreator implementation to the list of possible
 * creators. These creators are network technology dependent and there should 
 * be exactly one creator for each network technology.
 *
 * @param aCreator The creator to be registered. If the creator is NULL then
 * nothing happens.
 *
 * @return none
 */
void Factory::Register(MAbstractCreator* aCreator)
{
  DBG("Factory::Register");

  if (aCreator) iCreators.push_back(aCreator);
}


/** 
 * @memo Creates a new MAbstractConnection implementation specific to some
 * network technology.
 * @doc Creates a new MAbstractConnection implementation that is specific to
 * some certain network technology. The prototype string is given to each
 * registered creator one by one and if a creator recognizes the prototype it 
 * creates the asked connection object. Otherwise the prototype string is given
 * to the next creator.
 *
 * @param aProto Prototype string of the desired network technology.
 *
 * @return the newly created connection object or NULL if no matching creator
 * could be found
 */
MAbstractConnection* Factory::CreateConnectionL(const std::string& aProto)
{
  MAbstractConnection* retval = NULL;

  for (std::list<MAbstractCreator *>::iterator i = iCreators.begin();i != iCreators.end();++i) {
    if ((retval = (*i)->CreateConnectionL(aProto))) return retval;
  }

  ERR("Factory::CreateConnectionL : unknown prototype");
  ERR(aProto.c_str());

  return NULL;
}


/** 
 * @memo Creates a new MAbstractPinger object.
 * @doc Creates a new MAbstractConnection object that is suitable for the given
 * networking technology. The prototype string is given to each registered
 * creator one by one as long as a creator recognizes it or all the creators 
 * have been tried. If a creator recognizes the prototype string then it
 * creates a Pinger objects and returns that. The returned object is then 
 * passed to the original caller.
 *
 * @param aProto Prototype string of the desired network technology.
 * @param aAddress Remote device's address.
 *
 * @return the newly created connection object or NULL if no matching creator
 * could be found
 */
MAbstractPinger* Factory::CreatePingerL(const std::string& aProto, const std::string& aAddress)
{
  MAbstractPinger* retval = NULL;

  DBG(aProto.c_str());

  for (std::list<MAbstractCreator *>::iterator i = iCreators.begin();i != iCreators.end();++i) {
    if ((retval = (*i)->CreatePingerL(aProto, aAddress))) return retval;
  }

  ERR("Factory::CreatePingerL : unknown prototype");

  return NULL;
}

/** 
 * @memo Creates a new MAbstractMonitor object.
 * @doc Creates a new MAbstractMonitor object that is suitable for the given
 * networking technology. The prototype string is given to each registered
 * creator one by one as long as a creator recognizes it or all the creators 
 * have been tried. If a creator recognizes the prototype string then it
 * creates a Pinger objects and returns that. The returned object is then 
 * passed to the original caller.
 *
 * @param aProto Prototype string of the desired network technology.
 * @param aAddress Remote device's address.
 *
 * @return the newly created connection object or NULL if no matching creator
 * could be found
 */
MAbstractMonitor* Factory::CreateMonitorL(const std::string& aProto, const std::string& aAddress)
{
  MAbstractMonitor* retval = NULL;

  DBG(aProto.c_str());

  for (std::list<MAbstractCreator *>::iterator i = iCreators.begin();i != iCreators.end();++i) {
    if ((retval = (*i)->CreateMonitorL(aProto, aAddress))) return retval;
  }

  ERR("Factory::CreateMonitorL : unknown prototype");

  return NULL;
}



/** 
 * @memo Returns an array containing the names of all loaded plugins.
 * @doc Returns an array containing the names of all loaded plugins. Note that
 * caller is responsible of freeing the memory reserved for the returned array.
 * It's safe to call <code>delete</code> for the returned array even if there
 * are no items on it.
 *
 * @param aSize On return, this int contains the number of items on the list.
 * This parameter cannot be NULL.
 *
 * @return array of plugin names or NULL if no plugins are available
 */
std::string* Factory::GetPluginNamesL(int* aSize)
{
  std::string* retval;

  assert(aSize != NULL);
  *aSize = 0;

  if (iCreators.size() > 0) {
    retval = new std::string[iCreators.size()];

    for (std::list<MAbstractCreator *>::iterator i = iCreators.begin();i != iCreators.end();++i) {
      retval[(*aSize)++] = std::string((*i)->GetPrototype());
    }    

    return retval;
  }

  return NULL;
}
