/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name TCPConnection.cc
 * @memo TCP implementation of the MAbstractConnection interface.
 *
 * @version 0.2
 * date     30.06.2003
 * change   28.04.2010
 */

#include <stdio.h>
#include <iostream>
#include <sys/types.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>

#include "TCPConnection.h"


#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

const int WLAN_PEERHOOD = 8600;

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * and creates a new TCP socket.
 *
 * @param aIface interface which is wanted to use
 * @param aPluginBase plugin which uses this class (must be GPRS or WLAN)
 * @return none
 */
CTCPConnection::CTCPConnection(const std::string& aIface, const std::string& aPluginBase)
{
  iConnected = false;
  iIsListening = false;
  
  iPluginBase = aPluginBase;
  iFace = aIface;
  ifaces = NULL;
  iSocket = socket(AF_INET, SOCK_STREAM, 0);
  assert(iSocket != -1);
  iChecksum = 0;
}

/**
 * @memo Checks if required interface is found.
 * @doc Checks if required interface is found. This method
 * checks if the interface defined in the constructor could 
 * be found.
 *
 * @return false if interface couldn't be found
 */
bool CTCPConnection::GetInterfaces()
{
  CIFSearch search;
  if((ifaces = search.GetIFInfo(iFace)) == NULL)
    {
      ERR("CTCPConnection::GetInterfaces : Can't find interface %s", iFace.c_str());
      return false;
    }
  
  return true;
}

/**
 * @memo Default destructor.
 * @doc Default desttructor, deletes allocated variables
 *
 * @return none
 */
CTCPConnection::~CTCPConnection()
{
  if(ifaces != NULL)
    delete ifaces;
}
/**
 * @memo Connects to a remote device.
 * @doc Connects to a remote device for a given address and port
 * which is PeerHood's default PSM.
 *
 * @param aAddress The destination address.
 * @param aPort The destination port.
 *
 * @return true if the connection could be established
 */
bool CTCPConnection::Connect(const std::string& aAddress, int aPort)
{
  struct sockaddr_in address;
  const char *addr = NULL;
  struct hostent *shost = NULL;
  char tempBuf[INET_ADDRSTRLEN];

  if (iConnected) {
  	ERR("CTCPConnection::Connect : already connected");
    return false;
  }
	
  addr = aAddress.c_str();
  
  if( (shost = gethostbyname(addr)) == NULL)
    {
      ERR("CTCPConnection::Connect : gethostbyname");
      return false;
    } 
  
  
  memset(&tempBuf, 0, INET_ADDRSTRLEN);
  memset(&address, 0, sizeof(address));
  
  address.sin_family = AF_INET;
  address.sin_port = htons(aPort);
  memcpy(&address.sin_addr, shost->h_addr_list[0], sizeof(struct in_addr));
  
  if (connect(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
  	ERR("CTCPConnection::Connect : connect socket. Trying address: %s", inet_ntop(AF_INET,&address.sin_addr,tempBuf,sizeof(tempBuf)));
    return false;
  }
  
  if(iPluginBase == "GPRS")
    iRemoteAddress = std::string("GPRS:") + std::string(inet_ntop(AF_INET, &address.sin_addr, tempBuf, sizeof(tempBuf)));
  
  if(iPluginBase == "WLAN")
    iRemoteAddress = std::string("WLAN:") + std::string(inet_ntop(AF_INET, &address.sin_addr, tempBuf, sizeof(tempBuf)));
  
  iConnected = true;
  return true;
}

/**
 * @memo Kills the existing connection.
 * @doc Kills the existing connection. If there's no connection or it could not
 * be closed in a controlled way then the error code is returned. The existing
 * connection will be closed regardless of the return value.
 *
 * @return true if the connection was dropped in a controlled manner
 */
bool CTCPConnection::Disconnect()
{
  if (!iConnected) {
    ERR("CTCPConnection::Disconnect : Not connected");
    return false;
  }

  close(iSocket);
  iConnected = false;
  return true;
}


/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CTCPConnection::Close()
{
  close(iSocket);
}

/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. Note that this
 * function is not endianess-safe.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CTCPConnection::Write(const void* aOutBuf, int aLength)
{
  int written = 0;

  if (!iConnected) {
    ERR("CTCPConnection::Write : not connected");
    return -1;
   }

  //write actual length (needed because of MSG_WAITALL)
  aLength = htonl(aLength);
  written = write(iSocket, &aLength, sizeof(aLength));
    if(written <= 0)
    {
      ERR("CTCPConnection::Write : sending realLength failed");
      return written;
    }
  aLength = ntohl(aLength);

  written = write(iSocket, (char *)aOutBuf, aLength);
  return written;
}

/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection. Just like 
 * <code>Read</code> this function is not endianess-safe so it should be
 * treated more like a standard Posix sequential socket.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CTCPConnection::Read(void* aInBuf, int aLength)
{
  int retval = 0;

  if (!iConnected) {
    ERR("CTCPConnection::Read : not connected");
    return -1;
  }
  
  //receive actual length (needed because of MSG_WAITALL)
  int realLength = 0;
  if((retval = recv(iSocket, &realLength, sizeof(realLength), MSG_WAITALL)) <= 0)
    {
      ERR("CTCPConnection::Read : receiving realLength failed");
      return retval;
    }
  
  realLength = ntohl(realLength);
  retval = recv(iSocket, (char *)aInBuf, realLength, MSG_WAITALL);
  
  if(aLength < realLength)
    {
      ERR("CTCPConnection::Read : incoming more data than expected");
      return -1;
    }

  return retval;
}

/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const std::string& CTCPConnection::GetRemoteAddress()
{  
  return iRemoteAddress;
}

/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CTCPConnection::GetFd()
{
  //coomented because of setsockopt <-> bind problem
  //  if (!iConnected && !iIsListening) return -1;

  return iSocket;
}

/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted. This function will always listen the connection type's default
 * port. If some other port is needed then the <code>Listen(int aPort)</code>
 * should be used instead.
 *
 * @return true if successfull
 */
bool CTCPConnection::Listen()
{
   return Listen(WLAN_PEERHOOD);
}

/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted.
 *
 * @param aPort The port that should be listened.
 *
 * @return true if successfull
 */
bool CTCPConnection::Listen(int aPort)
{
  sockaddr_in address;

  if(!GetInterfaces())
    {
      ERR("CTCPConnection::Listen : GetInterfaces failed");
      return false;
    }

	memset(&address,0,sizeof(struct sockaddr_in));

  address = ifaces->iUcast;
  address.sin_port = htons(aPort);

  if (bind(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CTCPConnection::Listen : bind failed");
    return false;
  }

  if (listen(iSocket, 16) == -1) {
    ERR("CTCPConnection::Listen : listen failed");
    return false;
  }

  iIsListening = true;
  
  return true;
}

/**
 * @memo Accepts the incoming connection and returns a new connection object.
 * @doc Accepts the incoming connection and returns a new connection object. If
 * accepting fails then NULL is returned.
 *
 * @return new connection object or NULL in the case of an error
 */
MAbstractConnection* CTCPConnection::AcceptL()
{
  int newSocket;
  int addressLength;
  struct sockaddr_in address;
  CTCPConnection* retval = NULL;  
  char tempBuf[INET_ADDRSTRLEN];

	memset(&tempBuf, 0, INET_ADDRSTRLEN);
	memset(&address, 0, sizeof(struct sockaddr_in));

  addressLength = sizeof(address);

  if ((newSocket = accept(iSocket, (struct sockaddr *)&address, (socklen_t *)&addressLength)) == -1) {
    ERR("CTCPConnection::AcceptL : accept failed : ");
    return NULL;
  }
  
  retval = new CTCPConnection(iFace, "WLAN");
  if (retval->iSocket < 0)
    {
      ERR("CTCPConnection::AcceptL : socket failed");
      return NULL;
    }
  
  close(retval->iSocket);
  retval->iSocket = newSocket;
  retval->iConnected = true;
  
  if(retval->iPluginBase == "GPRS")
    retval->iRemoteAddress = std::string("GPRS:") + std::string(inet_ntop(AF_INET, &address.sin_addr, tempBuf, sizeof(tempBuf)));
  
  if(retval->iPluginBase == "WLAN")
    retval->iRemoteAddress = std::string("WLAN:") + std::string(inet_ntop(AF_INET, &address.sin_addr, tempBuf, sizeof(tempBuf)));

  iConnected = true;
  return retval;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CTCPConnection::IsListening()
{
  return iIsListening;
}


/**
 * @memo Tells if the connection is established.
 * @doc Tells if the connection is established or not.
 *
 * @return true if the connection is established
 */
bool CTCPConnection::IsConnected()
{
  return iConnected;
}

/**
 * @memo Tells if the connection has data waiting
 * @doc Checks if the connection has data to read.
 *
 * @return true if the data is ready
 */
bool CTCPConnection::HasData()
{
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
  fd_set set;
  FD_ZERO(&set);
  FD_SET(GetFd(), &set);

  switch (select(GetFd() + 1, &set, NULL, NULL, &timeout)) {
  case -1:
  	ERR("CTCPConnection::HasData(): Select error");
  	break;
    
  case 0:
    break;

  default:
    if (FD_ISSET(GetFd(), &set)){
      return true;
    }
    else {
      return false;
    }
  }
	return false;
}

unsigned int CTCPConnection::GetDeviceChecksum()
{
	return iChecksum;
}

void CTCPConnection::SetDeviceChecksum(unsigned int aChecksum)
{
	iChecksum = aChecksum;
}
