
#include "AbstractDBUSListener.h"

bool MAbstractDBUSListener::Connect()
{
	if(!SetupConnection()) return false;
		
		if(!RegisterSignals()) return false;
		
		return true;
}

void MAbstractDBUSListener::Disconnect()
{
	if(iConnection != NULL)
	{
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

void MAbstractDBUSListener::CheckInitialState()
{
	DBusMessage* msg = NULL;
	DBusPendingCall* statecall = NULL;
	DBusMessageIter iter;
	std::string request_path;
	const char* data_char;
	dbus_int32_t data_int32;
	dbus_uint32_t data_uint32;
	dbus_bool_t data_bool;
	int msgtype;
	
	for(std::list<MethodCall*>::iterator it = GetMethodCalls().begin(); it != GetMethodCalls().end(); ++it)
	{
		msg = dbus_message_new_method_call((*it)->serviceName.c_str(),(*it)->pathName.c_str(),(*it)->ifName.c_str(),(*it)->method.c_str());

		if(msg == NULL)
		{
			DBG("%s::CheckInitialState: Cannot create new method call.", iName.c_str());
			return;
		}

		if(!dbus_connection_send_with_reply(iConnection,msg,&statecall,-1))
		{
			DBG("%s::CheckInitialState: Cannot send method call with reply.", iName.c_str());
			return;
		}

		// Cannot do a pending call
		if(statecall == NULL)
		{
			DBG("%s::CheckInitialState: Cannot execute a pending call.", iName.c_str());
			return;
		}

		dbus_connection_flush(iConnection);
		dbus_message_unref(msg);

		msg = NULL;

		// Block until reply received
		dbus_pending_call_block(statecall);

		// Get the replymessage
		if((msg = dbus_pending_call_steal_reply(statecall)) == NULL)
		{
			DBG("%s::CheckInitialState: No reply received.", iName.c_str() );
			return;
		}

		if(!dbus_message_iter_init(msg,&iter)) return;

		while ((msgtype = dbus_message_iter_get_arg_type(&iter)) != DBUS_TYPE_INVALID)
		{
			switch((*it)->reaction)
			{
			case CHECK_STRING:
				if(msgtype == DBUS_TYPE_STRING)
				{
					dbus_message_iter_get_basic(&iter,&data_char);
					InitialCheckCharParam(data_char);
				}
				break;

			case CHECK_BOOL:
				if(msgtype == DBUS_TYPE_BOOLEAN)
				{
					dbus_message_iter_get_basic(&iter,&data_bool);
					InitialCheckBooleanParam(data_bool);
				}
				break;
			
			case CHECK_INT32:
				if(msgtype == DBUS_TYPE_INT32)
				{
					dbus_message_iter_get_basic(&iter,&data_int32);
					InitialCheckInt32Param(data_int32);
				}
				break;
			case CHECK_INT64:
				break;
			case CHECK_LONG:
				break;
			case CHECK_UINT32:
				if(msgtype == DBUS_TYPE_UINT32)
				{
					dbus_message_iter_get_basic(&iter,&data_uint32);
					InitialCheckUInt32Param(data_uint32);
				}
				break;
			case CHECK_UINT64:
				break;
			}
			if (!dbus_message_iter_next(&iter)) break;
		}

		// Free pending call
		dbus_pending_call_unref(statecall);
		statecall = NULL;

		dbus_message_unref(msg);
		msg = NULL;
		
	}
}

void MAbstractDBUSListener::CheckState()
{
	DBusMessage* message = NULL;

	// Check connection
	if(iConnection == NULL) return;

	// Allows messages to be read and marshalled from the wire on non-blocking connection
	// Second parameter is blocking time in milliseconds
	dbus_connection_read_write(iConnection,0);

	// Pop the first message
	if((message = dbus_connection_pop_message(iConnection)) != NULL)
	{
		HandleMessage(message);
		dbus_message_unref(message);
		message = NULL;
	}
	else return;
}


bool MAbstractDBUSListener::SetupConnection()
{
	DBusError error;

		// Initialize error
		dbus_error_init(&error);

		// Get system bus, private connection
		iConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM, &error);

		// Error
		if (iConnection == NULL)
		{
			if (dbus_error_is_set(&error))
			{
				ERR("%s::SetupConnection: Connection error: %s", iName.c_str(), error.message);
				
				// Free error
				dbus_error_free(&error);
			}
			return false;
		}
		else return true;
}


bool MAbstractDBUSListener::RegisterSignals()
{
	DBusError error;
	std::string signal_and_if;

	// Check connection
	if(iConnection == NULL) return false;
	
	for(std::list<SignalEntry*>::iterator it = GetSignalList().begin(); it != GetSignalList().end(); ++it)
	{
		
		// Create a signal to listen
		signal_and_if = "type=\'";
		signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
		signal_and_if += "\',interface=\'";
		signal_and_if += (*it)->ifName;
		signal_and_if += "\'";

		// Initialize error variable
		dbus_error_init(&error);

		// Connect to com.nokia.btcond.signal interface listening for 'signal'
		dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
		// Make sure it is sent
		dbus_connection_flush(iConnection);

		// If cannot listen 
		if (dbus_error_is_set(&error))
		{
			ERR("%s::RegisterSignals: Cannot add listening to signal: %s, reason: %s", iName.c_str(), signal_and_if.c_str(), error.message);
			dbus_error_free(&error);
			return false;
		}
		signal_and_if.clear();
	}

	return true;
}

void MAbstractDBUSListener::HandleMessage(DBusMessage* aMessage)
{
	DBusMessageIter iter;

	const char* data_char;
	dbus_int32_t data_int32;
	dbus_uint32_t data_uint32;
	dbus_bool_t data_bool;
	int msgtype, msgcount = 0;

	dbus_message_iter_init(aMessage, &iter);

	for(std::list<SignalEntry*>::iterator it = GetSignalList().begin(); it != GetSignalList().end(); ++it)
	{
		if(dbus_message_is_signal(aMessage,(*it)->ifName.c_str(),(*it)->signalName.c_str()))
		{
			// Signal type determines action
			switch((*it)->reaction)
			{
			case STATE_ACTIVE:
				iConverter->SetState(true);
				break;
			case STATE_PASSIVE:
				iConverter->SetState(false);
				break;
			case SHUTDOWN:
				iConverter->TriggerShutdown();
				break;
			case PWRSAVE_ON:
				break;
			case PWRSAVE_OFF:
				break;
			default:
				break;
			}

			// Signal parameter determines action
			while ((msgtype = dbus_message_iter_get_arg_type(&iter)) != DBUS_TYPE_INVALID)
			{
				msgcount++;
				
				switch((*it)->reaction)
				{
				case CHECK_STRING:
					if(msgtype == DBUS_TYPE_STRING)
					{
						dbus_message_iter_get_basic(&iter,&data_char);
						StateCheckCharParam((*it)->signalName,data_char);
					}
					break;

				case CHECK_BOOL:
					if(msgtype == DBUS_TYPE_BOOLEAN)
					{
						dbus_message_iter_get_basic(&iter,&data_bool);
						StateCheckBooleanParam((*it)->signalName,data_bool);
					}
					break;
				case CHECK_INT32:
					if(msgtype == DBUS_TYPE_INT32)
					{
						dbus_message_iter_get_basic(&iter,&data_int32);
						StateCheckInt32Param((*it)->signalName,data_int32);
					}
					break;
				case CHECK_INT64:
					break;
				case CHECK_LONG:
					break;
				case CHECK_UINT32:
					if(msgtype == DBUS_TYPE_UINT32)
					{
						dbus_message_iter_get_basic(&iter,&data_uint32);
						InitialCheckUInt32Param(data_uint32);
					}
					break;
				case CHECK_UINT64:
					break;
					
				default:
					break;
				}
				
				// Last message?
				if(!dbus_message_iter_next(&iter)) break;
				
				// Parameter count exceeded
				if(msgcount >= (*it)->amount) break;
			}
		}
	}
}

