/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdint.h>

typedef struct _SendItemDataData SendItemDataData;

struct _MicrofeedFeed {
	unsigned int reference_count;
	MicrofeedWeakReference* weak_reference;

	char* uri;
	char* name;
	MicrofeedFeedPermission feed_permission;
	MicrofeedItem* metadata_item;
	MicrofeedPublisher* publisher;
	MicrofeedFeedCallbacks callbacks;
	void* user_data;
	MicrofeedFeedIterator* iterators;
	int subscriber_count;
	MicrofeedDatabase* database;
	MicrofeedDatabaseIndex* database_index;
	MicrofeedDatabase* statuses_database;
	MicrofeedDatabase* item_data_database;
	MicrofeedMutex* mutex;
	int updating;
	MicrofeedStore* send_item_data_datas;
};

struct _MicrofeedFeedIterator {
	MicrofeedFeedIterator* previous;
	MicrofeedFeedIterator* next;

	MicrofeedFeed* feed;
	MicrofeedDatabaseIterator* database_iterator;
	int backwards : 1;
	int timeline : 1;
};

typedef struct {
	MicrofeedFeed* feed;
	const char* start_uid;
	const char* end_uid;
	unsigned int max_count;
	const char* bus_name;
} RepublishingData;

typedef struct {
	MicrofeedFeed* feed;
	MicrofeedItem* existing_item;
	MicrofeedItem* new_item;
} ModifyItemData;

struct _SendItemDataData {
	MicrofeedFeed* feed;
	char* uid;	
	MicrofeedStore* bus_names;
};

static void index_function(void* key, const size_t key_size, void* data, const size_t data_size, void** index_key, size_t* index_key_size);
static int compare_function(const void* key1, const size_t key1_size, const void* key2, const size_t key2_size);
static void* update_thread_periodical(void* data);
static void* update_thread_user_initiated(void* data);
static void* republish_thread(void* user_data);
static void* send_item_data_thread(void* data);
static void* modify_item_thread(void* user_data);
static void remove_old_items(MicrofeedFeed* feed);
static const char* send_item_data_data_get_uid(SendItemDataData* send_item_data_data);

MicrofeedFeed* microfeed_feed_new(MicrofeedPublisher* publisher, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, MicrofeedFeedCallbacks* callbacks, void* user_data) {
	MicrofeedFeed* feed;
	char* database_name;
	char* s;
	
	feed = (MicrofeedFeed*)microfeed_memory_allocate(MicrofeedFeed);

	feed->reference_count = 1;
	feed->weak_reference = NULL;

	feed->uri = strdup(uri);
	feed->name = strdup(name);
	feed->feed_permission = feed_permission;
	feed->metadata_item = microfeed_item_new(MICROFEED_ITEM_UID_FEED_METADATA, time(NULL));
	microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_NAME, name);
	s = microfeed_feed_permission_to_string(feed_permission);
	if (s && *s) {
		microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_PERMISSION, s);
	}
	free(s);
	feed->publisher = publisher;
	feed->iterators = NULL;
	if (callbacks) {
		feed->callbacks = *callbacks;
	}
	feed->user_data = user_data;

	feed->database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), uri, compare_function);
	feed->database_index = microfeed_database_get_index(feed->database, "uid", index_function);
	database_name = microfeed_util_string_concatenate(uri, ":microfeed.statuses", NULL);
	feed->statuses_database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), database_name, NULL);
	free(database_name);
	if (feed->callbacks.download_item_data) {
		database_name = microfeed_util_string_concatenate(uri, ":microfeed.item_data", NULL);
		feed->item_data_database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), database_name, NULL);
		free(database_name);
	}
	feed->mutex = microfeed_mutex_new();
	feed->send_item_data_datas = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)send_item_data_data_get_uid);

	return feed;	
}

/* Does NOT remove subscribers */
void microfeed_feed_free(MicrofeedFeed* feed) {
	MicrofeedFeedIterator* iterator;
	
	if (feed->callbacks.destroy) {
		feed->callbacks.destroy(feed, feed->user_data);
	}
	
	free(feed->uri);
	for (iterator = feed->iterators; iterator; iterator = iterator->next) {
		iterator->feed = NULL;
	}	
	
	microfeed_database_unref(feed->database);
	microfeed_database_unref(feed->statuses_database);
	if (feed->item_data_database) {
		microfeed_database_unref(feed->item_data_database);
	}
	microfeed_mutex_free(feed->mutex);

/* TODO: Free metadata item etc. */
	
	feed->uri = NULL;
	feed->publisher = NULL;
	feed->callbacks.destroy = NULL;
	feed->callbacks.update = NULL;
	feed->subscriber_count = 0;
	feed->iterators = NULL;
	microfeed_weak_reference_invalidate(feed->weak_reference);
	feed->weak_reference = NULL;

	free(feed);	
}

MicrofeedFeed* microfeed_feed_ref(MicrofeedFeed* feed) {
	microfeed_mutex_lock(feed->mutex);

	feed->reference_count++;

	microfeed_mutex_unlock(feed->mutex);
	
	return feed;
}
	
void microfeed_feed_unref(MicrofeedFeed* feed) {
	microfeed_mutex_lock(feed->mutex);

	feed->reference_count--;
	
	if (feed->reference_count == 0) {
		microfeed_feed_free(feed);
	} else {

		microfeed_mutex_unlock(feed->mutex);
	}
}

MicrofeedWeakReference* microfeed_feed_get_weak_reference(MicrofeedFeed* feed) {
	MicrofeedWeakReference* weak_reference;
	
	microfeed_mutex_lock(feed->mutex);

	weak_reference = feed->weak_reference = microfeed_weak_reference_new(feed, feed->weak_reference);

	microfeed_mutex_unlock(feed->mutex);
	
	return weak_reference;
}

MicrofeedPublisher* microfeed_feed_get_publisher(MicrofeedFeed* feed) {
	
	return feed->publisher;
}

void microfeed_feed_stop_update(MicrofeedFeed* feed) {
	microfeed_mutex_lock(feed->mutex);
	
	if (feed->updating && microfeed_publisher_get_singleton_update_thread(feed->publisher)) {
		microfeed_publisher_send_feed_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, feed->uri);
		feed->updating = 0;
	}

	microfeed_mutex_unlock(feed->mutex);
}

void microfeed_feed_replace_item(MicrofeedFeed* feed, MicrofeedItem* item) {
	const char* uid;
	size_t uid_size;
	uint64_t timestamp;
	void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	void* existing_key;
	size_t existing_key_size;
	void* existing_data;
	size_t existing_size;
	uint64_t existing_timestamp;
	void* status_data;
	size_t status_data_size;
	char status;
	MicrofeedItem* existing_item;

	microfeed_mutex_lock(feed->mutex);

	uid = microfeed_item_get_uid(item);
	uid_size = strlen(uid) + 1;
	timestamp = (uint64_t)microfeed_item_get_timestamp(item);
	key_size = sizeof(uint64_t) + uid_size;
	key = (char*)malloc(key_size);
	*((uint64_t*)key) = timestamp;
	memcpy(key + sizeof(uint64_t), uid, uid_size);
	microfeed_item_marshal_properties(item, &data, &data_size);
	if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &existing_key, &existing_key_size, &existing_data, &existing_size)) {
		existing_timestamp = *((uint64_t*)existing_key);
		if (timestamp != existing_timestamp || data_size != existing_size || memcmp(data, existing_data, existing_size)) {
			if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &status_data, &status_data_size)) {
				status = *((char*)status_data);
				free(status_data);
				if (!(status & (char)MICROFEED_ITEM_STATUS_ACTIVE)) {
					status |= (char)(MICROFEED_ITEM_STATUS_ACTIVE);
					microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
				}			
			} else {
				status = (char)MICROFEED_ITEM_STATUS_ACTIVE;
				microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
			}
				
			existing_item = microfeed_item_new_with_status(uid, existing_timestamp, status);
			microfeed_item_demarshal_properties(existing_item, existing_data, existing_size);
			microfeed_publisher_handle_item_property_change(feed->publisher, existing_item, item);
			microfeed_item_free(existing_item);
			
			microfeed_publisher_send_item_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_CHANGED, feed->uri, item);
//			if (timestamp != existing_timestamp) {
				microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
//			}
			microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
		}
		free(existing_key);
		free(existing_data);
	} else {	
		status = (char)MICROFEED_ITEM_STATUS_NEW;
		microfeed_publisher_handle_item_property_change(feed->publisher, NULL, item);
		microfeed_publisher_send_item_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->uri, item);
		microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
		microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
	}
	free(key);
	
	microfeed_mutex_unlock(feed->mutex);
}

void microfeed_feed_remove_item(MicrofeedFeed* feed, const char* uid) {	
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_mutex_lock(feed->mutex);

	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &key, &key_size, &data, &data_size)) {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(uid, timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		microfeed_publisher_handle_item_property_change(feed->publisher, item, NULL);
		microfeed_item_free(item);
		free(key);
		free(data);
		
		microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
		microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
		if (feed->item_data_database) {
			microfeed_database_remove_data(feed->item_data_database, uid, uid_size);
		}
		microfeed_publisher_send_item_uid_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->uri, uid);
	}

	microfeed_mutex_unlock(feed->mutex);
}

void microfeed_feed_remove_items(MicrofeedFeed* feed, const char* start_uid, const char* end_uid) {
	size_t uid_size;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;
	
	microfeed_mutex_lock(feed->mutex);
	
	uid_size = (start_uid ? strlen(start_uid) + 1 : 0);
	for (iterator = microfeed_database_index_iterate(feed->database_index, start_uid, uid_size, 0);
	     (microfeed_database_iterator_get(iterator, &key, &key_size, &data, &data_size)) && (!end_uid || strcmp ((const char*)key, end_uid) <= 0);
	     microfeed_database_iterator_next(iterator)) {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(key + sizeof(uint64_t), timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		microfeed_publisher_handle_item_property_change(feed->publisher, item, NULL);
		microfeed_item_free(item);
		
		microfeed_database_index_remove_data(feed->database_index, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		microfeed_database_remove_data(feed->statuses_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		if (feed->item_data_database) {
			microfeed_database_remove_data(feed->item_data_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		}
		microfeed_publisher_send_item_uid_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->uri, key + sizeof(uint64_t));
	}
	microfeed_database_iterator_free(iterator);
	
	microfeed_mutex_unlock(feed->mutex);	
}

void microfeed_feed_add_subscriber(MicrofeedFeed* feed, const char* bus_name) {
	microfeed_mutex_lock(feed->mutex);

	feed->subscriber_count++;
	feed->reference_count++;

	microfeed_mutex_unlock(feed->mutex);
}

int microfeed_feed_get_subscriber_count(MicrofeedFeed* feed) {
	int subscriber_count;

	microfeed_mutex_lock(feed->mutex);

	subscriber_count = feed->subscriber_count;

	microfeed_mutex_unlock(feed->mutex);

	return subscriber_count;
}

MicrofeedItem* microfeed_feed_get_item(MicrofeedFeed* feed, const char* uid) {
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_mutex_lock(feed->mutex);
	
	if (!microfeed_database_index_get_data(feed->database_index, uid, strlen(uid) + 1, &key, &key_size, &data, &data_size)) {
		item = NULL;
	} else {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new((char*)(key + sizeof(uint64_t)), (time_t)timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		free(data);
	}

	microfeed_mutex_unlock(feed->mutex);
	
	return item;	
}

const char* microfeed_feed_get_uri(MicrofeedFeed* feed) {

	return feed->uri;
}

const char* microfeed_feed_get_name(MicrofeedFeed* feed) {
	
	return feed->name;
}

void microfeed_feed_set_name(MicrofeedFeed* feed, const char* name) {
	if (strcmp(name, feed->name)) {
		free(feed->name);
		feed->name = strdup(name);
		/* TODO: Change metadata item. */
	}
}

MicrofeedItem* microfeed_feed_get_metadata_item(MicrofeedFeed* feed) {

	return feed->metadata_item;
}

MicrofeedFeedIterator* microfeed_feed_iterate(MicrofeedFeed* feed, const char* start_uid, int backwards) {
	MicrofeedFeedIterator* iterator;

	microfeed_mutex_lock(feed->mutex);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = feed;
	iterator->backwards = backwards;
	iterator->timeline = 0;
	iterator->database_iterator = microfeed_database_index_iterate(feed->database_index, start_uid, (start_uid ? strlen(start_uid) + 1 : 0), backwards);

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_mutex_unlock(feed->mutex);

	return iterator;
}

MicrofeedFeedIterator* microfeed_feed_iterate_timeline(MicrofeedFeed* feed, const time_t start_timestamp, int backwards) {
	MicrofeedFeedIterator* iterator;
	char* key;

	microfeed_mutex_lock(feed->mutex);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = feed;
	iterator->backwards = backwards;
	iterator->timeline = 1;
	if (start_timestamp) {
		key = (char*)malloc(sizeof(uint64_t) + 1);
		*((uint64_t*)key) = (uint64_t)start_timestamp;
		*(key + sizeof(uint64_t)) = 0;
		iterator->database_iterator = microfeed_database_iterate(feed->database, key, sizeof(uint64_t) + 1, backwards);
		free(key);
	} else {
		iterator->database_iterator = microfeed_database_iterate(feed->database, NULL, 0, backwards);
	}		

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_mutex_unlock(feed->mutex);

	return iterator;
}

void microfeed_feed_remove_subscriber(MicrofeedFeed* feed, const char* bus_name) {
	microfeed_mutex_lock(feed->mutex);

	feed->subscriber_count--;
	feed->reference_count--;
	if (feed->reference_count == 0) {
		microfeed_feed_free(feed);
	} else {

		microfeed_mutex_unlock(feed->mutex);
	}
}

void microfeed_feed_update(MicrofeedFeed* feed, const char* bus_name) {
	MicrofeedError* error;

	microfeed_mutex_lock(feed->mutex);

	if (feed->updating) {
		if (bus_name) {
			microfeed_publisher_send_feed_signal(feed->publisher, bus_name, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->uri);
		}
	} else {
		feed->updating = 1;
		microfeed_publisher_send_feed_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->uri);
	
		if (feed->callbacks.update) {
			if (microfeed_publisher_get_singleton_update_thread(feed->publisher)) {
				microfeed_mutex_unlock(feed->mutex);
				
				error = feed->callbacks.update(feed, (bus_name ? 1 : 0), feed->user_data);

				microfeed_mutex_lock(feed->mutex);
				
				if (error) {
					microfeed_publisher_send_error_signal(feed->publisher, NULL, microfeed_error_get_name(error), feed->uri, NULL, microfeed_error_get_message(error));
					microfeed_error_free(error);
				}
			} else {
				feed->reference_count++;
				if (bus_name) {
					microfeed_publisher_queue_thread(feed->publisher, update_thread_user_initiated, feed);
				} else {
					microfeed_publisher_queue_thread(feed->publisher, update_thread_periodical, feed);
				}
			}
		}
	}
	
	microfeed_mutex_unlock(feed->mutex);
}

void microfeed_feed_republish(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, unsigned int max_count, const char* bus_name) {
	RepublishingData* data;
	
	data = microfeed_memory_allocate(RepublishingData);
	data->feed = feed;
	data->start_uid = start_uid;
	data->end_uid = end_uid;
	data->max_count = max_count;
	data->bus_name = bus_name;
	
	microfeed_publisher_queue_thread(feed->publisher, republish_thread, data);
}

int microfeed_feed_set_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_set) {
	int retval = 0;
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	char status;
	
	microfeed_mutex_lock(feed->mutex);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &data, &data_size)) {
		retval = 1;
		status = *((char*)data);
		if (!(status & (char)status_to_set)) {
			if (status_to_set == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				feed->callbacks.mark_item(feed, uid, 1, feed->user_data);
			}
			status |= (char)(status_to_set);
			microfeed_publisher_send_status_changed_signal(feed->publisher, NULL, feed->uri, uid, status);
			if (status) {
				microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
			} else {
				microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
			}
		}
	} else if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &key, &key_size, &data, &data_size)) {
		if (status_to_set == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
			feed->callbacks.mark_item(feed, uid, 1, feed->user_data);
		}
		status = (char)status_to_set;
		microfeed_publisher_send_status_changed_signal(feed->publisher, NULL, feed->uri, uid, status);
		microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
	}

	microfeed_mutex_unlock(feed->mutex);	

	return retval;
}

int microfeed_feed_unset_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_unset) {
	int retval = 0;
	size_t uid_size;
	void* data;
	size_t data_size;
	char status;
	
	microfeed_mutex_lock(feed->mutex);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &data, &data_size)) {
		retval = 1;
		status = *((char*)data);
		if (status & (char)status_to_unset) {
			if (status_to_unset == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				feed->callbacks.mark_item(feed, uid, 0, feed->user_data);
			}
			status &= (char)(~status_to_unset);
			microfeed_publisher_send_status_changed_signal(feed->publisher, NULL, feed->uri, uid, status);
			if (status) {
				microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
			} else {
				microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
			}
		}			
	}

	microfeed_mutex_unlock(feed->mutex);	

	return retval;
}

void microfeed_feed_unset_item_statuses(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, MicrofeedItemStatus status_to_unset) {
	size_t uid_size;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	char status;
	
	microfeed_mutex_lock(feed->mutex);
	
	uid_size = (start_uid ? strlen(start_uid) + 1 : 0);
	for (iterator = microfeed_database_iterate(feed->statuses_database, start_uid, uid_size, 0);
	     (microfeed_database_iterator_get(iterator, &key, &key_size, &data, &data_size)) && (!end_uid || strcmp ((const char*)key, end_uid) <= 0);
	     microfeed_database_iterator_next(iterator)) {
		status = *((char*)data);
		if (status & (char)status_to_unset) {
			if (status_to_unset == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				feed->callbacks.mark_item(feed, key, 0, feed->user_data);
			}
			status &= (char)(~status_to_unset);
			microfeed_publisher_send_status_changed_signal(feed->publisher, NULL, feed->uri, (const char*)key, status);
			if (status) {
				microfeed_database_replace_data(feed->statuses_database, key, key_size, &status, sizeof(char));
			} else {
				microfeed_database_remove_data(feed->statuses_database, key, key_size);
			}
		}     	
	}
	microfeed_database_iterator_free(iterator);
	
	microfeed_mutex_unlock(feed->mutex);	
}

void microfeed_feed_send_item_data(MicrofeedFeed* feed, const char* uid, const char* bus_name) {
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	SendItemDataData* sid_data;
	const char* existing_bus_name;
	
	microfeed_mutex_lock(feed->mutex);
	
	if (!feed->item_data_database) {
		microfeed_publisher_send_error_signal(feed->publisher, bus_name, MICROFEED_ERROR_NO_ITEM_DATA, feed->uri, uid, "Feed does not support item data");
	} else if (!microfeed_database_index_get_data(feed->database_index, uid, strlen(uid) + 1, &key, &key_size, &data, &data_size)) {
		microfeed_publisher_send_error_signal(feed->publisher, bus_name, MICROFEED_ERROR_NO_SUCH_ITEM, feed->uri, uid, "Item does not exist");		
	}  else {
		if (microfeed_database_get_data(feed->item_data_database, uid, strlen(uid) + 1, &data, &data_size) && data_size > sizeof(uint64_t)) {
			microfeed_publisher_send_item_data_signal(feed->publisher, bus_name, feed->uri, uid, data + sizeof(uint64_t), data_size - sizeof(uint64_t));
			free(data);
		} else {
			if (!(sid_data = microfeed_store_get(feed->send_item_data_datas, uid, SendItemDataData))) {
				sid_data = microfeed_memory_allocate(SendItemDataData);
				sid_data->uid = strdup(uid);
				sid_data->feed = feed;
				sid_data->bus_names = microfeed_store_new_sorted(microfeed_store_compare_keys_direct, microfeed_store_get_key_direct);
				microfeed_store_insert(feed->send_item_data_datas, sid_data);

				microfeed_publisher_queue_thread(feed->publisher, send_item_data_thread, sid_data);
			}
			if (!(existing_bus_name = microfeed_store_get(sid_data->bus_names, bus_name, const char))) {
				microfeed_store_insert(sid_data->bus_names, strdup(bus_name));
			}
		}
	}
	
	microfeed_mutex_unlock(feed->mutex);	
}

void microfeed_feed_ref_item_data(MicrofeedFeed* feed, const char* uid) {
	size_t uid_size;
	uint64_t reference_count;
	size_t size;
	MicrofeedItem* item;
	uint64_t timestamp;
	void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	char status;

	microfeed_mutex_lock(feed->mutex);

	if (feed->item_data_database) {
		uid_size = strlen(uid) + 1;
		size = sizeof(uint64_t);
		if (microfeed_database_get_data_partial(feed->item_data_database, uid, uid_size, &reference_count, &size, 0) && size == sizeof(uint64_t)) {
			reference_count++;
			microfeed_database_replace_data_partial(feed->item_data_database, uid, uid_size, &reference_count, size, 0);
		} else {
			reference_count = 1;
			microfeed_database_replace_data(feed->item_data_database, uid, uid_size, &reference_count, size);
			item = microfeed_item_new(uid, time(NULL));
			timestamp = (uint64_t)microfeed_item_get_timestamp(item);
			key_size = sizeof(uint64_t) + uid_size;
			key = (char*)malloc(key_size);
			*((uint64_t*)key) = timestamp;
			memcpy(key + sizeof(uint64_t), uid, uid_size);
			microfeed_item_marshal_properties(item, &data, &data_size);
			microfeed_publisher_send_item_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->uri, item);
			microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
			microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
		}
	}

	microfeed_mutex_unlock(feed->mutex);
}

void microfeed_feed_unref_item_data(MicrofeedFeed* feed, const char* uid) {
	size_t uid_size;
	uint64_t reference_count;
	size_t size;

	microfeed_mutex_lock(feed->mutex);
	
	if (feed->item_data_database) {	
		uid_size = strlen(uid) + 1;
		size = sizeof(uint64_t);
		if (microfeed_database_get_data_partial(feed->item_data_database, uid, uid_size, &reference_count, &size, 0) && size == sizeof(uint64_t)) {
			reference_count--;
			if (reference_count == 0) {
				microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
				microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
				microfeed_database_remove_data(feed->item_data_database, uid, uid_size);
				microfeed_publisher_send_item_uid_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->uri, uid);				
			} else {
				microfeed_database_replace_data_partial(feed->item_data_database, uid, uid_size, &reference_count, size, 0);
			}
		}
	}
	
	microfeed_mutex_unlock(feed->mutex);
}

/* TODO: Should use defines. */
MicrofeedFeedPermission microfeedfeed_permission_from_string(const char* string) {
	MicrofeedFeedPermission feed_permission = MICROFEED_FEED_PERMISSION_NONE;

	while (*string == ':') {
		string++;
		if (!strncmp(string, "add", 3) && (string[3] == 0 || string[3] == ':')) {
			feed_permission |= MICROFEED_FEED_PERMISSION_ADD;
			string += 3;
		}
	}
	
	return feed_permission;
}

/* TODO: Should use defines. */
char* microfeed_feed_permission_to_string(MicrofeedFeedPermission feed_permission) {
	char buffer[1024];
	char* string = buffer;
	
	if (feed_permission & MICROFEED_FEED_PERMISSION_ADD) {
		memcpy(string, ":add", 4);
		string += 4;
	}
	*string = 0;
	
	return strdup(buffer);
}


void microfeed_feed_iterator_free(MicrofeedFeedIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->feed) {
		iterator->feed->iterators = iterator->next;
	}

	iterator->feed = NULL;
	microfeed_database_iterator_free(iterator->database_iterator);
	microfeed_memory_free(iterator);
}

MicrofeedFeed* microfeed_feed_iterator_get_feed(MicrofeedFeedIterator* iterator) {

	return iterator->feed;
}

MicrofeedItem* microfeed_feed_iterator_get_item(MicrofeedFeedIterator* iterator) {
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_mutex_lock(iterator->feed->mutex);
	
	if (!microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &data, &data_size)) {
		item = NULL;
	} else {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(key + sizeof(uint64_t), (time_t)timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
	}

	microfeed_mutex_unlock(iterator->feed->mutex);
	
	return item;
}

void microfeed_feed_iterator_next(MicrofeedFeedIterator* iterator) {
	microfeed_database_iterator_next(iterator->database_iterator);
}

int microfeed_feed_iterator_jump_and_remove_previous_items(MicrofeedFeedIterator* iterator, const char* uid) {
	int retval = 0;
	size_t uid_size;
	void* start_key;
	size_t start_key_size;
	void* end_key;
	size_t end_key_size;
	void* jump_key;
	size_t jump_key_size;
	const void* key;
	size_t key_size;
	void* data;
	const void* const_data;
	size_t data_size;
	
	microfeed_mutex_lock(iterator->feed->mutex);
	
	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(iterator->feed->database_index, uid, uid_size, &jump_key, &jump_key_size, &data, &data_size)) {
		retval = 1;
		free(data);
		if (microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &const_data, &data_size) && (key_size - sizeof(uint64_t) != uid_size || memcmp(key + sizeof(uint64_t), uid, uid_size))) {
			start_key = malloc(key_size);
			memcpy(start_key, key, key_size);
			start_key_size = key_size;
			end_key = malloc(key_size);
			memcpy(end_key, key, key_size);
			end_key_size = key_size;
			while (microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &const_data, &data_size) && (key_size != jump_key_size || memcmp(key, jump_key, jump_key_size))) {
				end_key = realloc(end_key, key_size);
				memcpy(end_key, key, key_size);
				end_key_size = key_size;
				microfeed_publisher_send_item_uid_signal(iterator->feed->publisher, NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, iterator->feed->uri, key + sizeof(uint64_t));
				microfeed_database_iterator_next(iterator->database_iterator);
			}
			free(jump_key);
			if (iterator->backwards) {		
				if (iterator->timeline) {
					microfeed_database_remove_data_range(iterator->feed->database, end_key, end_key_size, start_key, start_key_size);
				} else {
					microfeed_database_index_remove_data_range(iterator->feed->database_index, end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t), start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t));			
				}
				microfeed_database_remove_data_range(iterator->feed->statuses_database, end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t), start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t));
			} else {
				if (iterator->timeline) {
					microfeed_database_remove_data_range(iterator->feed->database, start_key, start_key_size, end_key, end_key_size);
				} else {
					microfeed_database_index_remove_data_range(iterator->feed->database_index, start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t), end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t));
				}
				microfeed_database_remove_data_range(iterator->feed->statuses_database, start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t), end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t));
			}
			free(start_key);
			free(end_key);
		}
	}
	
	microfeed_mutex_unlock(iterator->feed->mutex);

	return retval;
}

void microfeed_feed_call_modify_item_callback(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item) {
	ModifyItemData* data;
	
	data = microfeed_memory_allocate(ModifyItemData);
	data->feed = feed;
	if (existing_item) {
		data->existing_item = microfeed_item_duplicate(existing_item);
	}
	if (new_item) {
		data->new_item = microfeed_item_duplicate(new_item);
	}

	microfeed_mutex_lock(feed->mutex);

/* TODO: There must NOT be on-going update. Wait somewhere until update ends... */

	if (feed->callbacks.modify_item) {
		feed->reference_count++;
		microfeed_publisher_queue_thread(feed->publisher, modify_item_thread, data);
	} else {
		/* TODO: Send CannotModifyItem signal */
	}
	
	microfeed_mutex_unlock(feed->mutex);
}

static void index_function(void* key, const size_t key_size, void* data, const size_t data_size, void** index_key, size_t* index_key_size) {
	*index_key = key + sizeof(uint64_t);
	*index_key_size = key_size - sizeof(uint64_t);
}

static int compare_function(const void* key1, const size_t key1_size, const void* key2, const size_t key2_size) {
	int retvalue = 0;
	
	if (key1_size > sizeof(uint64_t) && key2_size > sizeof(int64_t)) {
		if (!(retvalue = *((uint64_t*)key1) - *((uint64_t*)key2))) {
			retvalue = strcmp(key1 + sizeof(uint64_t), key2 + sizeof(uint64_t));
		}
	}

	return retvalue;
}

static void* update_thread_periodical(void* data) {
	MicrofeedFeed* feed;
	MicrofeedError* error;
	
	feed = (MicrofeedFeed*)data;
	error = feed->callbacks.update(feed, 0, feed->user_data);

	microfeed_mutex_lock(feed->mutex);

	if (error) {
		microfeed_publisher_send_error_signal(feed->publisher, NULL, microfeed_error_get_name(error), feed->uri, NULL, microfeed_error_get_message(error));
		microfeed_error_free(error);
	} else {
		remove_old_items(feed);
	}
	microfeed_publisher_send_feed_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, feed->uri);
	feed->updating = 0;
	feed->reference_count--;
	if (feed->reference_count == 0) {
		microfeed_feed_free(feed);
	} else {

		microfeed_mutex_unlock(feed->mutex);
	}

	return NULL;
}

static void* update_thread_user_initiated(void* data) {
	MicrofeedFeed* feed;
	MicrofeedError* error;
	
	feed = (MicrofeedFeed*)data;
	error = feed->callbacks.update(feed, 1, feed->user_data);

	microfeed_mutex_lock(feed->mutex);
	
	if (error) {
		microfeed_publisher_send_error_signal(feed->publisher, NULL, microfeed_error_get_name(error), feed->uri, NULL, microfeed_error_get_message(error));
		microfeed_error_free(error);
	} else {
		remove_old_items(feed);	
	}
	microfeed_publisher_send_feed_signal(feed->publisher, NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, feed->uri);
	feed->updating = 0;
	feed->reference_count--;
	if (feed->reference_count == 0) {
		microfeed_feed_free(feed);
	} else {

		microfeed_mutex_unlock(feed->mutex);
	}
	
	return NULL;
}

static void* republish_thread(void* user_data) {
	RepublishingData* data;
	unsigned int count;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* item_data;
	size_t item_data_size;
	uint64_t timestamp;
	MicrofeedItem* item;
	void* status_data;
	size_t status_data_size;
	char status;
	

	data = (RepublishingData*)user_data;

	microfeed_mutex_lock(data->feed->mutex);
	
	microfeed_publisher_send_feed_signal(data->feed->publisher, data->bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED, data->feed->uri);

	count = 0;
	for (iterator = microfeed_database_iterate(data->feed->database, data->end_uid, (data->end_uid ? strlen(data->end_uid) + 1 : 0), 1);
	     microfeed_database_iterator_get(iterator, &key, &key_size, &item_data, &item_data_size);
	     microfeed_database_iterator_next(iterator)) {
	    timestamp = *((uint64_t*)key);
		if (microfeed_database_get_data(data->feed->statuses_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t), &status_data, &status_data_size)) {
			status = *((char*)status_data);
		} else {
			status = (char)MICROFEED_ITEM_STATUS_NONE;
		}
		item = microfeed_item_new_with_status(key + sizeof(uint64_t), (time_t)timestamp, (MicrofeedItemStatus)status);
		microfeed_item_demarshal_properties(item, item_data, item_data_size);
		microfeed_publisher_send_item_signal(data->feed->publisher, data->bus_name, MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED, data->feed->uri, item);
		microfeed_item_free(item);
	    if (data->start_uid && strcmp(data->start_uid, key + sizeof(uint64_t)) <= 0) {
	    	break;
	    }
	    if (++count == data->max_count) {
	    	break;
	    }
	}
	microfeed_database_iterator_free(iterator);

	microfeed_publisher_send_feed_signal(data->feed->publisher, data->bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED, data->feed->uri);

	microfeed_mutex_unlock(data->feed->mutex);	
	
	return NULL;
}

static void* send_item_data_thread(void* data) {
	SendItemDataData* sid_data;
	MicrofeedError* error;
	MicrofeedStoreIterator* iterator;
	void* item_data;
	size_t item_data_size;
	char* bus_name;

	sid_data = (SendItemDataData*)data;
	error = sid_data->feed->callbacks.download_item_data(sid_data->feed, sid_data->uid, &item_data, &item_data_size, sid_data->feed->user_data);

	microfeed_mutex_lock(sid_data->feed->mutex);
	
	if (error) {
		for (iterator = microfeed_store_iterate(sid_data->bus_names, NULL);
		     (bus_name = microfeed_store_iterator_get(iterator, char));
		     microfeed_store_iterator_next(iterator)) {		     	
			microfeed_publisher_send_error_signal(sid_data->feed->publisher, bus_name, microfeed_error_get_name(error), sid_data->feed->uri, NULL, microfeed_error_get_message(error));
			free(bus_name);
		}
		microfeed_store_iterator_free(iterator);
		microfeed_error_free(error);
	} else {
		for (iterator = microfeed_store_iterate(sid_data->bus_names, NULL);
		     (bus_name = microfeed_store_iterator_get(iterator, char));
		     microfeed_store_iterator_next(iterator)) {		     	
			microfeed_publisher_send_item_data_signal(sid_data->feed->publisher, bus_name, sid_data->feed->uri, sid_data->uid, item_data, item_data_size);
			free(bus_name);
		}
		microfeed_store_iterator_free(iterator);

		item_data = realloc(item_data, item_data_size + sizeof(uint64_t));
		memmove(item_data + sizeof(uint64_t), item_data, item_data_size);
		item_data_size += sizeof(uint64_t);
		*((uint64_t*)item_data) = 1;
		microfeed_database_replace_data(sid_data->feed->item_data_database, sid_data->uid, strlen(sid_data->uid) + 1, item_data, item_data_size);
		free(item_data);
	}
	microfeed_store_remove(sid_data->feed->send_item_data_datas, sid_data);
	
	microfeed_mutex_unlock(sid_data->feed->mutex);

	microfeed_store_free(sid_data->bus_names);
	free(sid_data->uid);
	microfeed_memory_free(sid_data);
	
	return NULL;
}

static void* modify_item_thread(void* user_data) {
	ModifyItemData* data;
	MicrofeedError* error;
	
	data = (ModifyItemData*)user_data;
	error = data->feed->callbacks.modify_item(data->feed, data->existing_item, data->new_item, data->feed->user_data);

	microfeed_mutex_lock(data->feed->mutex);
	
	if (error) {
		microfeed_publisher_send_error_signal(data->feed->publisher, NULL, microfeed_error_get_name(error), data->feed->uri, microfeed_item_get_uid((data->existing_item ? data->existing_item : data->new_item)), microfeed_error_get_message(error));
		microfeed_error_free(error);
	}
	
	microfeed_item_free(data->existing_item);
	microfeed_item_free(data->new_item);

	microfeed_mutex_unlock(data->feed->mutex);

	return NULL;	
}

static void remove_old_items(MicrofeedFeed* feed) {
	time_t timestamp;
	int keep_old_items;
	
	keep_old_items = atoi(microfeed_publisher_get_setting_value(feed->publisher, "keep_old_items", "0"));
	timestamp = time(NULL) + 60 * 60 * keep_old_items;
	
	/* TODO! */
	
}

static const char* send_item_data_data_get_uid(SendItemDataData* send_item_data_data) {
	
	return send_item_data_data->uid;
}
