
#ifndef MICROFEEDSUBSCRIBER_H
#define MICROFEEDSUBSCRIBER_H

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#include <time.h>

/**
 * @addtogroup MicrofeedSubscriber
 *
 * This module implementing a subscriber's connection to DBus system is used in the subscriber side.
 *
 * @{
 */

/**
 * Opaque data type representing Microfeed subscriber.
 */
typedef struct _MicrofeedSubscriber MicrofeedSubscriber;

typedef void* MicrofeedPublisherHandle;

/** @} */

#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedconfiguration.h>

/**
 * @addtogroup MicrofeedSubscriber
 * @{
 */
typedef void (*MicrofeedSubscriberReplyCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data);
typedef void (*MicrofeedSubscriberErrorCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data);
typedef void (*MicrofeedSubscriberFeedCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data);
typedef void (*MicrofeedSubscriberItemCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data);
typedef void (*MicrofeedSubscriberItemRemovedCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, void* user_data);
typedef void (*MicrofeedSubscriberStatusCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const MicrofeedItemStatus status, void* user_data);
typedef void (*MicrofeedSubscriberItemDataCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t length, void* user_data);

typedef struct {
	MicrofeedSubscriberErrorCallback error_occured;
	MicrofeedSubscriberFeedCallback feed_update_started;
	MicrofeedSubscriberFeedCallback feed_update_ended;
	MicrofeedSubscriberFeedCallback feed_republishing_started;
	MicrofeedSubscriberFeedCallback feed_republishing_ended;	
	MicrofeedSubscriberItemCallback item_added;
	MicrofeedSubscriberItemCallback item_changed;
	MicrofeedSubscriberItemCallback item_republished;
	MicrofeedSubscriberItemRemovedCallback item_removed;
	MicrofeedSubscriberStatusCallback item_status_changed;
	MicrofeedSubscriberItemDataCallback item_data_received;
} MicrofeedSubscriberCallbacks;

MicrofeedSubscriber* microfeed_subscriber_new(const char* object_path, MicrofeedConfiguration* configuration, DBusConnection* connection);
void microfeed_subscriber_free(MicrofeedSubscriber* subscriber);

int microfeed_subscriber_add_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_modify_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_remove_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_subscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_unsubscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_update_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_republish_items(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* start_uid, const char* end_uid, unsigned int max_count, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_create_publisher(MicrofeedSubscriber* subscriber, const char* publisher, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_destroy_publisher(MicrofeedSubscriber* subscriber, const char* publisher, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_send_item_data(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_mark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_unmark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_read_items(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* start_uid, const char* end_uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
MicrofeedConfiguration* microfeed_subscriber_get_configuration(MicrofeedSubscriber* subscriber);

/** @} */ 

#endif
