
#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <dirent.h>

struct _MicrofeedSubscriber {
	DBusConnection* connection;
	char* object_path;
	MicrofeedConfiguration* configuration;
	
	MicrofeedStore* publishers_by_identifier;
	MicrofeedStore* publishers_by_unique_connection_name;
};

typedef struct {
	MicrofeedSubscriber* subscriber;
	char* identifier;
	char* object_path;
	char* bus_name;
	char* unique_connection_name;
	time_t last_activity;
	MicrofeedStore* feeds;
} Publisher;

typedef struct {
	unsigned int reference_count;
	Publisher* publisher;
	char* uri;
	char* name;
	MicrofeedSubscriberCallbacks callbacks;
	void* user_data;
} Feed;

typedef struct {
	const char* name;
	void (*callback)(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
} SignalCallback;

typedef struct {
	Publisher* publisher;
	char* uri;
	char* uid;
	MicrofeedSubscriberErrorCallback callback;
	void* user_data;
} MethodReturnData;

static Publisher* publisher_new(MicrofeedSubscriber* subscriber, const char* identifier);
static void publisher_free(Publisher* publisher);
static Feed* feed_new(Publisher* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* user_data);
static void feed_free(Feed* feed);

static void call_publisher_method(Publisher* publisher, const char* uri, const char* uid, DBusMessage* message, MicrofeedSubscriberErrorCallback callback, void* user_data);
static MicrofeedItem* parse_item_from_message(DBusMessageIter* iter);
static int parse_item_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return, MicrofeedItem** item_return);
static int parse_feed_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return);
static Publisher* get_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data);
static Feed* get_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data);
static const char* feed_get_uri(Feed* feed);
static const char* publisher_get_identifier(Publisher* publisher);
static const char* publisher_get_unique_connection_name(Publisher* publisher);
static void object_unregister(DBusConnection* connection, void* user_data);
static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data);

static void signal_feed_update_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_update_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_republishing_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_feed_republishing_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_data(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_added(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_republished(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_removed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);
static void signal_item_status_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message);

static SignalCallback signal_callbacks[] = {
	{ MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, signal_feed_update_started },
	{ MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, signal_feed_update_ended },
	{ MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED, signal_feed_republishing_started },
	{ MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED, signal_feed_republishing_ended },
	{ MICROFEED_SIGNAL_NAME_ITEM_DATA, signal_item_data },
	{ MICROFEED_SIGNAL_NAME_ITEM_ADDED, signal_item_added },
	{ MICROFEED_SIGNAL_NAME_ITEM_CHANGED, signal_item_changed },
	{ MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED, signal_item_republished },
	{ MICROFEED_SIGNAL_NAME_ITEM_REMOVED, signal_item_removed },
	{ MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED, signal_item_status_changed },
	{ NULL, NULL }
};

static DBusObjectPathVTable object_vtable = {
	object_unregister,
	object_message
};	

MicrofeedSubscriber* microfeed_subscriber_new(const char* object_path, MicrofeedConfiguration* configuration, DBusConnection* connection) {
	MicrofeedSubscriber* subscriber;
	
	subscriber = microfeed_memory_allocate(MicrofeedSubscriber);
	subscriber->connection = connection;
	subscriber->object_path = strdup(object_path);
	subscriber->configuration = configuration;
	
	subscriber->publishers_by_identifier = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                           (MicrofeedStoreGetKeyFunction)publisher_get_identifier);
	subscriber->publishers_by_unique_connection_name = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
	                                                                       (MicrofeedStoreGetKeyFunction)publisher_get_unique_connection_name);

	if (!dbus_connection_register_object_path(subscriber->connection, subscriber->object_path, &object_vtable, subscriber)) {
	
		return NULL;
	}
	if (!dbus_connection_add_filter(subscriber->connection, object_message, subscriber, NULL)) {

		return NULL;
	}
	
	return subscriber;
}

void microfeed_subscriber_free(MicrofeedSubscriber* subscriber) {
	/* TODO */
}

int microfeed_subscriber_add_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedItem* item, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	DBusMessageIter iter;
	const char* uid;
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, microfeed_item_get_uid(item), callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_ADD_ITEM);
		dbus_message_iter_init_append(message, &iter);
		uid = microfeed_item_get_uid(item);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
		for (iterator = microfeed_item_iterate_properties(item, NULL);
		     microfeed_item_iterator_get(iterator, &key, &value);
		     microfeed_item_iterator_next(iterator)) {
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
		}
		microfeed_item_iterator_free(iterator);
		call_publisher_method(publisher, uri, microfeed_item_get_uid(item), message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_modify_item(MicrofeedSubscriber*subscriber, const char* publisher_identifier, const char* uri, MicrofeedItem* item, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	DBusMessageIter iter;
	const char* uid;
	MicrofeedItemIterator* iterator;
	const char* key;
	const char* value;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, microfeed_item_get_uid(item), callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_MODIFY_ITEM);
		dbus_message_iter_init_append(message, &iter);
		uid = microfeed_item_get_uid(item);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
		for (iterator = microfeed_item_iterate_properties(item, NULL);
		     microfeed_item_iterator_get(iterator, &key, &value);
		     microfeed_item_iterator_next(iterator)) {
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
			dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
		}
		microfeed_item_iterator_free(iterator);
		call_publisher_method(publisher, uri, microfeed_item_get_uid(item), message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_remove_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, uid, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_REMOVE_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;

}

int microfeed_subscriber_subscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberCallbacks* callbacks, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	Feed* feed;
	DBusMessage* message;

	if ((publisher = get_publisher(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
			if (callback) {
				callback(subscriber, publisher_identifier, uri, NULL, MICROFEED_ERROR_FEED_ALREADY_SUBSCRIBED, "Subscribed a feed that is already subscribed.", user_data);
			}
		} else {
			feed = feed_new(publisher, uri, callbacks, user_data);
			microfeed_store_insert(publisher->feeds, feed);
		
			message = dbus_message_new_method_call(feed->publisher->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_SUBSCRIBE_FEED);
			dbus_message_append_args(message, DBUS_TYPE_STRING, &feed->uri, DBUS_TYPE_INVALID);
			call_publisher_method(publisher, uri, NULL, message, callback, user_data);
			dbus_message_unref(message);
			retvalue = 1;
		}
	}
	
	return retvalue;
}

int microfeed_subscriber_unsubscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	Publisher* publisher;
	
	if ((feed = get_feed(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UNSUBSCRIBE_FEED);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &feed->uri, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	
		publisher = feed->publisher;
		microfeed_store_remove(feed->publisher->feeds, feed);
		feed_free(feed);
	}
	
	return retvalue;
}

int microfeed_subscriber_update_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	
	if ((feed = get_feed(subscriber, publisher_identifier, uri, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UPDATE_FEED);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_republish_items(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* start_uid, const char* end_uid, unsigned int max_count, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	const char* start_uid_dbus;
	const char* end_uid_dbus;
	dbus_uint16_t max_count_dbus;

	if ((feed = get_feed(subscriber, publisher_identifier, uri, start_uid, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_REPUBLISH_ITEMS);
		start_uid_dbus = (start_uid ? start_uid : "");
		end_uid_dbus = (end_uid ? end_uid : "");
		max_count_dbus = max_count;
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &start_uid_dbus, DBUS_TYPE_STRING, &end_uid_dbus, DBUS_TYPE_UINT16, &max_count_dbus, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, start_uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_send_item_data(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, uid, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_SEND_ITEM_DATA);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, NULL, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_create_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = microfeed_store_get(subscriber->publishers_by_identifier, publisher_identifier, Publisher))) {
		if (callback) {
			callback(subscriber, publisher_identifier, NULL, NULL, MICROFEED_ERROR_PUBLISHER_ALREADY_EXISTS, "Trying to create a publisher that already exists.", user_data);
		}
	} else if (!(publisher = publisher_new(subscriber, publisher_identifier))) {
		if (callback) {
			callback(subscriber, publisher_identifier, NULL, NULL, MICROFEED_ERROR_INVALID_PUBLISHER_IDENTIFIER, "Publisher identifier was invalid.", user_data);
		}
	} else if (microfeed_configuration_get_publisher_directory(subscriber->configuration, publisher_identifier)) {
		if (callback) {
			callback(subscriber, publisher_identifier, NULL, NULL, MICROFEED_ERROR_PUBLISHER_ALREADY_EXISTS, "Trying to create a publisher that already exists.", user_data);
		}		
	} else {
		microfeed_store_insert(subscriber->publishers_by_identifier, publisher);

		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_CREATE_PUBLISHER);
		call_publisher_method(publisher, NULL, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}

	return retvalue;
}

int microfeed_subscriber_destroy_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_DESTROY_PUBLISHER);
		call_publisher_method(publisher, NULL, NULL, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_mark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_MARK_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}

int microfeed_subscriber_unmark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	int retvalue = 0;
	Publisher* publisher;
	DBusMessage* message;
	
	if ((publisher = get_publisher(subscriber, publisher_identifier, NULL, NULL, callback, user_data))) {
		message = dbus_message_new_method_call(publisher->bus_name, publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_UNMARK_ITEM);
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
		call_publisher_method(publisher, uri, uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;	
}

int microfeed_subscriber_read_items(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* start_uid, const char* end_uid, MicrofeedSubscriberReplyCallback callback, void* user_data) {
	int retvalue = 0;
	Feed* feed;
	DBusMessage* message;
	const char* start_uid_dbus;
	const char* end_uid_dbus;

	if ((feed = get_feed(subscriber, publisher_identifier, uri, start_uid, callback, user_data))) {
		message = dbus_message_new_method_call(feed->publisher->bus_name, feed->publisher->object_path, MICROFEED_DBUS_INTERFACE_PUBLISHER, MICROFEED_METHOD_NAME_READ_ITEMS);
		start_uid_dbus = (start_uid ? start_uid : "");
		end_uid_dbus = (end_uid ? end_uid : "");
		dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &start_uid_dbus, DBUS_TYPE_STRING, &end_uid_dbus, DBUS_TYPE_INVALID);
		call_publisher_method(feed->publisher, feed->uri, start_uid, message, callback, user_data);
		dbus_message_unref(message);
		retvalue = 1;
	}
	
	return retvalue;
}


MicrofeedConfiguration* microfeed_subscriber_get_configuration(MicrofeedSubscriber* subscriber) {
	
	return subscriber->configuration;
}


static Publisher* publisher_new(MicrofeedSubscriber* subscriber, const char* identifier) {
	Publisher* publisher;
	char* colon;

	publisher = microfeed_memory_allocate(Publisher);
	publisher->subscriber = subscriber;
	publisher->identifier = strdup(identifier); 
	if (!(colon = strchr(publisher->identifier, ':')) || colon[1] == 0) {
		free(publisher->identifier);
		microfeed_memory_free(publisher);
		publisher = NULL;
	} else {
		*colon = 0;
		publisher->object_path = microfeed_util_string_concatenate(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, publisher->identifier, NULL);
		publisher->bus_name = strdup(colon + 1);
		*colon = ':';
		publisher->unique_connection_name = NULL;
		publisher->last_activity = 0;
		publisher->feeds = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
		                                       (MicrofeedStoreGetKeyFunction)feed_get_uri);
	}
	
	return publisher;
}

static void publisher_free(Publisher* publisher) {
	char buffer[1024];

	if (publisher->unique_connection_name) {
		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", publisher->unique_connection_name);
		dbus_bus_remove_match(publisher->subscriber->connection, buffer, NULL);
		snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s'", publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_PUBLISHER, publisher->object_path);
		dbus_bus_remove_match(publisher->subscriber->connection, buffer, NULL);
		snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s',destination='%s'", publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, publisher->object_path, dbus_bus_get_unique_name(publisher->subscriber->connection));
		dbus_bus_remove_match(publisher->subscriber->connection, buffer, NULL);
		snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s'", publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_ERROR, publisher->object_path);
		dbus_bus_remove_match(publisher->subscriber->connection, buffer, NULL);
		snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s',destination ='%s'", publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION, publisher->object_path, dbus_bus_get_unique_name(publisher->subscriber->connection));
		dbus_bus_remove_match(publisher->subscriber->connection, buffer, NULL);
	}
	
	free(publisher->identifier);
	free(publisher->object_path);
	free(publisher->bus_name);
	free(publisher->unique_connection_name);
	publisher->subscriber = NULL;
	publisher->identifier = NULL;
	publisher->object_path = NULL;
	publisher->bus_name = NULL;
	publisher->unique_connection_name = NULL;
	microfeed_store_free(publisher->feeds);
	microfeed_memory_free(publisher);
}

static Feed* feed_new(Publisher* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* user_data) {
	Feed* feed;
	
	feed = microfeed_memory_allocate(Feed);
	feed->reference_count = 1;
	feed->publisher = publisher;
	feed->uri = strdup(uri);
	feed->callbacks = *callbacks;
	feed->user_data = user_data;
	
	return feed;
}
	
static void feed_free(Feed* feed) {
	free(feed->uri);
	free(feed->name);
	feed->publisher = NULL;
	feed->uri = NULL;
	feed->name = NULL;
	feed->user_data = NULL;
	microfeed_memory_free(feed);
}
/*
Feed* feed_ref(Feed* feed) {
	feed->reference_count++;
	
	return feed;
}

void feed_unref(Feed* feed) {
	feed->reference_count--;
	if (feed->reference_count == 0) {
		feed_free(feed);
	}
}
*/
static void signal_feed_update_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	
	if (parse_feed_signal(publisher, message, &feed) && feed->callbacks.feed_update_started) {
		feed->callbacks.feed_update_started(subscriber, publisher->identifier, feed->uri, feed->user_data);
	}
}

static void signal_feed_update_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	
	if (parse_feed_signal(publisher, message, &feed) && feed->callbacks.feed_update_ended) {
		feed->callbacks.feed_update_ended(subscriber, publisher->identifier, feed->uri, feed->user_data);
	}
}

static void signal_feed_republishing_started(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	
	if (parse_feed_signal(publisher, message, &feed) && feed->callbacks.feed_republishing_started) {
		feed->callbacks.feed_republishing_started(subscriber, publisher->identifier, feed->uri, feed->user_data);
	}
}

static void signal_feed_republishing_ended(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	
	if (parse_feed_signal(publisher, message, &feed) && feed->callbacks.feed_republishing_ended) {
		feed->callbacks.feed_republishing_ended(subscriber, publisher->identifier, feed->uri, feed->user_data);
	}
}

static void signal_item_data(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusMessageIter iter;
	char* uri;
	Feed* feed;
	char* uid;
	DBusMessageIter subiter;
	char* data;
	int length;
		
	if (dbus_message_iter_init(message, &iter) && dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
		dbus_message_iter_get_basic(&iter, &uri);
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed)) && feed->callbacks.item_data_received &&
		    dbus_message_iter_next(&iter) && dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
			dbus_message_iter_get_basic(&iter, &uid);
			if (dbus_message_iter_next(&iter) && dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_ARRAY &&
			    dbus_message_iter_get_element_type(&iter) == DBUS_TYPE_BYTE) {
				dbus_message_iter_recurse(&iter, &subiter);
				dbus_message_iter_get_fixed_array(&subiter, &data, &length);
				feed->callbacks.item_data_received(subscriber, publisher->identifier, uri, uid, data, length, feed->user_data);
			}
		}
	}
}

static void signal_item_added(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	
	if (parse_item_signal(publisher, message, &feed, &item) && feed->callbacks.item_added) {
		feed->callbacks.item_added(subscriber, publisher->identifier, feed->uri, item, feed->user_data);
		microfeed_item_free(item);
	}
}

static void signal_item_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	
	if (parse_item_signal(publisher, message, &feed, &item) && feed->callbacks.item_changed) {
		feed->callbacks.item_changed(subscriber, publisher->identifier, feed->uri, item, feed->user_data);
		microfeed_item_free(item);
	}
}

static void signal_item_republished(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	Feed* feed;
	MicrofeedItem* item;
	
	if (parse_item_signal(publisher, message, &feed, &item) && feed->callbacks.item_republished) {
		feed->callbacks.item_republished(subscriber, publisher->identifier, feed->uri, item, feed->user_data);
		microfeed_item_free(item);
	}
}

static void signal_item_removed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusError error;
	char* uri;
	char* uid;
	Feed* feed;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID) &&
	    (feed = microfeed_store_get(publisher->feeds, uri, Feed)) && feed->callbacks.item_removed) {
		feed->callbacks.item_removed(subscriber, publisher->identifier, uri, uid, feed->user_data);
	}
}

static void signal_item_status_changed(MicrofeedSubscriber* subscriber, Publisher* publisher, DBusMessage* message) {
	DBusError error;
	char* uri;
	char* uid;
	char status;
	Feed* feed;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_BYTE, &status, DBUS_TYPE_INVALID) &&
	    (feed = microfeed_store_get(publisher->feeds, uri, Feed)) && feed->callbacks.item_status_changed) {
		feed->callbacks.item_status_changed(subscriber, publisher->identifier, uri, uid, (MicrofeedItemStatus)status, feed->user_data);
	}
}

static const char* feed_get_uri(Feed* feed) {

	return feed->uri;
}

static const char* publisher_get_unique_connection_name(Publisher* publisher) {
	
	return publisher->unique_connection_name;
}

static const char* publisher_get_identifier(Publisher* publisher) {
	
	return publisher->identifier;
}

static Publisher* get_publisher(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	Publisher* publisher = NULL;
	
	if (!(publisher = microfeed_store_get(subscriber->publishers_by_identifier, publisher_identifier, Publisher))) {
		if (!(publisher = publisher_new(subscriber, publisher_identifier))) {
			if (callback) {
				callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_INVALID_PUBLISHER_IDENTIFIER, "Publisher identifier was invalid.", user_data);
			}	    
		} else if (!microfeed_configuration_get_provider_name(subscriber->configuration, publisher->bus_name)) {
			publisher_free(publisher);
			publisher = NULL;
			if (callback) {
				callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_NO_SUCH_PROVIDER, "Provider for the publisher does not exist.", user_data);
			}
		} else {
			microfeed_store_insert(subscriber->publishers_by_identifier, publisher);
		}
	}

	return publisher;
}

static Feed* get_feed(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	Publisher* publisher;
	Feed* feed = NULL;
	
	if (!(publisher = microfeed_store_get(subscriber->publishers_by_identifier, publisher_identifier, Publisher))) {
		if (callback) {
			callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_NO_SUCH_PUBLISHER, "Publisher does not exist.", user_data);
		}	    
	} else if (!(feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
		if (callback) {
			callback(subscriber, publisher_identifier, uri, uid, MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to access a feed that is not subscribed.", user_data);
		}
	}
	
	return feed;
}

static MicrofeedItem* parse_item_from_message(DBusMessageIter* iter) {
	MicrofeedItem* item = NULL;
	char* uid;
	dbus_uint64_t timestamp;
	unsigned char status;
	char* key;
	char* value;

	if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
		dbus_message_iter_get_basic(iter, &uid);
		if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT64) {
			dbus_message_iter_get_basic(iter, &timestamp);
			if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_BYTE) {
				dbus_message_iter_get_basic(iter, &status);
				item = microfeed_item_new_with_status(uid, (time_t)timestamp, (MicrofeedItemStatus)status);
				while (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
					dbus_message_iter_get_basic(iter, &key);
					if (dbus_message_iter_next(iter) && dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING) {
						dbus_message_iter_get_basic(iter, &value);
						microfeed_item_set_property(item, key, value);
					}
				}
			}
		} else {
			item = microfeed_item_new(uid, 0);
		}
	}
	
	return item;
}

static int parse_item_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return, MicrofeedItem** item_return) {
	int retvalue = 0;
	DBusMessageIter iter;
	char* uri;
	Feed* feed;
	MicrofeedItem* item;
	
	if (dbus_message_iter_init(message, &iter) && dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
		dbus_message_iter_get_basic(&iter, &uri);
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed)) && 
		    (item = parse_item_from_message(&iter))) {
			*feed_return = feed;
			*item_return = item;
			retvalue = 1;
		}
	}

	return retvalue;
}

static int parse_feed_signal(Publisher* publisher, DBusMessage* message, Feed** feed_return) {
	int retvalue = 0;
	DBusError error;
	const char* uri;
	Feed* feed;
	
	dbus_error_init(&error);
	if (dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		if ((feed = microfeed_store_get(publisher->feeds, uri, Feed))) {
			*feed_return = feed;
			retvalue = 1;
		}
	} else {
		dbus_error_free(&error);
	}
	
	return retvalue;
}

static void handle_publisher_method_return(DBusPendingCall* pending, void* user_data) {
	MethodReturnData* data;
	DBusMessage* reply;
	DBusError error;
	const char* error_name;
	const char* error_message;
	char buffer[1024];
	
	data = (MethodReturnData*)user_data;
	reply = dbus_pending_call_steal_reply(pending);
	if (!reply) {
		if (data->callback) {
			data->callback(data->publisher->subscriber, data->publisher->identifier, data->uri, data->uid, MICROFEED_ERROR_DBUS_MESSAGE_FAILED, "No reply from the publisher.", data->user_data);
		}
	} else if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR) {
		if (data->callback) {
			if (!(error_name = dbus_message_get_error_name(reply))) {
				error_name = MICROFEED_ERROR_UNKNOWN;
			}
			dbus_error_init(&error);
			if (!dbus_message_get_args(reply, &error, DBUS_TYPE_STRING, &error_message, DBUS_TYPE_INVALID)) {
				error_message = NULL;
			}
			data->callback(data->publisher->subscriber, data->publisher->identifier, data->uri, data->uid, error_name, error_message, data->user_data);
		}
	} else {
		if (!data->publisher->unique_connection_name) {
			data->publisher->unique_connection_name = strdup(dbus_message_get_sender(reply));
			microfeed_store_insert(data->publisher->subscriber->publishers_by_unique_connection_name, data->publisher);

			snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", data->publisher->unique_connection_name);
			dbus_bus_add_match(data->publisher->subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s'", data->publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_PUBLISHER, data->publisher->object_path);
			dbus_bus_add_match(data->publisher->subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s',destination='%s'", data->publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, data->publisher->object_path, dbus_bus_get_unique_name(data->publisher->subscriber->connection));
			dbus_bus_add_match(data->publisher->subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s'", data->publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_ERROR, data->publisher->object_path);
			dbus_bus_add_match(data->publisher->subscriber->connection, buffer, NULL);
			snprintf(buffer, 1024, "type='signal',sender='%s',interface='%s',path='%s',destination ='%s'", data->publisher->unique_connection_name, MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION, data->publisher->object_path, dbus_bus_get_unique_name(data->publisher->subscriber->connection));
			dbus_bus_add_match(data->publisher->subscriber->connection, buffer, NULL);
		}
		if (data->callback) {
			data->callback(data->publisher->subscriber, data->publisher->identifier, data->uri, data->uid, NULL, NULL, data->user_data);
		}
	}
	if (reply) {
		dbus_message_unref(reply);
	}
	
	if (microfeed_store_get_size(data->publisher->feeds) == 0) {
		microfeed_store_remove(data->publisher->subscriber->publishers_by_identifier, data->publisher);
		microfeed_store_remove(data->publisher->subscriber->publishers_by_unique_connection_name, data->publisher);
		publisher_free(data->publisher);
	}
	
	free(data->uri);
	free(data->uid);
	microfeed_memory_free(data);		
}

static void call_publisher_method(Publisher* publisher, const char* uri, const char* uid, DBusMessage* message, MicrofeedSubscriberErrorCallback callback, void* user_data) {
	DBusPendingCall* pending_call;
	MethodReturnData* data;
	
	dbus_connection_send_with_reply(publisher->subscriber->connection, message, &pending_call, -1);
	if (pending_call) {
		data = microfeed_memory_allocate(MethodReturnData);
		data->publisher = publisher;
		if (uri) {
			data->uri = strdup(uri);
		} else {
			data->uri = NULL;
		}
		if (uid) {
			data->uid = strdup(uid);
		} else {
			data->uid = NULL;
		}
		data->callback = callback;
		data->user_data = user_data;
		dbus_pending_call_set_notify(pending_call, handle_publisher_method_return, data, NULL);
	}
}

static void object_unregister(DBusConnection* connection, void* user_data) {
}

static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	MicrofeedSubscriber* subscriber;
	const char* name;
	const char* old_owner;
	const char* new_owner;
	MicrofeedStoreIterator* iterator;
	Feed* feed;
	int i;
	Publisher* publisher;
	DBusMessage* reply;
	DBusError error;
	const char* error_name;
	char* error_message;
	char* uri;
	char* uid;

	subscriber = (MicrofeedSubscriber*)user_data;
	if (dbus_message_is_signal(message, "org.freedesktop.DBus", "NameOwnerChanged") &&
	    dbus_message_get_args(message, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &old_owner, DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID) &&
	    name[0] == ':' && new_owner[0] == 0 && !strcmp(name, old_owner) && (publisher = microfeed_store_get(subscriber->publishers_by_unique_connection_name, name, Publisher))) {
		for (iterator = microfeed_store_iterate(publisher->feeds, NULL); (feed = microfeed_store_iterator_get(iterator, Feed)); microfeed_store_iterator_next(iterator)) {
			feed->callbacks.error_occured(subscriber, publisher->identifier, feed->uri, NULL, MICROFEED_ERROR_PROVIDER_CLOSED_CONNECTION, "Provider closed connection", feed->user_data);	
			feed_free(feed);
		}
/* TODO: If there is a pending call, the return of it will crash because publisher has been destroyed. */
		microfeed_store_iterator_free(iterator);
		microfeed_store_remove(subscriber->publishers_by_unique_connection_name, publisher);
		microfeed_store_remove(subscriber->publishers_by_identifier, publisher);
		publisher_free(publisher);
	}

	if ((publisher = microfeed_store_get(subscriber->publishers_by_unique_connection_name, dbus_message_get_sender(message), Publisher))) {
		publisher->last_activity = time(NULL);
		if (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_SUBSCRIBER, MICROFEED_METHOD_NAME_PING)) {
			reply = dbus_message_new_method_return(message);
			dbus_connection_send(connection, reply, NULL);
			dbus_message_unref(reply);
			result = DBUS_HANDLER_RESULT_HANDLED;
		} else if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_SIGNAL && (dbus_message_has_interface(message, MICROFEED_DBUS_INTERFACE_ERROR) || dbus_message_has_interface(message, MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION))) {
			dbus_error_init(&error);
			if ((error_name = dbus_message_get_member(message)) && dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_STRING, &error_message, DBUS_TYPE_INVALID)) {
				if ((feed = microfeed_store_get(publisher->feeds, uri, Feed)) && feed->callbacks.error_occured) { 
					feed->callbacks.error_occured(subscriber, publisher->identifier, uri, uid, error_name, error_message, feed->user_data);
				}
			}
		} else {
			for (i = 0; signal_callbacks[i].name; i++) {
				if (dbus_message_is_signal(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, signal_callbacks[i].name) ||
				    dbus_message_is_signal(message, MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION, signal_callbacks[i].name)) {
					signal_callbacks[i].callback(subscriber, publisher, message);
					result = DBUS_HANDLER_RESULT_HANDLED;
					break;
				}
			}
		}
	}
	
	return result;
}

