/*
 * This file is part of libdicto
 *
 * Copyright (C) 2007-2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * SECTION:dictowidget
 * @short_description: Audio dictation widget
 * @stability: unstable
 * @see_also: Dicto
 *
 * #DictoWidget is an ui audio dictation interface. It presents and list of audio notes and Record, Play, Stop and Delete functionality.
 * An seekbar and current position/length of current clip is also displayed.
 *
 * The widget uses an simple #Dicto object to do the actual recording, playback and note housekeeping.
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gst/gst.h>

#ifdef WITH_HILDON
#include <hildon/hildon.h>
#if HILDON_CHECK_VERSION(2,2,0)
#define WITH_HILDON_2_2
#endif
#endif

#include "dicto.h"
#include "dicto-ui.h"

#include "dicto-ui-marshal.h"

#ifndef HILDON_MARGIN_DEFAULT
#define HILDON_MARGIN_DEFAULT 4
#endif

G_DEFINE_TYPE(DictoWidget, dicto_widget, GTK_TYPE_HBOX);

#define DICTO_WIDGET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DICTO_WIDGET_TYPE, DictoWidgetPriv))

typedef struct _DictoWidgetPriv DictoWidgetPriv; 
struct _DictoWidgetPriv
{
	/* Our dicto object to use */
	Dicto *dicto;

	/* Containers */
	GtkWidget *vbox;
	GtkWidget *hbox;

	/* Labels */
	GtkWidget *lbl_time;
	GtkWidget *lbl_len;
	GtkWidget *position;

	/* Note file list */
	GtkWidget *scroll;
	GtkWidget *file_tree;
	GtkListStore *file_list;
	guint file_cnt;

	/* */
	time_t time;

	/* Buttons */
	GtkWidget *btn_record;
	GtkWidget *btn_play;
	GtkWidget *btn_stop;
	GtkWidget *btn_delete;

	GtkWidget *level;
	gfloat ndb;

	gboolean allow_delete;
	gboolean seeking;
};

/* Signal IDs */
enum {
	SIGNAL_DELETE_QUERY,
	LAST_SIGNAL,
};

/* Property IDs */
enum {
	PROP_0,
	PROP_DICTO_OBJECT,
	PROP_ENABLE_DELETE,
	PROP_TIMESTAMP,
	PROP_LAST,
};

enum {
	NOTE_ID=0,
	NOTE_NAME,
	NOTE_MAX
};

static guint signals[LAST_SIGNAL]={ 0 };

static void dicto_widget_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void dicto_widget_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec   *pspec);

static gboolean dicto_widget_record_cb(GtkWidget *widget, gpointer data);
static gboolean dicto_widget_play_cb(GtkWidget *widget, gpointer data);
static gboolean dicto_widget_stop_cb(GtkWidget *widget, gpointer data);
static gboolean dicto_widget_delete_cb(GtkWidget *widget, gpointer data);
static gboolean dicto_widget_ready_cb(GtkWidget *widget, gpointer data);
static gboolean dicto_widget_level_cb(GtkWidget *widget, gdouble r, gdouble d, gdouble p, gpointer data);
static gboolean dicto_widget_error_cb(GtkWidget *widget, gpointer error, gpointer data);

static void dicto_widget_tree_activate_note_cb(GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data);
#ifdef WITH_HILDON
static void dicto_widget_tree_tap_note_cb(GtkTreeView *tree_view, GtkTreePath *path, gpointer data);
#endif
static void dicto_widget_tree_select_note_cb(GtkTreeSelection *ts, gpointer data);

static gboolean dicto_widget_position_cb(Dicto *d, gdouble left, gpointer data);

static void dicto_widget_position_seek_cb(GtkRange *range, gpointer data);
static gboolean dicto_widget_position_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data);
static gboolean dicto_widget_position_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data);

static void dicto_widget_set_length(DictoWidget *dw, gdouble cur_secs, gdouble tot_secs);
static void dicto_widget_update_view_mode(DictoWidgetPriv *priv);

/******************************************************************************/

static void
dicto_widget_finalize(GObject *object)
{
DictoWidget *dw;
DictoWidgetPriv *priv;

dw=DICTO_WIDGET(object);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

if (priv->dicto)
	g_object_unref(priv->dicto);
G_OBJECT_CLASS(dicto_widget_parent_class)->finalize(object);
}

static void
dicto_widget_class_init(DictoWidgetClass *class)
{
GObjectClass *object_class;
GtkWidgetClass *widget_class;
GParamSpec *pspec;

object_class=(GObjectClass*) class;
widget_class=(GtkWidgetClass*) class;

g_type_class_add_private(object_class, sizeof (DictoWidgetPriv));

object_class->set_property=dicto_widget_set_property;
object_class->get_property=dicto_widget_get_property;
object_class->finalize=dicto_widget_finalize;

signals[SIGNAL_DELETE_QUERY]=g_signal_new("delete-query", G_OBJECT_CLASS_TYPE(object_class),
        G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(DictoWidgetClass, delete_query),
        NULL, NULL, _dicto_BOOLEAN__STRING, G_TYPE_BOOLEAN, 1, G_TYPE_STRING);

pspec=g_param_spec_object("dicto","Dicto object","Dicto object to use for recording and playback", DICTO_TYPE, G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
g_object_class_install_property(object_class, PROP_DICTO_OBJECT, pspec);

pspec=g_param_spec_boolean("enable-delete","Note delete", "Enable deletion button", TRUE, G_PARAM_WRITABLE);
g_object_class_install_property(object_class, PROP_ENABLE_DELETE, pspec);

pspec=g_param_spec_uint("time-stamp","Time stamp", "UNIX time to use for filename generation, 0 for automatic", 0, G_MAXUINT32,0, G_PARAM_READWRITE);
g_object_class_install_property(object_class, PROP_TIMESTAMP, pspec);
}

static GtkWidget *
get_scrolled_container(void)
{
GtkWidget *s;

#ifndef WITH_HILDON_2_2
s=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(s), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
s=hildon_pannable_area_new();
#endif

return s;
}

static GtkWidget *
get_tree_view(void)
{
GtkWidget *t;

#ifndef WITH_HILDON_2_2
t=gtk_tree_view_new();
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(t), TRUE);
#else
t=hildon_gtk_tree_view_new(HILDON_UI_MODE_EDIT);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(t), FALSE);
#endif

return t;
}

static void
dicto_widget_init(DictoWidget *dw)
{
GtkWidget *v, *h;
DictoWidgetPriv *priv=DICTO_WIDGET_GET_PRIVATE(dw);
GtkTreeViewColumn *column;
GtkCellRenderer *renderer;
GtkTreeSelection *selection;

priv->time=0;
priv->ndb=0;
priv->vbox=gtk_vbox_new(FALSE, 2);
v=gtk_vbox_new(FALSE, 2);
h=gtk_hbox_new(FALSE, 2);
priv->hbox=gtk_hbox_new(FALSE, 2);

priv->lbl_time=gtk_label_new("");
priv->lbl_len=gtk_label_new("");

priv->btn_record=gtk_button_new();
gtk_button_set_image(GTK_BUTTON(priv->btn_record), gtk_image_new_from_stock(GTK_STOCK_MEDIA_RECORD, GTK_ICON_SIZE_BUTTON));

priv->btn_play=gtk_button_new();
gtk_button_set_image(GTK_BUTTON(priv->btn_play), gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON));

priv->btn_stop=gtk_button_new();
gtk_button_set_image(GTK_BUTTON(priv->btn_stop), gtk_image_new_from_stock(GTK_STOCK_MEDIA_STOP, GTK_ICON_SIZE_BUTTON));

priv->btn_delete=gtk_button_new();
gtk_button_set_image(GTK_BUTTON(priv->btn_delete), gtk_image_new_from_stock(GTK_STOCK_DELETE, GTK_ICON_SIZE_BUTTON));
priv->allow_delete=FALSE;

#if GTK_CHECK_VERSION(2,10,0)
gtk_button_set_image_position(GTK_BUTTON(priv->btn_record), GTK_POS_TOP);
gtk_button_set_image_position(GTK_BUTTON(priv->btn_play), GTK_POS_TOP);
gtk_button_set_image_position(GTK_BUTTON(priv->btn_stop), GTK_POS_TOP);
gtk_button_set_image_position(GTK_BUTTON(priv->btn_delete), GTK_POS_TOP);
#endif

priv->seeking=FALSE;
#ifdef WITH_HILDON_2_2
priv->position=hildon_gtk_hscale_new();
#else
priv->position=gtk_hscale_new_with_range(0, 1, 1);
gtk_scale_set_draw_value(GTK_SCALE(priv->position), FALSE);
#endif

priv->scroll=get_scrolled_container();

priv->file_tree=get_tree_view();
priv->file_list=gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);

gtk_tree_view_set_model(GTK_TREE_VIEW(priv->file_tree), GTK_TREE_MODEL(priv->file_list));

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes("Note", renderer, "text", NOTE_NAME, NULL);
gtk_tree_view_column_set_sort_column_id(column, NOTE_NAME);
gtk_tree_view_column_set_sort_indicator(column, TRUE);
gtk_tree_view_append_column(GTK_TREE_VIEW(priv->file_tree), column);

gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(priv->file_tree), TRUE);
gtk_tree_view_set_search_column(GTK_TREE_VIEW(priv->file_tree), 1);
gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(priv->file_list), NOTE_NAME, GTK_SORT_ASCENDING);

selection=gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->file_tree));
gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

priv->level=gtk_progress_bar_new();
gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(priv->level), GTK_PROGRESS_BOTTOM_TO_TOP);

gtk_box_pack_start(GTK_BOX(priv->vbox), priv->hbox, TRUE, TRUE, HILDON_MARGIN_DEFAULT);

gtk_box_pack_start(GTK_BOX(h), priv->lbl_time, FALSE, FALSE, HILDON_MARGIN_DEFAULT);
gtk_box_pack_start(GTK_BOX(h), priv->position, TRUE, TRUE, HILDON_MARGIN_DEFAULT);
gtk_box_pack_start(GTK_BOX(h), priv->lbl_len, FALSE, FALSE, HILDON_MARGIN_DEFAULT);

gtk_box_pack_start(GTK_BOX(priv->vbox), h, FALSE, FALSE, HILDON_MARGIN_DEFAULT);

gtk_box_pack_start(GTK_BOX(v), priv->btn_record, FALSE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(v), priv->btn_play, FALSE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(v), priv->btn_stop, FALSE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(v), priv->btn_delete, FALSE, TRUE, 0);

gtk_container_add(GTK_CONTAINER(priv->scroll), priv->file_tree);
gtk_box_pack_start(GTK_BOX(priv->hbox), priv->scroll, TRUE, TRUE, HILDON_MARGIN_DEFAULT);
gtk_box_pack_start(GTK_BOX(priv->hbox), priv->level, FALSE, FALSE, HILDON_MARGIN_DEFAULT);
gtk_box_pack_start(GTK_BOX(priv->hbox), v, FALSE, FALSE, HILDON_MARGIN_DEFAULT);

gtk_widget_show(priv->vbox);
gtk_box_pack_start(GTK_BOX(dw), priv->vbox, TRUE, TRUE, 0);

gtk_widget_set_sensitive(priv->btn_play, FALSE);
gtk_widget_set_sensitive(priv->btn_record, FALSE);
gtk_widget_set_sensitive(priv->btn_stop, FALSE);
gtk_widget_set_sensitive(priv->btn_delete, FALSE);

g_signal_connect(G_OBJECT(priv->btn_record), "clicked", G_CALLBACK(dicto_widget_record_cb), dw);
g_signal_connect(G_OBJECT(priv->btn_play), "clicked", G_CALLBACK(dicto_widget_play_cb), dw);
g_signal_connect(G_OBJECT(priv->btn_stop), "clicked", G_CALLBACK(dicto_widget_stop_cb), dw);
g_signal_connect(G_OBJECT(priv->btn_delete), "clicked", G_CALLBACK(dicto_widget_delete_cb), dw);

#ifdef WITH_HILDON
g_signal_connect(G_OBJECT(priv->file_tree), "hildon-row-tapped", G_CALLBACK(dicto_widget_tree_tap_note_cb), dw);
#endif
g_signal_connect(G_OBJECT(priv->file_tree), "row-activated", G_CALLBACK(dicto_widget_tree_activate_note_cb), dw);

g_signal_connect(G_OBJECT(priv->position), "value-changed", G_CALLBACK(dicto_widget_position_seek_cb), dw);
g_signal_connect(G_OBJECT(priv->position), "button-press-event", G_CALLBACK(dicto_widget_position_press_cb), dw);
g_signal_connect(G_OBJECT(priv->position), "button-release-event", G_CALLBACK(dicto_widget_position_release_cb), dw);

g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(dicto_widget_tree_select_note_cb), dw);
}

static void
dicto_widget_position_seek_cb(GtkRange *range, gpointer data)  
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
DictoState s;

g_return_if_fail(IS_DICTO_WIDGET(dw));
priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_if_fail(priv);
g_return_if_fail(priv->dicto);

s=dicto_get_state(priv->dicto);
if ((s==DICTO_STATE_PLAYING || s==DICTO_STATE_READY) && priv->seeking) {
	if (dicto_seek(priv->dicto, gtk_range_get_value(range)))
		dicto_widget_position_cb(priv->dicto, 0, dw);
}
}

static gboolean
dicto_widget_position_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(IS_DICTO_WIDGET(dw), FALSE);
priv=DICTO_WIDGET_GET_PRIVATE(dw);
priv->seeking=TRUE;

return FALSE;
}

static gboolean
dicto_widget_position_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(IS_DICTO_WIDGET(dw), FALSE);
priv=DICTO_WIDGET_GET_PRIVATE(dw);
priv->seeking=FALSE;

return FALSE;
}

static void
dicto_widget_tree_select_note_cb(GtkTreeSelection *ts, gpointer data)
{
GtkTreeIter iter;
GtkTreeModel *model;
gchar *file;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
DictoState s;

g_return_if_fail(IS_DICTO_WIDGET(dw));
priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_if_fail(priv);
g_return_if_fail(priv->dicto);

s=dicto_get_state(priv->dicto);
if (s==DICTO_STATE_PLAYING || s==DICTO_STATE_RECORDING)
	return;

if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(ts), &model, &iter))
	return;

gtk_tree_model_get(model, &iter, NOTE_ID, &file,  -1);
if (!file)
	return;

g_debug("Selected: %s", file);

if (!dicto_prepare(priv->dicto, file))
	dicto_widget_update_view_mode(priv);

g_free(file);
}

#ifdef WITH_HILDON
static void
dicto_widget_tree_tap_note_cb(GtkTreeView *tree_view, GtkTreePath *path, gpointer data)
{
GtkTreeIter iter;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
DictoState s;
gchar *file;

g_return_if_fail(IS_DICTO_WIDGET(dw));
priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_if_fail(priv);
g_return_if_fail(priv->dicto);

s=dicto_get_state(priv->dicto);
if (s==DICTO_STATE_PLAYING || s==DICTO_STATE_RECORDING)
	return;

g_return_if_fail(gtk_tree_model_get_iter(GTK_TREE_MODEL(priv->file_list), &iter, path));
gtk_tree_model_get(GTK_TREE_MODEL(priv->file_list), &iter, NOTE_ID, &file, -1);
if (!file)
	return;
g_debug("File %s tapped", file);

if (!dicto_prepare(priv->dicto, file))
	dicto_widget_update_view_mode(priv);

g_free(file);
}
#endif

static void
dicto_widget_tree_activate_note_cb(GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data)
{
GtkTreeIter iter;
gchar *file;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
GtkTreeModel *model;

g_return_if_fail(IS_DICTO_WIDGET(dw));

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_if_fail(priv);

#ifdef WITH_HILDON
if (!gtk_tree_model_get_iter(GTK_TREE_MODEL(priv->file_list), &iter, path))
	return;
#else
if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view)), NULL, &iter))
    return;
#endif

model=gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
if (!model)
    return;
gtk_tree_model_get(model, &iter, NOTE_ID, &file, -1);

g_debug("File %s activated", file);

if (!dicto_play(priv->dicto, file))
	dicto_widget_update_view_mode(priv);

g_free(file);
}

static gboolean
dicto_widget_position_cb(Dicto *d, gdouble left, gpointer data)
{
gdouble len=0, pos=0;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(IS_DICTO(d), TRUE);
g_return_val_if_fail(IS_DICTO_WIDGET(dw), TRUE);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

if (dicto_get_position(d, &len, &pos)) {
	g_debug("%f/%f", pos, len);
	dicto_widget_set_length(dw, pos, len);
	if (len>0 || pos>0)
		gtk_range_set_range(GTK_RANGE(priv->position), 0.0, len>=pos ? len : pos);
	if (dicto_get_state(priv->dicto)==DICTO_STATE_PLAYING && !priv->seeking)
		gtk_range_set_value(GTK_RANGE(priv->position), pos);
}

return FALSE;
}

static gboolean
dicto_widget_playing_cb(Dicto *d, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(d, TRUE);
g_return_val_if_fail(data, TRUE);

priv=DICTO_WIDGET_GET_PRIVATE(dw);
dicto_widget_update_view_mode(priv);
return TRUE;
}

static gboolean
dicto_widget_recording_cb(Dicto *d, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(d, TRUE);
g_return_val_if_fail(data, TRUE);

priv=DICTO_WIDGET_GET_PRIVATE(dw);
dicto_widget_update_view_mode(priv);
return TRUE;
}

static gboolean
dicto_widget_stopped_record_cb(Dicto *d, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(d, TRUE);
g_return_val_if_fail(data, TRUE);

priv=DICTO_WIDGET_GET_PRIVATE(dw);

dicto_widget_update_view_mode(priv);
return TRUE;
}

static gboolean
dicto_widget_stopped_play_cb(Dicto *d, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(d, TRUE);
g_return_val_if_fail(data, TRUE);

priv=DICTO_WIDGET_GET_PRIVATE(dw);

dicto_widget_update_view_mode(priv);
return TRUE;
}

static void
dicto_widget_append_file(gpointer key, gpointer value, gpointer data)
{
DictoWidgetPriv *priv=(DictoWidgetPriv *)data;
GtkListStore *store=priv->file_list;
GtkTreeIter iter;

g_debug("%s => %s", (gchar *)key, (gchar *)value);

gtk_list_store_append(store, &iter);
gtk_list_store_set(store, &iter, NOTE_ID, key, NOTE_NAME, value, -1);
priv->file_cnt++;
}

static gboolean
dicto_widget_updatelist_cb(Dicto *d, gpointer data)
{
GHashTable *notes;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

g_return_val_if_fail(d, TRUE);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

gtk_list_store_clear(priv->file_list);
notes=dicto_notes_get_list(d);

priv->file_cnt=0;
g_hash_table_foreach(notes, dicto_widget_append_file, priv);

return TRUE;
}

static void
dicto_widget_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
DictoWidget *dw;
DictoWidgetPriv *priv;
Dicto *d;

g_return_if_fail(IS_DICTO_WIDGET(object));
dw=DICTO_WIDGET(object);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

switch (prop_id) {
	case PROP_DICTO_OBJECT:
		g_debug("Set dicto object");
		d=g_value_get_object(value);
		g_return_if_fail(IS_DICTO(d));

		if (priv->dicto) {
			/* XXX: disconnect signals */
			dicto_stop(priv->dicto);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_position_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_playing_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_recording_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_stopped_play_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_stopped_record_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_updatelist_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_ready_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_level_cb), dw);
			g_signal_handlers_disconnect_by_func(G_OBJECT(priv->dicto), G_CALLBACK(dicto_widget_error_cb), dw);
			g_object_unref(priv->dicto);
		}
		g_object_ref(d);
		priv->dicto=d;
		dicto_widget_updatelist_cb(priv->dicto, dw);
		dicto_widget_update_view_mode(priv);
		g_signal_connect(G_OBJECT(priv->dicto), "position", G_CALLBACK(dicto_widget_position_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "playing", G_CALLBACK(dicto_widget_playing_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "recording", G_CALLBACK(dicto_widget_recording_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "stopped-play", G_CALLBACK(dicto_widget_stopped_play_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "stopped-record", G_CALLBACK(dicto_widget_stopped_record_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "refresh", G_CALLBACK(dicto_widget_updatelist_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "ready", G_CALLBACK(dicto_widget_ready_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "level", G_CALLBACK(dicto_widget_level_cb), dw);
		g_signal_connect(G_OBJECT(priv->dicto), "error", G_CALLBACK(dicto_widget_error_cb), dw);
	break;
	case PROP_ENABLE_DELETE:
		priv->allow_delete=g_value_get_boolean(value);
	break;
	case PROP_TIMESTAMP:
		priv->time=(time_t)g_value_get_uint(value);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
}
}

static void
dicto_widget_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec   *pspec)
{
DictoWidget *dw;
DictoWidgetPriv *priv;

g_return_if_fail(IS_DICTO_WIDGET(object));
dw=DICTO_WIDGET(object);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

switch (prop_id) {
	case PROP_DICTO_OBJECT:
		g_value_set_object(value, priv->dicto);
	break;
	case PROP_TIMESTAMP:
		g_value_set_uint(value, (guint)priv->time);
	break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
}
}

/******************************************************************************/

static void
dicto_widget_position_clear(DictoWidgetPriv *priv)
{
gtk_label_set_markup(GTK_LABEL(priv->lbl_time), "<span size='x-large'>--:--</span>");
gtk_label_set_markup(GTK_LABEL(priv->lbl_len), "<span size='x-large'>--:--</span>");
}

static void
dicto_widget_position_set(DictoWidgetPriv *priv, guint min, gdouble secs)
{
gchar buffer[48];

g_snprintf(buffer, sizeof(buffer), "<span size='x-large'>%u:%05.2f</span>", min, secs);
gtk_label_set_markup(GTK_LABEL(priv->lbl_time), buffer);
}

static void
dicto_widget_length_set(DictoWidgetPriv *priv, guint min, gdouble secs)
{
gchar buffer[48];

g_snprintf(buffer, sizeof(buffer), "<span size='x-large'>%u:%05.2f</span>", min, secs);
gtk_label_set_markup(GTK_LABEL(priv->lbl_len), buffer);
}

static void
dicto_widget_update_view_mode(DictoWidgetPriv *priv)
{
DictoState mode;

g_return_if_fail(priv);
g_return_if_fail(priv->dicto);

mode=dicto_get_state(priv->dicto);
g_debug("VMode: %d", mode);
switch (mode) {
	case DICTO_STATE_ERROR:
		/* XXX: What should we do ?*/
	case DICTO_STATE_STOPPED:
		/* XXX: Add check if we have something to play */
		dicto_widget_position_clear(priv);
		gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(priv->level), 0.0);
		gtk_widget_set_sensitive(priv->btn_play, FALSE);
		gtk_widget_set_sensitive(priv->btn_record, TRUE);
		gtk_widget_set_sensitive(priv->btn_stop, FALSE);
		gtk_widget_set_sensitive(priv->btn_delete, FALSE);

		gtk_widget_set_sensitive(priv->file_tree, TRUE);
		gtk_widget_set_sensitive(priv->position, FALSE);
		gtk_range_set_value(GTK_RANGE(priv->position), 0.0);
#ifndef WITH_HILDON_2_2
		gtk_range_set_update_policy(GTK_RANGE(priv->position), GTK_UPDATE_CONTINUOUS);
#endif
	break;
	case DICTO_STATE_READY:
		dicto_widget_position_set(priv, 0, 0);
		gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(priv->level), 0.0);
		gtk_widget_set_sensitive(priv->btn_record, TRUE);
		gtk_widget_set_sensitive(priv->btn_play, TRUE);
		gtk_widget_set_sensitive(priv->btn_stop, FALSE);
		gtk_widget_set_sensitive(priv->btn_delete, priv->allow_delete);

		gtk_widget_set_sensitive(priv->position, TRUE);
		gtk_widget_set_sensitive(priv->file_tree, TRUE);
		gtk_range_set_value(GTK_RANGE(priv->position), 0.0);
		gtk_range_set_update_policy(GTK_RANGE(priv->position), GTK_UPDATE_CONTINUOUS);
	break;
	case DICTO_STATE_RECORDING:
		dicto_widget_position_clear(priv);
		gtk_range_set_range(GTK_RANGE(priv->position), 0.0, 0.1);		
		dicto_widget_position_set(priv, 0, 0);
	case DICTO_STATE_PLAYING:
#ifndef WITH_HILDON_2_2
		gtk_range_set_update_policy(GTK_RANGE(priv->position), GTK_UPDATE_DELAYED);
#endif
		gtk_widget_set_sensitive(priv->btn_play, FALSE);
		gtk_widget_set_sensitive(priv->btn_record, FALSE);
		gtk_widget_set_sensitive(priv->btn_stop, TRUE);
		gtk_widget_set_sensitive(priv->btn_delete, FALSE);

		gtk_widget_set_sensitive(priv->file_tree, FALSE);
		gtk_widget_set_sensitive(priv->position, mode==DICTO_STATE_PLAYING);
	break;
}
}

static gchar *
dicto_widget_get_selected_file(GtkTreeView *list)
{
gchar *file;
GtkTreeIter iter;
GtkTreeModel *model;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(list)), NULL, &iter))
        return NULL;

model=gtk_tree_view_get_model(GTK_TREE_VIEW(list));
if (!model)
        return NULL;

gtk_tree_model_get(model, &iter, NOTE_ID, &file, -1);
return file;
}

static void
dicto_widget_set_length(DictoWidget *dw, gdouble cur_secs, gdouble tot_secs)
{
DictoWidgetPriv *priv;
guint min=0;

g_return_if_fail(dw);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

if (cur_secs<0.0) {
	dicto_widget_position_clear(priv);
	return;
}

if (cur_secs>=60.0) {
	min=cur_secs/60.0;
	cur_secs-=min*60.0;
}

dicto_widget_position_set(priv, min, cur_secs);

if (tot_secs<0)
	return;
min=0;
if (tot_secs>=60.0) {
	min=tot_secs/60.0;
	tot_secs-=min*60.0;
}

dicto_widget_length_set(priv, min, tot_secs);
}

static gboolean
dicto_widget_error_cb(GtkWidget *widget, gpointer error, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_val_if_fail(priv, TRUE);

g_debug("Got error from dicto");
return FALSE;
}

static gboolean
dicto_widget_level_cb(GtkWidget *widget, gdouble r, gdouble d, gdouble p, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
gfloat tmp;

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_val_if_fail(priv, TRUE);

tmp=dicto_normalize_dB(d);
if (fabsf(priv->ndb-tmp)>0.1 || tmp<0.15) {
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(priv->level), dicto_normalize_dB(d));
	priv->ndb=tmp;
}

return FALSE;
}

static gboolean
dicto_widget_record_cb(GtkWidget *widget, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
gchar *f;

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_val_if_fail(priv, TRUE);

f=dicto_filename_time_new(priv->dicto, priv->time);
g_return_val_if_fail(f, TRUE);
dicto_record(priv->dicto, f);
g_free(f);
return TRUE;
}

static gboolean
dicto_widget_ready_cb(GtkWidget *widget, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_val_if_fail(priv, TRUE);

g_debug("Ready to play");
dicto_widget_update_view_mode(priv);

return TRUE;
}

static gboolean
dicto_widget_play_cb(GtkWidget *widget, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv;
gchar *f;

priv=DICTO_WIDGET_GET_PRIVATE(dw);
g_return_val_if_fail(priv, TRUE);

f=dicto_widget_get_selected_file(GTK_TREE_VIEW(priv->file_tree));
if (!f)
	return TRUE;

if (!dicto_play(priv->dicto, f))
	dicto_widget_update_view_mode(priv);

g_free(f);
return TRUE;
}

static gboolean
dicto_widget_stop_cb(GtkWidget *widget, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv=DICTO_WIDGET_GET_PRIVATE(dw);

g_return_val_if_fail(dw, TRUE);

dicto_stop(priv->dicto);

return TRUE;
}

static gboolean
dicto_widget_delete_cb(GtkWidget *widget, gpointer data)
{
gchar *file;
DictoWidget *dw=(DictoWidget *)data;
DictoWidgetPriv *priv=DICTO_WIDGET_GET_PRIVATE(dw);
gboolean ret=FALSE;

g_return_val_if_fail(dw, TRUE);

file=dicto_widget_get_selected_file(GTK_TREE_VIEW(priv->file_tree));
if (!file) {
	g_warning("Failed to get selection from tree view");
	return TRUE;
}

g_signal_emit(dw, signals[SIGNAL_DELETE_QUERY], 0, file, &ret);
if (ret) {
	dicto_delete(priv->dicto, file);
	dicto_widget_update_view_mode(priv);
}

return TRUE;
}

/**
 * dicto_widget_delete_query_dialog:
 * @w: Main window
 * @file:
 *
 * An example handler for delete query.
 */
#ifdef WITH_HILDON
gboolean
dicto_widget_delete_query_dialog(GtkWindow *w, const gchar *file)
{
GtkWidget *nh;
gint r;

nh=hildon_note_new_confirmation(w, "Delete ?");
r=gtk_dialog_run(GTK_DIALOG(nh));
gtk_widget_destroy(nh);

return (r==GTK_RESPONSE_OK);
}

#else
gboolean
dicto_widget_delete_query_dialog(GtkWindow *w, const gchar *file)
{
GtkWidget *dialog;
gint r;

dialog=gtk_message_dialog_new(w, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, "Delete note '%s' ?", file);
r=gtk_dialog_run(GTK_DIALOG(dialog));
gtk_widget_destroy(dialog);

switch (r) {
	case GTK_RESPONSE_YES:
	case GTK_RESPONSE_OK:
		return TRUE;
}
return FALSE;
}

#endif

/**
 * dicto_widget_get_formats:
 *
 * Returns: A #GtkListStore with 3 columns, 1=ID, 2=Description, 3=Pointer to the #DictoFormat struct
 */
GtkListStore *
dicto_widget_get_formats(DictoWidget *dw)
{
DictoFormats i=0;
GtkListStore *ls;
GtkTreeIter iter;

ls=gtk_list_store_new(3, G_TYPE_INT, G_TYPE_STRING, G_TYPE_POINTER);
while (i<FORMAT_LAST) {
	DictoFormat *f=dicto_get_format(dicto_widget_get_dicto(dw), i);

	if (!f) {
		i++;
		continue;
	}
		
	gtk_list_store_append(ls, &iter);
	gtk_list_store_set(ls, &iter, 0, f->id, 1, f->desc, 2, f, -1);
	i++;
}

return ls;
}

/**
 * dicto_widget_get_dicto:
 * @dw:
 *
 * Get the #Dicto object used. The returned value is owned by the widget so don't free it.
 */
Dicto *
dicto_widget_get_dicto(DictoWidget *dw)
{
DictoWidgetPriv *priv;

g_return_val_if_fail(dw, FALSE);
priv=DICTO_WIDGET_GET_PRIVATE(dw);

return priv->dicto;
}

/**
 * dicto_widget_new:
 * @d: A #Dicto object
 *
 * Create a new #DictoWidget using given #Dicto object. Free with g_object_unref.
 */
DictoWidget *
dicto_widget_new(Dicto *d)
{
g_return_val_if_fail(d, NULL);
return g_object_new(DICTO_WIDGET_TYPE, "dicto", d, NULL);
}

