#include "item.h"
#include <QStringList>
#include <QMessageBox>
#include <QUuid>

PropertiesItem::PropertiesItem(QObject *parent) :
    QObject(parent)
{

}

PropertiesItem::PropertiesItem(const PropertiesItem& item) :
    QObject(item.parent())
{
    QHash<QString, QString>::const_iterator i;
    for (i = item.Properties.constBegin(); i != item.Properties.constEnd(); ++i){
         Properties[i.key()] = i.value();
    }
}

int PropertiesItem::GetIntProperty(QString name)
{
    QString p = GetProperty(name);
    if (!p.isEmpty())
        return p.toInt();
    return 0;
}

QString PropertiesItem::GetProperty(QString name)
{
    if (Properties.contains(name))
        return Properties[name];
    return "";
}

void PropertiesItem::SetProperty(QString name, QString value)
{
    Properties[name] = value;
}



Item::Item(QObject *parent) :
    PropertiesItem(parent)
{
    SetUUID(QUuid::createUuid().toString());
    Name = "";
    Description = "";
    SetQuantity(1);
    SetSize(1);
    SetUsability(1);
}

Item::Item(const Item& item) :
    PropertiesItem(item)
{
    Name = item.Name;
    Description = item.Description;
}

QString Item::GetDescription()
{
    if (!Description.isEmpty())
        return Description;

    if (GetIsQuiver() && GetArrows() > 0)
        return QString("Contains %1 arrows").arg(GetArrows());

    if (GetEnduranceRestore() > 0)
#ifdef Q_WS_MAEMO_5
        return QString("Restores %1 END. points").arg(GetEnduranceRestore());
#else
        return QString("Restores %1 ENDURANCE points").arg(GetEnduranceRestore());
#endif

    if (GetEnduranceDelta() > 0)
#ifdef Q_WS_MAEMO_5
        return QString("Adds %1 END. points to your total").arg(GetEnduranceDelta());
#else
        return QString("Adds %1 ENDURANCE points to your total").arg(GetEnduranceDelta());
#endif

    if (GetCombatskillDelta() > 0){
        if (GetCanBeEquipped())
            return QString("Adds %1 points to your COMBAT SKILL when used in combat").arg(GetCombatskillDelta());
        else
            return QString("Increase your COMBAT SKILL by %1 points for the duration of a fight").arg(GetCombatskillDelta());
    }

    return "";
}

void Item::Use(QScriptEngine* engine, QScriptValue* functions)
{
    QScriptValue objectValue = engine->newQObject(this);
    engine->globalObject().setProperty("Item", objectValue);

    //Specific function:
    QScriptValue f = functions->property(QString("use_%1").arg(Name.replace("'", "").replace(" ", "")));
    if (f.isValid()){
        f.call();
        return;
    }

    //Generic functions (healing items, ...):
    if (GetEnduranceRestore() || GetCombatskillDelta()){
        QString name = GetEnduranceRestore() ? "EnduranceRestore" : "CombatskillDelta";
        f = functions->property(QString("use_%1").arg(name));
        if (f.isValid()){
            f.call();
        }else{
            QMessageBox msgBox(NULL);
            msgBox.setWindowTitle(QString("Warning").arg(name));

            msgBox.setIcon(QMessageBox::Warning);
            msgBox.setText(QString("No function found to use item: %1").arg(Name));
            msgBox.exec();
        }
    }
    SetUsability(GetUsability() - 1);
}

bool Item::GetCanBeEquipped()
{
    if (!HasProperty("CanBeEquipped"))
        return true;
    return GetBoolProperty("CanBeEquipped");
}

void Item::SetCanBeEquipped(bool value)
{
    if (!value && GetIsEquipped())
        Unequip();
    SetBoolProperty("CanBeEquipped", value);
}

void Item::SetArmorClass(QString value)
{
    int aClass = Armor::GetClass(value);
    SetProperty("ArmorClass", QString::number(aClass));
}

QList<int> Item::GetWeaponClasses()
{
    QList<int> res;
    QStringList str = GetProperty("Classes").split(";", QString::SkipEmptyParts);
    foreach (QString s, str){
        res.append(s.toInt());
    }
    return res;
}

void Item::AddWeaponClass(int iClass)
{
    if (iClass != Weapon::None){
        QString str = GetProperty("Classes");
        str.append(QString(";%1").arg(iClass));
        SetProperty("Classes", str);
    }
}

void Item::AddWeaponClass(QString value)
{
    AddWeaponClass(Weapon::GetClass(value));
}

bool Item::HasWeaponClass(QString value)
{
    return GetWeaponClasses().contains(Weapon::GetClass(value));
}

bool Item::HasWeaponClass(int value)
{
    return GetWeaponClasses().contains(value);
}

int Item::GetTotalSize()
{
    if (GetQuantity())
        return GetSize() * GetQuantity();
    return GetSize();
}

bool Item::GetCanBeDropped()
{
    if (HasProperty("sys_CanBeDropped"))
        return GetBoolProperty("sys_CanBeDropped");
    return true;
}


Weapon::Weapon(QObject *parent) :
    Item(parent)
{
    SetIsWeapon(true);
    SetCanBeEquipped(true);
    SetHands(1);
}

Weapon::Weapon(const Weapon& weapon) :
        Item(weapon)
{
}

int Weapon::GetClass(QString wclass)
{
    if (!QString::compare(wclass, "Dagger", Qt::CaseInsensitive))
        return Weapon::Dagger;
    else if (!QString::compare(wclass, "Spear", Qt::CaseInsensitive))
        return Weapon::Spear;
    else if (!QString::compare(wclass, "Mace", Qt::CaseInsensitive))
        return Weapon::Mace;
    else if (!QString::compare(wclass, "ShortSword", Qt::CaseInsensitive) || !QString::compare(wclass, "Short Sword", Qt::CaseInsensitive))
        return Weapon::ShortSword;
    else if (!QString::compare(wclass, "Warhammer", Qt::CaseInsensitive))
        return Weapon::WarHammer;
    else if (!QString::compare(wclass, "Axe", Qt::CaseInsensitive))
        return Weapon::Axe;
    else if (!QString::compare(wclass, "Sword", Qt::CaseInsensitive))
        return Weapon::Sword;
    else if (!QString::compare(wclass, "Quarterstaff", Qt::CaseInsensitive))
        return Weapon::Quarterstaff;
    else if (!QString::compare(wclass, "Broadsword", Qt::CaseInsensitive))
        return Weapon::Broadsword;
    else if (!QString::compare(wclass, "Bow", Qt::CaseInsensitive))
        return Weapon::Bow;
    return Weapon::None;
}

int Weapon::GetHands()
{
    QList<int> classes = Classes();
    if (classes.contains(Spear) ||
        (classes.contains(Broadsword) && !classes.contains(Sword)) ||
        classes.contains(Quarterstaff))
        return 2;
    return 1;
}

void Weapon::AddClass(int iClass)
{
    AddWeaponClass(iClass);
}

void Weapon::AddClass(QString value)
{
    AddWeaponClass(value);
}

bool Weapon::HasClass(QString value)
{
    return Classes().contains(GetClass(value));
}

bool Weapon::HasClass(int value)
{
    return Classes().contains(value);
}

QList<int> Weapon::Classes()
{
    return GetWeaponClasses();
}

QString Weapon::ClassDescription(int Class)
{
    switch (Class)
    {
    case Dagger:
        return "Dagger";
        break;
    case Spear:
        return "Spear";
        break;
    case Mace:
        return "Mace";
        break;
    case ShortSword:
        return "Short Sword";
        break;
    case WarHammer:
        return "Warhammer";
        break;
    case Axe:
        return "Axe";
        break;
    case Sword:
        return "Sword";
        break;
    case Quarterstaff:
        return "Quarterstaff";
        break;
    case Broadsword:
        return "Broadsword";
        break;
    case Bow:
        return "Bow";
        break;
    default:
        return "";
        break;
    }
}

ShopItem::ShopItem(QObject *parent) :
    Weapon(parent)
{
    Price = 0;
    SellPrice = 0;
    UnlimitedQuantity = false;
}

Armor::Armor(QObject *parent) :
    Item(parent)
{
    SetIsArmor(true);
    SetCanBeEquipped(true);
}

Armor::Armor(const Armor& armor) :
        Item(armor)
{
}

int Armor::GetClass(QString sClass)
{
    if (!QString::compare(sClass, "Head", Qt::CaseInsensitive))
        return Armor::Head;
    else if (!QString::compare(sClass, "Body", Qt::CaseInsensitive))
        return Armor::Body;
    else if (!QString::compare(sClass, "Hand", Qt::CaseInsensitive))
        return Armor::Hand;
    else if (!QString::compare(sClass, "Shield", Qt::CaseInsensitive))
        return Armor::Shield;
    return Armor::None;
}

void Armor::SetClass(int iClass)
{
    SetArmorClass(iClass);
}

int Armor::GetClass()
{
    return GetArmorClass();
}

int Armor::GetHands()
{
    if (GetClass() == Armor::Shield)
        return 1;
    return 0;
}

bool Armor::HasClass(QString value)
{
    return GetArmorClass() == Armor::GetClass(value);
}
