#ifndef BOOK_H
#define BOOK_H

#include <QObject>
#include <QtXml>
#include "item.h"
#include "version.h"

class Enemy : public QObject
{
    Q_OBJECT
public:
    explicit Enemy(QObject *parent = 0);

    QString Name;
    int CombatSkill;
    int Endurance;
    bool CanBeEvaded;

    QStringList DisciplineImmunities;
    QStringList WoundedBy;
    QString EnterScript;
    QString ExitScript;
    QString RoundScript;
    QString DamageScript;

public slots:
    bool IsDead();
    void SetEndurance(int value) { Endurance = value; }
    int GetEndurance() { return Endurance; }
};

//!  Describes a choice in a section
/*!

*/
class Choice : public QObject
{
    Q_OBJECT

public:
    explicit Choice(QObject *parent = 0);

    bool Enabled;
    int Target;
    QString Text;
    bool Random;
    int MinValue;
    int MaxValue;
    bool ShowWithEnemies;

    QString Script;

private:
    bool m_Abort;

public slots:
    /** Returns true if the choice is enabled*/
    bool IsEnabled() { return Enabled;}
    /** Sets if the choice is enabled*/
    void SetEnabled(bool value) { Enabled = value; }
    /** Abort this choice (used in the choice script to avoid section navigation)*/
    void Abort() { m_Abort = true; }
    /** Returns true if the choice has been aborted*/
    bool IsAborted() { return m_Abort; }
    /** Accepts the choice (used in the choice script to accept section navigation)*/
    void Accept() {m_Abort = false; }
};

class Book;
//!  Describes a section of the book
/*!

*/
class Section : public QObject
{
    Q_OBJECT
public:
    explicit Section(QObject *parent = 0);

    Book* GetBook() { return (Book*)parent(); }
    int Number;    
    QString Text;
    QList<Choice*> Choices;
    QList<Enemy*> Enemies;

    QString EnterScript;
    QString ExitScript;
    QString ShowScript;

    bool HasRandomChoices();
    int UserChoicesCount();

    bool AutoSave;
    int MustEat;
    int DidntEatEnduranceLoss;
    bool CanHunt;
    int GoldCrowns;
    QList<Item*> Items;
    QList<ShopItem*> ShopItems;

    bool HasPickableItems();
    Choice* GetChoiceFromTarget(int);

public slots:
    /** Returns the choice with the given index*/
    Choice* GetChoice(int);
    /** Enables the choice with the given index*/
    void EnableChoice(int);
    /** Disables the choice with the given index*/
    void DisableChoice(int);
    /** Returns the number of weapons in the section*/
    int GetWeaponsCount();
    /** Returns the number of enemies in the section*/
    int GetEnemiesCount();
    /** Removes enemies from the section*/
    void RemoveEnemies();
    /** Sets the must eat property (number of meals to eat)*/
    void SetMustEat(int value) { MustEat = value; }
    /** Removes one item from the section, returns true if the item has been remove*/
    bool RemoveItem(QString name);
    /** Adds one item to the section, returns the item added*/
    Item* AddItem(QString name, bool special);
    /** Copies all the items of the section to the given section*/
    void CopyItemsToSection(int section);
};


class Book : public QObject
{
    Q_OBJECT
public:
    enum series{
        Unknown,
        Kai,
        Magnakai,
        GrandMaster,
        NewOrder
    };

    explicit Book(QObject *parent = 0);

    QString FileName;
    int Version;
    int Revision;
    VersionClass MinInterpreterVersion;
    QString FunctionsScript;
    QString BaseDirectory;
    QString Serie;
    QString Title;
    QString Authors;
    int Number;
    QString ScriptedBy;
    QHash<int, Section*> Sections;

    bool Load(QString filename, bool full=true);
    bool Save(QString filename);
    QString GetVersion();
    int GetSerie();
    Section* GetSection(int section);

private:
    QString AddGenericFunctions(QString functions);
    Choice* ParseChoice(Section* section, QXmlStreamReader* reader);
    Enemy* ParseEnemy(Section* section, QXmlStreamReader* reader);
    Weapon* ParseWeapon(Section* section, QXmlStreamReader* reader);
    Item* ParseItem(Section* section, QXmlStreamReader* reader);
    ShopItem* ParseShopItem(Section* section, QXmlStreamReader* reader);
    QString GetAttribute(QXmlStreamAttributes attrs, QString name, QString defaultValue="");
};

#endif // BOOK_H
