#ifndef PLAYER_H
#define PLAYER_H

#include <QObject>
#include "kaidiscipline.h"
#include "book.h"

//!  Describes the player
/*!

*/
class Player : public QObject
{
    Q_OBJECT
public:
    explicit Player(QObject *parent = 0);
    Player(const Player& player);

    int BaseCombatSkill;
    int BaseEndurance;
    int Endurance;

    int GoldCrowns;
    int TemporaryCombatSkillDelta;


    int GetCombatSkill();
    int GetCombatSkill(Enemy*);
    int GetCombatSkill(Enemy*, QStringList*);
    QStringList GetCombatSkillDescription(Enemy*);

    QList<KaiDiscipline*> KaiDisciplines;
    void RemoveDiscipline(QString name);

    QList<Weapon*> Weapons(bool includeSpecial=false);
    QList<Item*> BackpackItems();
    QList<Item*> SpecialItems();
    QList<Item*> Items;
    QList<Item*> HiddenItems;

    int DeathsCount;

    static QString GetRankDescription(int serie, int kaiDisciplinesCount);
    QHash<QString, QString> Properties;
    void CheckEndurance();
    void Dump();
    KaiDiscipline* GetDiscipline(QString name);
    /** Returns the serie of the player (maximum serie of the disciplines)*/
    int GetSerie();

    int GetBaseEndurance(QStringList*);
    QStringList GetBaseEnduranceDescription();

private:
    bool m_Backpack;
    bool m_CanHunt;
    bool GetBoolProperty(QString name) { return GetProperty(name).toInt() == 1;}
    void SetBoolProperty(QString name, bool value) { SetProperty(name, value ? "1" : "0"); }

public slots:
    /** Remove all disciplines (used at the beginning of a new serie when disciplines change)
     *  It doesn't removes the disciplines, but set their "serie" to 0 (zero) so they are not
     *  counted.
    */
    void RemoveDisciplines();

    /** Return true if the player is dead */
    bool IsDead();

    /** Clear all the player's properties */
    void ClearProperties();
    /** Get a player property */
    QString GetProperty(QString name);
    /** Set a player property */
    void SetProperty(QString name, QString value);
    /** Set a integer player property */
    void SetProperty(QString name, int value) { SetProperty(name, QString::number(value)); }
    /** Remove a property */
    void RemoveProperty(QString name);
    /** Returns true if the player has the given property set */
    bool HasProperty(QString name) { return Properties.contains(name); }

    /** Return true if the player has learned the discipline */
    bool HasDiscipline(QString name);
    /** Return true if the player has complete the lore circle */
    bool HasLoreCircle(QString name);
    /** Return the number of discipline the player has learned */
    int GetDisciplineCount();
    /** Return true if the player has learned the discipline and it is active */
    bool IsDisciplineActive(QString name);
    /** Return true if the player has the Weaponskill in a weapon class */
    bool HasWeaponskillIn(QString className);
    /** Return true if the player has the Weaponmastery in a weapon class */
    bool HasWeaponmasteryIn(QString className);
    /** Return true if the player has a bonus due to the Weaponskill (check equipped weapon) */
    bool HasWeaponskillBonus();
    /** Return the player's base endurance (the max value) includeing bonuses */
    int GetBaseEndurance();
    /** Return the player's base endurance (the max value) without bonuses */
    int GetBaseEnduranceNoBonus() { return BaseEndurance; }
    /** Sets the player's base endurance (the max value) */
    void SetBaseEnduranceNoBonus(int value) { BaseEndurance = value; }
    /** Return the player's current endurance */
    int GetEndurance();
    /** Set the player's current endurance */
    void SetEndurance(int);
    /** Set the player's current endurance without check on base endurance */
    void ForceEndurance(int);
    /** Return the player's base combat skill */
    int GetBaseCombatSkill();
    /** Set the player's current endurance */
    void SetBaseCombatSkill(int);
    /** Set the player's temporary combat skill delta.
        It will be set to 0 after a combat or when leaving a section */
    void SetTempCombatSkillDelta(int);
    /** Return true if the player has an item with the given name */
    bool HasItem(QString name);
    /** Return true if the player has a special item with the given name */
    bool HasSpecialItem(QString name);
    /** Adds to the player an item (or special item) with the given name */
    Item* AddItem(QString name, bool special);
    /** Remove one item from the player, returns true if the item has been removed */
    bool RemoveItem(QString name);
    /** Return one item from the player */
    Item* GetItem(QString name);

    /** Adds to the player a meal with the given name */
    Item* AddMeal(QString name);
    /** Adds to the player a healing item with the given name and effect */
    Item* AddHealingItem(QString name, int enduranceRestore);

    /** Gets the maximum number of arrows the player can carry */
    int GetMaxArrows();
    /** Gets the number of arrows the player has */
    int GetArrows();
    /** Sets the number of arrows the player has */
    void SetArrows(int value);

    /** Gets the gold crowns the player has */
    int GetGoldCrowns();
    /** Sets the gold crowns the player has */
    void SetGoldCrowns(int);

    /** Return the number of special items */
    int SpecialItemsCount();
    /** Return the number of backpack items */
    int ItemsCount();
    /** Return the number of meals */
    int MealsCount();
    /** Returns a list of all the items */
    QStringList GetAllItems();
    /** Returns a list of all the special items */
    QStringList GetSpecialItems();
    /** Returns a list of all the backpack items */
    QStringList GetItems();
    /** Returns a list of all the meals */
    QStringList GetMeals();

    /** Hides an item from the inventory */
    bool HideItem(QString name);
    /** Restore a hidden item to the inventory */
    bool RestoreHiddenItem(QString name);
    /** Get the list of hidden items */
    QStringList GetHiddenItems();

    /** Returns true if the player can hunt */
    bool CanHunt() const { return m_CanHunt; }
    /** Sets if the player can hunt */
    void SetCanHunt(bool value) { m_CanHunt = value; }

    /** Returns true if the player has the backpack */
    bool HasBackpack() const { return m_Backpack; }
    /** Sets if the player has the backpack */
    void SetHasBackpack(bool value);
    /** Returns true if the player's backpack is full */
    bool IsBackpackFull();
    /** Returns the player's backpack free space */
    int GetBackpackFreeSpace();

    /** Returns true if the player has at least one quiver */
    bool HasQuiver();
    /** Returns true if the player can use a bow (has a bow, a quiver and at least one arrow) */
    bool CanUseBow();
    /** Returns the bonus to use when picking a random number when using a bow */
    int GetBowBonus(bool includeDisciplines=true);
    /** Returns the bonus to use when picking a random number when throwing a weapon */
    int GetThrowBonus();

    /** Returns the numer of normal (non special) weapons */
    int WeaponsCount();
    /** Returns the list of (normal) weapons from the player's inventory */
    QStringList GetWeapons();
    /** Returns the list of all weapons from the player's inventory */
    QStringList GetAllWeapons();
    /** Returns the weapon currently equipped (or NULL if no weapon is equipped) */
    Weapon* GetEquippedWeapon();
    /** Removes a weapon from the player's inventory */
    void RemoveWeapon(QString name);
    /** Adds a weapon to the player's inventory with the given name and class (checks for weapon maximum number)*/
    Weapon* AddWeapon(QString name, QString classname);
    /** Adds a special item weapon to the player's inventory with the given name and class */
    Weapon* AddSpecialWeapon(QString name, QString classname);
    /** Returns true if the player has the given weapon */
    bool HasWeapon(QString name);
    /** Returns true if the player has at least one weapon of the given class */
    bool HasWeaponOfClass(QString wClass);
    /** Returns a list of armors currently equipped */
    QList<Armor*> GetEquippedArmors();
    /** Returns the weapon with the given name (or NULL) */
    Weapon* GetWeapon(QString name);

    /** Returns true if the given item is equipped */
    bool IsEquipped(QString name);
    /** Equips the given item */
    void Equip(QString name);

    /** Returns true if the player can use a shield */
    bool GetCanUseShield();
    /** Sets if the player can use a shield */
    void SetCanUseShield(bool value);

    /** Returns the player's available hands */
    int GetHands();
    /** Sets the player's available hands */
    void SetHands(int value);
    /** Returns the player's free hands */
    int GetFreeHands();

    /** Returns the name of the item used to set the temporary comba skill bonus*/
    QString GetTemporaryCombatSkillItem() { return GetProperty("TemporaryCombatSkillItem"); }
    /** Sets the name of the item used to set the temporary comba skill bonus*/
    void SetTemporaryCombatSkillItem(QString value) { SetProperty("TemporaryCombatSkillItem", value); }

    /** Returns the action chart's notes*/
    QString GetNotes() { return GetProperty("Notes"); }
    /** Sets the action chart's notes*/
    void SetNotes(QString value) { SetProperty("Notes", value); }
    /** Appends a line to the action chart's notes*/
    void AppendNotesLine(QString value);    
};

#endif // PLAYER_H
