#ifndef ITEM_H
#define ITEM_H

#include <QObject>
#include <QHash>
#include <QScriptEngine>

class PropertiesItem : public QObject
{
    Q_OBJECT

public:
    explicit PropertiesItem(QObject *parent = 0);
    PropertiesItem(const PropertiesItem& item);

    QHash<QString, QString> Properties;

public slots:
    bool HasProperty(QString name) { return Properties.contains(name); }
    QString GetProperty(QString name);
    void SetProperty(QString name, QString value);
    bool GetBoolProperty(QString name) { return GetProperty(name).toInt() == 1;}
    void SetBoolProperty(QString name, bool value) { SetProperty(name, value ? "1" : "0"); }
};


//!  Describes a item
/*!

*/
class Item : public PropertiesItem
{
    Q_OBJECT
public:
    explicit Item(QObject *parent = 0);
    Item(const Item& item);

    QString Name;
    QString Description;

    bool IsUsable() { return GetIsMeal() || GetEnduranceRestore() || GetCombatskillDelta(); }
    void Use(QScriptEngine* engine, QScriptValue* functions);

public slots:
    /** Returns the item name */
    QString GetName() { return Name; }

    /** Returns the item UUID */
    QString GetUUID() { return GetProperty("UUID"); }
    /** Sets the item UUID */
    void SetUUID(QString value) { SetProperty("UUID", value); }

    /** Returns true if the item is a backpack item*/
    bool GetIsBackpackItem() { return !GetIsSpecial() && !GetIsWeapon(); }

    /** Returns true if the item is a special item*/
    bool GetIsSpecial() { return GetBoolProperty("IsSpecial"); }
    /** Sets if the item is a special item*/
    void SetIsSpecial(bool value) { SetBoolProperty("IsSpecial", value); }

    /** Returns true if the item is a backpack */
    bool GetIsBackpack() { return GetBoolProperty("IsBackpack"); }
    /** Sets if the item is a backpack */
    void SetIsBackpack(bool value) { SetBoolProperty("IsBackpack", value); }

    /** Returns true if the item is a quiver */
    bool GetIsQuiver() { return GetBoolProperty("IsQuiver"); }
    /** Sets if the item is a quiver */
    void SetIsQuiver(bool value) { SetBoolProperty("IsQuiver", value); }

    /** Returns the number of arrows the item contains */
    int GetArrows() { return GetProperty("Arrows").toInt(); }
    /** Sets the number of arrows the item contains */
    void SetArrows(int value) { SetProperty("Arrows", QString::number(value)); }

    /** Returns true if the item is a meal */
    bool GetIsMeal() { return GetBoolProperty("IsMeal"); }
    /** Sets if the item is a meal */
    void SetIsMeal(bool value) { SetBoolProperty("IsMeal", value); }

    /** Returns the endurance points restored by the item when used */
    int GetEnduranceRestore() { return GetProperty("EnduranceRestore").toInt(); }
    /** Sets the endurance points restored by the item when used */
    void SetEnduranceRestore(int value) { SetProperty("EnduranceRestore", QString::number(value)); }

    /** Returns the endurance points bonus when equipped */
    int GetEnduranceDelta() { return GetProperty("EnduranceDelta").toInt(); }
    /** Sets the endurance points bonus when equipped */
    void SetEnduranceDelta(int value) { SetProperty("EnduranceDelta", QString::number(value)); }

    /** Returns the item description */
    QString GetDescription();
    /** Sets the item description */
    void SetDescription(QString value) { Description = value; }

    /** Returns the combat skill bonus when equipped */
    int GetCombatskillDelta() { return GetProperty("CombatskillDelta").toInt(); }
    /** Sets the combat skill bonus when equipped */
    void SetCombatskillDelta(int value) { SetProperty("CombatskillDelta", QString::number(value)); }

    /** Returns the item quantity */
    int GetQuantity() { return GetProperty("Quantity").toInt(); }
    /** Sets the item quantity */
    void SetQuantity(int value) { SetProperty("Quantity", QString::number(value)); }

    /** Returns the item size (space used in the backpack) */
    int GetSize() { return GetProperty("Size").toInt(); }
    /** Sets the item size (space used in the backpack) */
    void SetSize(int value) { SetProperty("Size", QString::number(value)); }

    /** Returns true if the item is equipped */
    bool GetIsEquipped() { return GetBoolProperty("IsEquipped"); }
    /** Sets if the item is equipped */
    void SetIsEquipped(bool value) { SetBoolProperty("IsEquipped", value); }
    /** Equip the item */
    void Equip() { if (GetCanBeEquipped()) SetIsEquipped(true); }
    /** Unequip the item */
    void Unequip() { SetIsEquipped(false); }

    /** Returns true if the item is an arrow */
    bool GetIsArrow() { return GetBoolProperty("IsArrow"); }
    /** Sets if the item is an arrow*/
    void SetIsArrow(bool value) { SetBoolProperty("IsArrow", value); }

    /** Returns true if the item is a weapon */
    bool GetIsWeapon() { return GetBoolProperty("IsWeapon"); }
    /** Sets if the item is a weapon*/
    void SetIsWeapon(bool value) { SetBoolProperty("IsWeapon", value); }

    /** Returns true if the item can be equipped */
    bool GetCanBeEquipped();
    /** Sets if the item can be equipped */
    void SetCanBeEquipped(bool value);

    /** Returns true if the item is an armor */
    bool GetIsArmor() { return GetBoolProperty("IsArmor"); }
    /** Sets if the item is an armor */
    void SetIsArmor(bool value) { SetBoolProperty("IsArmor", value); }

    /** Returns the armor class */
    int GetArmorClass() { return GetProperty("ArmorClass").toInt(); }
    /** Sets the armor class */
    void SetArmorClass(int value) { SetProperty("ArmorClass", QString::number(value)); }
    /** Sets the armor class */
    void SetArmorClass(QString value);

    /** Returns the hands needed to equip this item */
    int GetHands() { return GetProperty("Hands").toInt(); }
    /** Sets the hands needed to equip this item */
    void SetHands(int value) { SetProperty("Hands", QString::number(value)); }

    /** Returns the bonus this item gives when picking a random number */
    int GetRandomBonus() { return GetProperty("RandomBonus").toInt(); }
    /** Sets the bonus this item gives when picking a random number */
    void SetRandomBonus(int value) { SetProperty("RandomBonus", QString::number(value)); }
};

//!  Describes a weapon
/*!

*/

class Weapon : public Item
{
    Q_OBJECT
public:
    /*! List of weapon classes */
    enum WeaponsClass
    {
       None = -1,
       Dagger,
       Spear,
       Mace,
       ShortSword,
       WarHammer,
       Axe,
       Sword,
       Quarterstaff,
       Broadsword,
       Bow
    };

    explicit Weapon(QObject *parent = 0);
    Weapon(const Weapon& weapon);

    static QString ClassDescription(int Class);
    static int GetClass(QString);

public slots:
    /** Returns a list of the weapon classes */
    QList<int> Classes();
    /** Adds a weapon class */
    void AddClass(int iClass);
    /** Returns the hands needed to equip this weapon */
    int GetHands();

};

class ShopItem : public Weapon
{
    Q_OBJECT
public:
    explicit ShopItem(QObject *parent = 0);

    int Price;
    int SellPrice;
    bool UnlimitedQuantity;

    bool IsWeapon() { return Classes().count() > 0; }
};

//!  Describes an armor
/*!

*/

class Armor : public Item
{
    Q_OBJECT
public:
    /*! List of armor classes */
    enum ArmorClass
    {
       None = -1,
       Head,
       Body,
       Hand,
       Shield
    };

    explicit Armor(QObject *parent = 0);
    Armor(const Armor& armor);

    static int GetClass(QString);

public slots:
    /** Returns the armor class */
    int GetClass();
    /** Sets the armor class */
    void SetClass(int iClass);
    /** Returns the hands needed to equip this armor */
    int GetHands();
};

#endif // ITEM_H
