#include "book.h"
#include "kaidiscipline.h"

Enemy::Enemy(QObject *parent) :
    QObject(parent)
{
    CombatSkill = 0;
    Endurance = 0;
    Name = "";
    CanBeEvaded = false;
}

bool Enemy::IsDead()
{
    return Endurance <= 0;
}

Choice::Choice(QObject *parent) :
    QObject(parent)
{
    Enabled = true;
    Target = 0;
    Text = "";
    Random = false;
    ShowWithEnemies = false;
    MinValue = 0;
    MaxValue = 0;
    m_Abort = false;
}

Section::Section(QObject *parent) :
    QObject(parent)
{
    Number = 0;
    Text = "";
    AutoSave = true;
    Choices.clear();
    Enemies.clear();
    Items.clear();
    ShopItems.clear();
    GoldCrowns = 0;
    MustEat = 0;
    DidntEatEnduranceLoss = 0;
    CanHunt = true;
}

bool Section::HasPickableItems()
{
    return GoldCrowns > 0 || Items.count() > 0;
}

bool Section::HasRandomChoices()
{
    bool res = false;
    foreach(Choice* c, Choices){
        if (c->Random && c->Enabled){
            res = true;
            break;
        }
    }
    return res;
}

Choice* Section::GetChoiceFromTarget(int target)
{
    foreach (Choice* c, Choices){
        if (c->Target == target)
            return c;
    }
    return NULL;
}

Choice* Section::GetChoice(int index)
{
    return Choices.at(index);
}

int Section::UserChoicesCount()
{
    int res = 0;
    foreach(Choice* c, Choices){
        if (!c->Random)
            res++;
    }
    return res;
}

int Section::GetWeaponsCount()
{
    int res = 0;
    foreach (Item* i, Items){
        Weapon* w = qobject_cast<Weapon*>(i);
        if (w)
            res++;
    }
    return res;
}

int Section::GetEnemiesCount()
{
    int res = 0;
    foreach(Enemy* e, Enemies){
        if (!e->IsDead())
            res++;
    }
    return res;
}

void Section::RemoveEnemies()
{
    foreach(Enemy* e, Enemies)
        delete e;
    Enemies.clear();
}


bool Section::RemoveItem(QString name)
{
    if (!name.isEmpty()){
        Item* found = NULL;
        foreach(Item* i, Items){
            if (QString::compare(i->Name, name) == 0){
                found = i;
                break;
            }
        }
        if (found){
            Items.removeOne(found);
            return true;
        }
    }
    return false;
}


Item* Section::AddItem(QString name, bool special)
{
    if (!name.isEmpty()){
        Item* i = new Item(this);
        i->Name = name;
        i->SetIsSpecial(special);
        Items.append(i);
        return i;
    }
    return NULL;
}

void Section::CopyItemsToSection(int section)
{
    Book* book = GetBook();
    if (book){
        Section* s = book->GetSection(section);
        if (s)
            s->Items = Items;
    }
}

Book::Book(QObject *parent) :
    QObject(parent)
{
    Title = "";
    Sections.clear();
}

QString Book::GetVersion()
{
    return QString("%1.%2").arg(Version).arg(Revision);
}

int Book::GetSerie()
{
    if (!QString::compare(Serie, "Kai", Qt::CaseInsensitive))
        return Book::Kai;
    else if (!QString::compare(Serie, "Magnakai", Qt::CaseInsensitive))
        return Book::Magnakai;
    else if (!QString::compare(Serie, "Grand Master", Qt::CaseInsensitive))
        return Book::GrandMaster;
    else if (!QString::compare(Serie, "New Order", Qt::CaseInsensitive))
        return Book::NewOrder;
    return Book::Unknown;
}

Section* Book::GetSection(int section)
{
    if(Sections.contains(section))
        return Sections[section];
    return NULL;
}

bool Book::Load(QString filename, bool full)
{
    QFile file(filename);
    if(!file.open(QIODevice::ReadOnly))
        return false;

    Section* s = NULL;
    QXmlStreamReader xml(&file);
    while(!xml.atEnd() &&
          !xml.hasError()) {
        QXmlStreamReader::TokenType token = xml.readNext();
        if(token == QXmlStreamReader::StartDocument)
            continue;

        if(token == QXmlStreamReader::StartElement) {
            QString name = xml.name().toString();
            QXmlStreamAttributes attrs = xml.attributes();
            if (name == "book"){
                QFileInfo info(filename);
                BaseDirectory = info.absolutePath();
                Serie = attrs.value("serie").toString();
                Title = attrs.value("title").toString();
                Authors = attrs.value("authors").toString();
                Number = attrs.value("number").toString().toInt();
                Version = attrs.value("version").toString().toInt();
                Revision = attrs.value("revision").toString().toInt();
                ScriptedBy = attrs.value("scriptedby").toString();
                Sections.clear();

                if (!full)
                    break;
            }else if(name == "functions"){
                FunctionsScript = xml.readElementText();
            }else if (name == "section"){
                s = new Section(this);

                s->Number = attrs.value("number").toString().toInt();
                s->AutoSave = attrs.hasAttribute("autosave") ?
                              attrs.value("autosave").toString().toInt() == 1 : true;
                s->MustEat = attrs.value("musteat").toString().toInt();
                s->DidntEatEnduranceLoss = attrs.hasAttribute("didnteatenduranceloss") ?
                                           attrs.value("didnteatenduranceloss").toString().toInt() : 3;
                s->CanHunt = attrs.hasAttribute("canhunt") ?
                             attrs.value("canhunt").toString().toInt() == 1 : true;
                s->GoldCrowns = attrs.value("goldcrowns").toString().toInt();
            }else if(name == "text"){
                s->Text = xml.readElementText();
            }else if(name == "enterscript"){
                s->EnterScript = xml.readElementText();                
            }else if(name == "exitscript"){
                s->ExitScript = xml.readElementText();
            }else if(name == "showscript"){
                s->ShowScript = xml.readElementText();
            }else if(name == "choice" || name == "randomchoice"){
                Choice* c = ParseChoice(s, &xml);
                s->Choices.append(c);
            }else if(name == "enemy"){
                Enemy* e = ParseEnemy(s, &xml);
                s->Enemies.append(e);
            }else if(name == "weapon"){
                Weapon* w = ParseWeapon(s, &xml);
                s->Items.append(w);
            }else if(name == "item"){
                Item* i = ParseItem(s, &xml);
                s->Items.append(i);
            }else if(name == "shopitem"){
                ShopItem* si = ParseShopItem(s, &xml);
                s->ShopItems.append(si);
            }
        }else if(token == QXmlStreamReader::EndElement) {
            if (xml.name().toString() == "section")
                Sections[s->Number] = s;
        }
    }

    FunctionsScript = AddGenericFunctions(FunctionsScript);

    file.close();
    FileName = filename;

    if (GetSerie() == Book::Magnakai)
        LoreCircles::InitCircles();
    else
        LoreCircles::ClearCircles();

    return !xml.hasError();
}

QString Book::AddGenericFunctions(QString functions)
{
    QString res = functions.trimmed();
    QString generic = "use_EnduranceRestore: function()  { \
                           Player.SetEndurance(Player.GetEndurance() + Item.GetEnduranceRestore()); \
                       }, \
                       use_CombatskillDelta: function()  { \
                           Player.SetTempCombatSkillDelta(Item.GetCombatskillDelta()); \
                           Player.SetTemporaryCombatSkillItem(Item.GetName()); \
                       }, \
                       use_Adgana: function()  { \
                           var used = parseInt(Player.GetProperty('Adgana_used')); \
                           if (isNaN(used)) \
                               used = 0; \
                           if (used > 0) \
                             Player.SetTempCombatSkillDelta(3); \
                           else \
                             Player.SetTempCombatSkillDelta(6); \
                           Player.SetTemporaryCombatSkillItem(Item.GetName()); \
                           Player.SetProperty('Adgana_used', used + 1); \
                       }, \
                       endeffect_Adgana: function()  { \
                           Window.DisableChoiceButtons(); \
                           var used = parseInt(Player.GetProperty('Adgana_used')); \
                           if (isNaN(used)) \
                               used = 0; \
                           var rnd = Window.GetRandomNumber(); \
                           if (used == 0 && rnd <= 1 || used > 0 && rnd <= 3){ \
                               Window.MessageBox('Info', 'You have become addicted to Adgana'); \
                               Player.SetBaseEnduranceNoBonus(Player.GetBaseEnduranceNoBonus() - 4); \
                               Player.SetProperty('Adgana_Addicted', 'true'); \
                           } \
                           Window.EnableChoiceButtons(); \
                       },";
    if (!res.isEmpty() && res.startsWith("({")){
        res.insert(2, generic);
    }else{
        res = QString("({%1})").arg(generic.remove(generic.length()- 1, 1));
    }
    return res;
}

Choice* Book::ParseChoice(Section* section, QXmlStreamReader* reader)
{    
    QString eName = reader->name().toString();
    Choice* c = new Choice(section);

    QXmlStreamAttributes attrs = reader->attributes();
    c->Random = eName == "randomchoice";
    c->Target = attrs.value("target").toString().toInt();
    c->MinValue = attrs.value("minvalue").toString().toInt();
    c->MaxValue = attrs.value("maxvalue").toString().toInt();
    c->ShowWithEnemies = attrs.value("showwithenemies").toString().toInt() == 1;

    while(!reader->atEnd() &&
          !reader->hasError()) {
        QXmlStreamReader::TokenType token = reader->readNext();
        QString name = reader->name().toString();
        if(token == QXmlStreamReader::StartElement){
            if (name == "text")
                c->Text = reader->readElementText();
            else if (name == "script")
                c->Script = reader->readElementText();
        }else if(token == QXmlStreamReader::EndElement && name == eName)
            break;
    }

    return c;
}

Enemy* Book::ParseEnemy(Section* section, QXmlStreamReader* reader)
{
    QString eName = reader->name().toString();
    Enemy* e = new Enemy(section);

    QXmlStreamAttributes attrs = reader->attributes();
    e->Name = attrs.value("name").toString();
    e->CombatSkill = attrs.value("skill").toString().toInt();
    e->Endurance = attrs.value("endurance").toString().toInt();
    e->CanBeEvaded = attrs.value("canbeevaded").toString().toInt() == 1;

    QString immune = attrs.value("immuneto").toString();
    if (!immune.isEmpty())
        e->DisciplineImmunities = immune.split(";", QString::SkipEmptyParts);

    QString wounded = attrs.value("woundedby").toString();
    if (!wounded.isEmpty())
        e->WoundedBy = wounded.split(";", QString::SkipEmptyParts);

    while(!reader->atEnd() &&
          !reader->hasError()) {
        QXmlStreamReader::TokenType token = reader->readNext();
        QString name = reader->name().toString();
        if(token == QXmlStreamReader::StartElement){
            if (name == "enterscript")
                e->EnterScript = reader->readElementText();
            else if (name == "exitscript")
                e->ExitScript = reader->readElementText();
            else if (name == "roundscript")
                e->RoundScript = reader->readElementText();
            else if (name == "damagescript")
                e->DamageScript = reader->readElementText();
        }else if(token == QXmlStreamReader::EndElement && name == eName)
            break;
    }
    return e;
}

Weapon* Book::ParseWeapon(Section* section, QXmlStreamReader* reader)
{
    QString eName = reader->name().toString();
    Weapon* w = new Weapon(section);

    QXmlStreamAttributes attrs = reader->attributes();
    w->Name = attrs.value("name").toString();
    w->Description = attrs.value("description").toString();
    if (attrs.hasAttribute("canbeequipped"))
        w->SetCanBeEquipped(attrs.value("canbeequipped").toString().toInt() == 1);
    w->SetIsSpecial(attrs.value("special").toString().toInt() == 1);
    w->SetCombatskillDelta(attrs.value("combatskilldelta").toString().toInt());
    w->SetRandomBonus(attrs.value("randombonus").toString().toInt());

    if (attrs.hasAttribute("hands"))
        w->SetHands(attrs.value("hands").toString().toInt());
    QString tclass = attrs.value("class").toString();
    QStringList ele = tclass.split(";");
    foreach(QString wclass, ele){
        int iclass = Weapon::GetClass(wclass);
        if (iclass != Weapon::None)
            w->AddClass(iclass);
    }

    while(!reader->atEnd() &&
          !reader->hasError()) {
        QXmlStreamReader::TokenType token = reader->readNext();
        QString name = reader->name().toString();
        if(token == QXmlStreamReader::EndElement && name == eName)
            break;
    }
    return w;
}

Item* Book::ParseItem(Section* section, QXmlStreamReader* reader)
{
    QString eName = reader->name().toString();
    Item* i = new Item(section);

    QXmlStreamAttributes attrs = reader->attributes();
    i->Name = attrs.value("name").toString();
    i->Description = attrs.value("description").toString();
    if (attrs.hasAttribute("canbeequipped"))
        i->SetCanBeEquipped(attrs.value("canbeequipped").toString().toInt() == 1);
    i->SetIsSpecial(attrs.value("special").toString().toInt() == 1);
    i->SetIsMeal(attrs.value("ismeal").toString().toInt() == 1);
    i->SetIsArmor(attrs.value("isarmor").toString().toInt() == 1);
    if (attrs.hasAttribute("armorclass")){
        i->SetArmorClass(attrs.value("armorclass").toString());
        if (i->GetArmorClass() == Armor::Shield)
            i->SetHands(1);
    }
    i->SetIsArrow(attrs.value("isarrow").toString().toInt() == 1);
    if (attrs.hasAttribute("quantity"))
        i->SetQuantity(attrs.value("quantity").toString().toInt());
    i->SetIsBackpack(attrs.value("isbackpack").toString().toInt() == 1);
    i->SetIsQuiver(attrs.value("isquiver").toString().toInt() == 1);
    i->SetArrows(attrs.value("arrows").toString().toInt());
    i->SetCombatskillDelta(attrs.value("combatskilldelta").toString().toInt());
    i->SetRandomBonus(attrs.value("randombonus").toString().toInt());
    i->SetEnduranceDelta(attrs.value("endurancedelta").toString().toInt());
    i->SetEnduranceRestore(attrs.value("endurancerestore").toString().toInt());
    i->SetSize(attrs.value("size").toString().toInt());
    if (i->GetSize() == 0)
        i->SetSize(1);
    if (i->GetEnduranceRestore() && i->Description.isEmpty())
        i->Description = QString("Restores %1 ENDURANCE POINTS").arg(i->GetEnduranceRestore());

    while(!reader->atEnd() &&
          !reader->hasError()) {
        QXmlStreamReader::TokenType token = reader->readNext();
        QString name = reader->name().toString();
        if(token == QXmlStreamReader::EndElement && name == eName)
            break;
    }
    return i;
}

ShopItem* Book::ParseShopItem(Section* section, QXmlStreamReader* reader)
{
    QString eName = reader->name().toString();
    ShopItem* i = new ShopItem(section);

    QXmlStreamAttributes attrs = reader->attributes();
    i->Name = attrs.value("name").toString();
    i->Description = attrs.value("description").toString();
    if (attrs.hasAttribute("canbeequipped"))
        i->SetCanBeEquipped(attrs.value("canbeequipped").toString().toInt() == 1);
    i->SetIsSpecial(attrs.value("special").toString().toInt() == 1);
    i->SetIsMeal(attrs.value("ismeal").toString().toInt() == 1);
    i->SetIsArrow(attrs.value("isarrow").toString().toInt() == 1);
    if (attrs.hasAttribute("quantity"))
        i->SetQuantity(attrs.value("quantity").toString().toInt());
    i->SetIsBackpack(attrs.value("isbackpack").toString().toInt() == 1);
    i->SetIsQuiver(attrs.value("isquiver").toString().toInt() == 1);
    i->SetArrows(attrs.value("arrows").toString().toInt());
    i->SetCombatskillDelta(attrs.value("combatskilldelta").toString().toInt());
    i->SetRandomBonus(attrs.value("randombonus").toString().toInt());
    i->SetEnduranceDelta(attrs.value("endurancedelta").toString().toInt());
    i->SetEnduranceRestore(attrs.value("endurancerestore").toString().toInt());
    i->SetSize(attrs.value("size").toString().toInt());
    if (i->GetSize() == 0)
        i->SetSize(1);
    if (i->GetEnduranceRestore() && i->Description.isEmpty())
        i->Description = QString("Restores %1 ENDURANCE POINTS").arg(i->GetEnduranceRestore());

    i->UnlimitedQuantity = attrs.value("unlimitedquantity").toString().toInt() == 1;

    i->Price = attrs.value("price").toString().toInt();
    i->SellPrice = attrs.value("sellprice").toString().toInt();

    if (attrs.hasAttribute("hands"))
        i->SetHands(attrs.value("hands").toString().toInt());
    int iclass = Weapon::GetClass(attrs.value("weaponclass").toString());
    if (iclass != Weapon::None){
        i->AddClass(iclass);
        i->SetIsWeapon(true);
    }else
        i->SetIsWeapon(false);

    while(!reader->atEnd() &&
          !reader->hasError()) {
        QXmlStreamReader::TokenType token = reader->readNext();
        QString name = reader->name().toString();
        if(token == QXmlStreamReader::EndElement && name == eName)
            break;
    }
    return i;
}

bool Book::Save(QString filename)
{
    QFile file(filename);
    if(!file.open(QIODevice::WriteOnly | QIODevice::Truncate))
        return false;

    QXmlStreamWriter writer(&file);
    writer.setAutoFormatting(true);
    QXmlStreamAttributes attrs;

    writer.writeStartDocument();

    writer.writeStartElement("book");
    attrs.clear();
    attrs.append("title", Title);
    attrs.append("serie", Serie);
    attrs.append("number", QString::number(Number));
    attrs.append("authors", Authors);
    attrs.append("version", QString::number(Version));
    attrs.append("revision", QString::number(Revision));
    attrs.append("scriptedby", ScriptedBy);
    writer.writeAttributes(attrs);

    writer.writeTextElement("functions", FunctionsScript);

    foreach (Section* s, Sections){
        writer.writeStartElement("section");
        attrs.clear();
        attrs.append("number", QString::number(s->Number));
        if (!s->AutoSave)
            attrs.append("autosave", "0");
        if (s->MustEat > 0)
            attrs.append("musteat", QString::number(s->MustEat));
        if (s->DidntEatEnduranceLoss != 3)
            attrs.append("didnteatenduranceloss", QString::number(s->DidntEatEnduranceLoss));
        if (!s->CanHunt)
            attrs.append("canhunt", "0");
        if (s->GoldCrowns > 0)
            attrs.append("goldcrowns", QString::number(s->GoldCrowns));
        writer.writeAttributes(attrs);

        if (!s->EnterScript.isEmpty())
            writer.writeTextElement("enterscript", s->EnterScript);
        if (!s->ExitScript.isEmpty())
            writer.writeTextElement("exitscript", s->ExitScript);

        writer.writeTextElement("text", s->Text);

        foreach (Enemy* e, s->Enemies){
            writer.writeStartElement("enemy");
            attrs.clear();
            attrs.append("name", e->Name);
            attrs.append("skill", QString::number(e->CombatSkill));
            attrs.append("endurance", QString::number(e->Endurance));
            if (e->CanBeEvaded)
                attrs.append("canbeevaded", "1");
            if (e->DisciplineImmunities.count() > 0)
                attrs.append("immuneto", e->DisciplineImmunities.join(";"));
            if (e->WoundedBy.count() > 0)
                attrs.append("woundedby", e->WoundedBy.join(";"));            
            writer.writeAttributes(attrs);

            if (!e->EnterScript.isEmpty())
                writer.writeTextElement("enterscript", e->EnterScript);
            if (!e->ExitScript.isEmpty())
                writer.writeTextElement("exitscript", e->ExitScript);
            if (!e->RoundScript.isEmpty())
                writer.writeTextElement("roundscript", e->RoundScript);
            if (!e->DamageScript.isEmpty())
                writer.writeTextElement("damagescript", e->DamageScript);

            writer.writeEndElement();
        }

        foreach (Item* i, s->Items){
            attrs.clear();
            attrs.append("name", i->Name);
            if (!i->Description.isEmpty())
                attrs.append("description", i->Description);
            if (i->GetIsSpecial())
                attrs.append("special", "1");

            Weapon* w = qobject_cast<Weapon*>(i);
            if (w){
                writer.writeStartElement("weapon");
                QString wclass;
                foreach (int c, w->Classes()){
                    wclass.append(Weapon::ClassDescription(c));
                    wclass.append(";");
                }
                if (wclass.count()){
                    wclass.remove(wclass.count() - 1, 1);
                    attrs.append("class", wclass);
                }
                writer.writeAttributes(attrs);

                writer.writeEndElement();
            }else{
                writer.writeStartElement("item");

                if (i->GetIsMeal())
                    attrs.append("ismeal", "1");
                if (i->GetIsBackpack())
                    attrs.append("isbackpack", "1");
                if (i->GetCombatskillDelta() != 0)
                    attrs.append("combatskilldelta", QString::number(i->GetCombatskillDelta()));
                if (i->GetEnduranceDelta() != 0)
                    attrs.append("endurancedelta", QString::number(i->GetEnduranceDelta()));
                if (i->GetEnduranceRestore() != 0)
                    attrs.append("endurancerestore", QString::number(i->GetEnduranceRestore()));
                if (i->GetSize() > 1)
                    attrs.append("size", QString::number(i->GetSize()));
                writer.writeAttributes(attrs);

                writer.writeEndElement();
            }
        }

        foreach (ShopItem* i, s->ShopItems){
            writer.writeStartElement("shopitem");

            attrs.clear();
            attrs.append("name", i->Name);
            if (!i->Description.isEmpty())
                attrs.append("description", i->Description);
            if (i->GetIsSpecial())
                attrs.append("special", "1");
            if (i->GetIsMeal())
                attrs.append("ismeal", "1");
            if (i->GetIsBackpack())
                attrs.append("isbackpack", "1");
            if (i->GetCombatskillDelta() != 0)
                attrs.append("combatskilldelta", QString::number(i->GetCombatskillDelta()));
            if (i->GetEnduranceDelta() != 0)
                attrs.append("endurancedelta", QString::number(i->GetEnduranceDelta()));
            if (i->GetEnduranceRestore() != 0)
                attrs.append("endurancerestore", QString::number(i->GetEnduranceRestore()));
            if (i->GetSize() > 1)
                attrs.append("size", QString::number(i->GetSize()));
            if (i->UnlimitedQuantity)
                attrs.append("unlimitedquantity", "1");
            attrs.append("price", QString::number(i->Price));
            attrs.append("sellprice", QString::number(i->SellPrice));
            QString wclass;
            foreach (int c, i->Classes()){
                wclass.append(Weapon::ClassDescription(c));
                wclass.append(";");
            }
            if (wclass.count()){
                wclass.remove(wclass.count() - 1, 1);
                attrs.append("weaponclass", wclass);
            }
            writer.writeAttributes(attrs);

            writer.writeEndElement();
        }


        foreach (Choice* c, s->Choices){
            if (c->Random)
                writer.writeStartElement("randomchoice");
            else
                writer.writeStartElement("choice");

            attrs.clear();
            attrs.append("target", QString::number(c->Target));
            if (c->Random){
                attrs.append("minvalue", QString::number(c->MinValue));
                attrs.append("maxvalue", QString::number(c->MaxValue));
            }
            writer.writeAttributes(attrs);

            if (!c->Script.isEmpty())
                writer.writeTextElement("script", c->Script);
            writer.writeTextElement("text", c->Text);

            writer.writeEndElement();
        }

        writer.writeEndElement();
    }

    writer.writeEndElement();
    writer.writeEndDocument();
    file.close();
    FileName = filename;
    return true;
}

QString Book::GetAttribute(QXmlStreamAttributes attrs, QString name, QString defaultValue)
{
    if (attrs.hasAttribute(name))
        return attrs.value(name).toString();
    return defaultValue;
}
