#include "downloadwindow.h"
#include "ui_downloadwindow.h"
#include <QMessageBox>
#include <QDir>
#include <QDebug>
#include "zip/unzip.h"

DownloadWindow::DownloadWindow(QWidget *parent, BookEntry* book, QString url, QString filename) :
    QDialog(parent),
    ui(new Ui::DownloadWindow)
{
    ui->setupUi(this);

    m_Url = url;
    m_Filename = filename;
    m_Book = book;
    m_Nam = new QNetworkAccessManager(this);
    connect(m_Nam, SIGNAL(finished(QNetworkReply *)), this, SLOT(ReqFinishedSlot(QNetworkReply *)));
    m_CurrentReply = NULL;

    connect(ui->pbnCancel, SIGNAL(clicked()), this, SLOT(CancelSlot()));
#ifdef Q_WS_MAEMO_5
    setMinimumHeight(300);
#endif
    m_CanClose = false;
    m_Error = false;
}

DownloadWindow::~DownloadWindow()
{
    delete ui;
}

int DownloadWindow::exec()
{
    if (m_Book != NULL){
        m_TempFilename = QString("%1/%2").arg(QDir::tempPath()).arg("temp.zip");
        m_File = new QFile(m_TempFilename);
        if (!m_File->open(QIODevice::WriteOnly)){
            delete m_File;
            m_File = NULL;
            ui->lblInfo->setText("Error opening file");
            qDebug() << "Error opening file" << m_TempFilename;
        }else{
            qDebug() << "Downloading from" << m_Book->DownloadUrl;
            QUrl url(m_Book->DownloadUrl);
            if (!url.isValid())
                ui->lblInfo->setText("Invalid url");
            else{
                QNetworkRequest request(url);
                m_CurrentReply = m_Nam->get(request);
                m_CurrentReply->ignoreSslErrors();
                connect(m_CurrentReply, SIGNAL(readyRead()), this, SLOT(ReqReadyReadSlot()));
                connect(m_CurrentReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(ReqErrorSlot(QNetworkReply::NetworkError)));
                connect(m_CurrentReply, SIGNAL(downloadProgress(qint64, qint64)), SLOT(ReqDownloadProgressSlot(qint64, qint64)));
                ui->lblInfo->setText("Downloading file");
            }
        }
    }else{
        m_TempFilename = QString("%1/%2").arg(QDir::tempPath()).arg("download.tmp");
        m_File = new QFile(m_TempFilename);
        if (!m_File->open(QIODevice::WriteOnly)){
            delete m_File;
            m_File = NULL;
            ui->lblInfo->setText("Error opening file");
            qDebug() << "Error opening file" << m_TempFilename;
        }else{
            qDebug() << "Downloading from" << m_Url;
            QUrl url(m_Url);
            if (!url.isValid())
                ui->lblInfo->setText("Invalid url");
            else{
                QNetworkRequest request(url);
                m_CurrentReply = m_Nam->get(request);
                m_CurrentReply->ignoreSslErrors();
                connect(m_CurrentReply, SIGNAL(readyRead()), this, SLOT(ReqReadyReadSlot()));
                connect(m_CurrentReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(ReqErrorSlot(QNetworkReply::NetworkError)));
                connect(m_CurrentReply, SIGNAL(downloadProgress(qint64, qint64)), SLOT(ReqDownloadProgressSlot(qint64, qint64)));
                ui->lblInfo->setText("Downloading file");
            }
        }
    }
    return QDialog::exec();
}

void DownloadWindow::ReqReadyReadSlot()
{
    if (m_File != NULL){
        QByteArray red = m_CurrentReply->readAll();
        if (QString::fromUtf8(red).contains("Project Aon: Errors / 404")){
            ReqErrorSlot(QNetworkReply::ContentNotFoundError);
            m_CurrentReply->abort();
            ui->lblInfo->setText(QString("Error downloading %1 - Server replied: Not Found").arg(m_Url));
            m_Error = true;
            return;
        }
        m_File->write(red);
    }
}

void DownloadWindow::ReqErrorSlot(QNetworkReply::NetworkError)
{
    ui->lblInfo->setText(m_CurrentReply->errorString());
    qDebug() << "Network error" << m_CurrentReply->errorString();
}

void DownloadWindow::ReqDownloadProgressSlot(qint64 bytesReceived, qint64 bytesTotal)
{
    if (bytesTotal > 0){
        ui->lblInfo->setText(QString("Downloading file [%1 / %2]").arg(GetSize(bytesReceived)).arg(GetSize(bytesTotal)));
        ui->progressBar->setValue((float)bytesReceived / (float)bytesTotal * 100.0);
    }
}

void DownloadWindow::ReqFinishedSlot(QNetworkReply* reply)
{
    reply->deleteLater();
    disconnect(m_CurrentReply, SIGNAL(readyRead()), this, SLOT(ReqReadyReadSlot()));
    disconnect(m_CurrentReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(ReqErrorSlot(QNetworkReply::NetworkError)));
    m_CurrentReply = NULL;

    if (m_File){
        bool ok = true;
        m_File->close();

        if (!reply->error() == QNetworkReply::NoError || m_Error){
            ok = false;
        }else{
            if (m_Book != NULL){
                ui->lblInfo->setText("Unzipping file");
                ui->lblInfo->update();
                UnZip* unzip = new UnZip();
                if (!unzip->isZipArchive(m_File->fileName())){
                    ui->lblInfo->setText("Not a valid zip file");
                    ok = false;
                }else{
                    if (unzip->openArchive(m_File->fileName()) != UnZip::Ok){
                        ui->lblInfo->setText("Error opening zip file");
                        ok = false;
                    }else{
                        if (unzip->extractAll(QDir(Books::Directory()), UnZip::ExtractPaths) != UnZip::Ok){
                            ui->lblInfo->setText("Error unzipping file");
                            ok = false;
                        }
                    }
                }
                QFile::remove(m_TempFilename);
                delete unzip;
            }else{
                if (QFile::exists(m_Filename))
                    QFile::remove(m_Filename);
                if (!QFile::copy(m_TempFilename, m_Filename)){
                    ui->lblInfo->setText("Error writing file");
                    ok = false;
                }else{
                    QFile::remove(m_TempFilename);
                }
            }
        }
        if (ok){
            ui->lblInfo->setText("Done!");
            m_File = NULL;
            accept();
        }
    }
}

void DownloadWindow::CancelSlot()
{
    if (m_CurrentReply)
        m_CurrentReply->abort();
    m_CanClose = true;
    reject();
}

QString DownloadWindow::GetSize(qint64 size)
{
    if (size / 1024000 > 0)
        return QString("%1Mb").arg(size / 1024000.0, 0, 'f', 2);
    return QString("%1Kb").arg(size / 1024.0, 0, 'f', 2);
}

void DownloadWindow::closeEvent(QCloseEvent* event)
{
    if (m_CanClose){
        event->accept();
    }else{
        event->ignore();
    }
}
