/*
 * This file is part of the Liquid project.
 * It realizes additional graphic effects and Ice tool for editor.
 *
 * Copyright (C) 2009 Kirpichonock K.N. <kirpiche@cs.karelia.ru>
 * Copyright (C) 2009 Volkov A.A. <volkov@cs.karelia.ru>
 * Copyright (C) 2009 Dmitriev V.V. <vdmitrie@cs.karelia.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include "filters.h"
#include "math.h"
#include <QDebug>

// Constructor
Filters::Filters()
{
    radius = 1;
}

// Implements blur of neighbourhood of pixel (i,j)
QRgb Filters::qrgb(int i, int j, QImage *image) {
    QRgb rgb;
    int k,l;
    int fl = 1;
    int flc = fl * 50;
    int r = 0;
    int g = 0;
    int b = 0;

    sum = fl*fl << 2;    

    for (k = i - upperBorder; k <= i + lowerBorder; k++)
        for (l = j - leftBorder; l <= j + rightBorder; l++)
            if((k != i) && (l != j)) {
                rgb = image->pixel(k, l);
                r += fl * ((rgb >> 16) & 0xff);
                g += fl * ((rgb >> 8) & 0xff);
                b += fl * (rgb & 0xff);                
            }

    rgb = image->pixel(i, j);
    r += flc * ((rgb >> 16) & 0xff);
    g += flc * ((rgb >> 8) & 0xff);
    b += flc * (rgb & 0xff);
    sum += flc;

    r /=  sum;
    g /=  sum;
    b /=  sum;

    r = r > 255 ? 255 : r;
    b = b > 255 ? 255 : b;
    g = g > 255 ? 255 : g;

    return qRgb(r,g,b);
}

// Implements blur of all workspace
QImage  Filters::blur(QImage *image) {
    imageWidth = image->width();
    imageHeight = image->height();
    QRgb rgb;
    QImage *theImage = new QImage(image->width(), image->height(), QImage::Format_RGB32);
    *theImage = *image;    

    for (int i = 0 ; i < imageWidth; i++)
        for (int j = 0; j < imageHeight; j++) {
            checking(i, j, image);
            int r = 0;
            int g = 0;
            int b = 0;

            for (int k = i - upperBorder; k <= i + lowerBorder; k++)
                for (int l = j - leftBorder; l <= j + rightBorder; l++){
                        rgb = image->pixel(k, l);
                        r += ((rgb >> 16) & 0xff);
                        g += ((rgb >> 8) & 0xff);
                        b += (rgb & 0xff);                
                }
            sum = pow(2*radius+1,2);
            r =  r/(sum);
            g =  g/(sum);
            b =  b/(sum);

            r = r > 255 ? 255 : r;
            b = b > 255 ? 255 : b;
            g = g > 255 ? 255 : g;
            theImage->setPixel(i, j, qRgb(r,b,g));
        }

    *image = *theImage;
    delete theImage;
    return *image;
}

// Implements blur of region (Ice Tool)
QImage Filters::blurCircle(int i, int j, int r, QImage *image) {
    int k;
    imageWidth = image->width();
    imageHeight = image->height();

    QImage *theImage = new QImage(imageWidth, imageHeight, QImage::Format_RGB32);
    *theImage = *image;

    int rad = r;
    int x = 0;
    int y = rad;
    int d = 3 - 2 * rad;

    do {
        // Blur horizontally
        for (k = j - y; k <= j + y; k++)
            if ((k >= 0) && (k < imageHeight)) {
                if ((i + x >= 0) && (i + x < imageWidth)) {
                    checking(i + x, k, image);
                    theImage->setPixel(i + x, k, qrgb(i + x, k, image));
                }
                if ((i - x >= 0) && (i - x < imageWidth)) {
                    checking(i - x, k, image);
                    theImage->setPixel(i - x, k, qrgb(i - x, k, image));
              }
            }

        // Blur vertically
        for (k = j - x; k <= j + x; k++)
            if ((k >= 0) && (k < imageHeight)) {
                if ((i + y >= 0) && (i + y < imageWidth)) {
                    checking(i + y, k, image);
                    theImage->setPixel(i + y, k, qrgb(i + y, k, image));
                }
                if ((i - y >= 0) && (i - y < imageWidth)) {
                    checking(i - y, k, image);
                    theImage->setPixel(i-y, k, qrgb(i - y, k, image));
                }
            }

        if (d < 0)
            d += (4*x) + 6;
        else {
            d += 10 + 4*(x - y);
            y--;
        }
   } while(x++ < y);

   *image = *theImage;
   delete theImage;
   return *image;
}

// Checks boundaries of blur
void Filters::checking(int i, int j, QImage *) {
    if (i < radius)
        upperBorder = i;
    else
        upperBorder = radius;

    if (i > imageWidth - radius - 1)
        lowerBorder = imageWidth - i - 1;
    else
        lowerBorder = radius;

    if (j < radius)
        leftBorder = j;
    else
        leftBorder = radius;

    if (j > imageHeight - radius - 1)
        rightBorder = imageHeight - j - 1;
    else
        rightBorder = radius;
}

// Makes kernel for gaussian blur
float* Filters::makeKernel(float rad){
    int r = (int)ceil(rad);
    int rows = r * 2 + 1;
    float *kernel = new float[rows];
    float sigma = rad/3;
    float sigma22 = 2 * sigma * sigma;
    float sigmaPi2 = 2*M_PI*sigma;
    float sqrtSigmaPi2 = (float)sqrt(sigmaPi2);
    float radius2 = rad*rad;
    float total = 0;
    int index = 0;
    for (int row = -r; row <= r; row++) {
        float distance = row*row;
        if (distance > radius2)
            kernel[index] = 0;
        else
            kernel[index] =  exp(-(distance)/sigma22) / sqrtSigmaPi2;

        total += kernel[index];
        index++;
    }
    for (int i = 0; i < rows; i++)
        kernel[i] /= total;
    m_sum = total;
    return kernel;
}

