#include "lastgowh.h"
#include "ui_lastgowh.h"
#include "utils.h"
#include <QtDebug>
#include <QMap>
#include <QFile>
#include <QFileDialog>
#include <QDomDocument>
#include "requests.h"

LastGoWH::LastGoWH(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::LastGoWH)
{
    ui->setupUi(this);
    loadSettings();

    if((api_key != "") | (api_secret != ""))
        ac = new ApiClient(api_key, api_secret);

    initializeMediaPlayer();

    connectMenuActions();
    connectSlots();

    isPlaying = false;
    isTuning = false;
    isGettingPlayList = false;

    playListIndex = 0;

    if((api_key == "") | (api_secret == ""))
    {
        ui->playStopButton->setEnabled(false);
        ui->statusLabel->setText("Status: you must import API key before using LastGo.");
    }
}

void LastGoWH::requestReplied(QMap<QString, QString> reply)
{
    if(reply["status"] == "ok")
    {
        qDebug() << reply["status"];
    }
    else if(reply["status"] == "failed")
    {
        qDebug() << reply["error"];
        qDebug() << reply["message"];
    }
}

void LastGoWH::radioGetPlaylistReplied(QList<LastfmTrack *> tracks)
{
    playlistTracks = tracks;
    playListIndex = 0;

    if(isPlaying == true)
        playTrack(playListIndex);
}

LastGoWH::~LastGoWH()
{
    saveSettings();
    delete ui;
}

void LastGoWH::setupDesktopUI()
{
    this->resize(556, 287);
    this->setMaximumSize(556, 287);
    this->setMinimumSize(556, 287);
}

void LastGoWH::playStopButtonClicked()
{
    if(isPlaying == true)
        isPlaying = false;
    else
        isPlaying = true;

    if(isPlaying)
    {
        if((username == "") | (password == ""))
        {
            isPlaying = false;
            showSettingsDialog();
            return;
        }

        if(sk == "")
        {
            ac->authGetMobileSession(username, password);
            return;
        }

        if(playlistTracks.count() == 0)
        {
            ac->radioGetPlaylist(sk);
        }
        else
        {
            if(playListIndex < playlistTracks.count())
            {
                playTrack(playListIndex);
            }
            else
            {
                ac->radioGetPlaylist(sk);
            }
        }
    }
    else
    {
        playListIndex += 1;

        setPlayButtonToPlay();
        ui->titleDataLabel->clear();
        ui->artistDataLabel->clear();
        ui->albumDataLabel->clear();
        ui->trackImage->setPixmap(QPixmap(":/images/images/vinile.jpg"));
        ui->progressBar->setValue(0);

        player->stop();
    }
}

void LastGoWH::initializeMediaPlayer()
{
    player = new QMediaPlayer(this);
    playlist = new QMediaPlaylist;

    // hack to make things work with maemo qt-mobility version
    #if defined(Q_WS_MAEMO_5)
        playlist->setMediaObject(player);
    #else
        player->setPlaylist(playlist);
    #endif
}

void LastGoWH::showSettingsDialog()
{
    SettingsDialog dialog(this, this);
    dialog.setWindowTitle("Settings");
    dialog.showNormal();
    dialog.exec();
}

void LastGoWH::setUsername(QString user)
{
    username = user;
}

void LastGoWH::setPassword(QString pass)
{
    password = pass;
}

void LastGoWH::saveSettings()
{
    QSettings settings("LastGo", "LastGo");

    settings.setValue("username", username);
    settings.setValue("password", password);
    settings.setValue("sk", sk);
    settings.setValue("api_key", api_key);
    settings.setValue("api_secret", api_secret);
}

void LastGoWH::loadSettings()
{
    QSettings settings("LastGo", "LastGo");

    username = settings.value("username").toString();
    password = settings.value("password").toString();
    sk = settings.value("sk").toString();
    api_key = settings.value("api_key").toString();
    api_secret = settings.value("api_secret").toString();
}

void LastGoWH::importApiKey()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Import Api Key"), "/home/user", tr("Api Key Files (*.xml)"));

    if(!filename.isEmpty())
    {
        QFile file(filename);

        if(file.open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QString data = QString(file.readAll());
            QDomDocument doc("API");
            doc.setContent(data.toUtf8());

            api_key = doc.elementsByTagName("api_key").at(0).toElement().text();
            api_secret = doc.elementsByTagName("api_secret").at(0).toElement().text();

            if((api_key != "") | (api_secret != ""))
            {
                ui->playStopButton->setEnabled(true);
                ui->statusLabel->setText("");
                ac = new ApiClient(api_key, api_secret);
            }

            file.close();
        }
    }
}

void LastGoWH::connectMenuActions()
{
    connect(ui->actionSettings, SIGNAL(triggered()), this, SLOT(showSettingsDialog()));
    connect(ui->actionImportApiKey, SIGNAL(triggered()), this, SLOT(importApiKey()));
    connect(ui->actionRadioList, SIGNAL(triggered()), this, SLOT(selectRadio()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(showAbout()));
}

void LastGoWH::connectSlots()
{
    connect(ui->playStopButton, SIGNAL(clicked()), this, SLOT(playStopButtonClicked()));
    connect(ui->nextTrackButton, SIGNAL(clicked()), this, SLOT(nextButtonClicked()));
    connect(player, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(playerStateChanged(QMediaPlayer::State)));
    connect(player, SIGNAL(positionChanged(qint64)), this, SLOT(trackPositionChanged(qint64)));
    connect(ac, SIGNAL(radioGetPlaylistData(QList<LastfmTrack*>)), this, SLOT(radioGetPlaylistReplied(QList<LastfmTrack*>)));
    connect(ac, SIGNAL(authGetMobileSessionData(QMap<QString, QString>)), this, SLOT(authGetMobileSessionReplied(QMap<QString, QString>)));
    connect(ac, SIGNAL(radioTuneData(QMap<QString,QString>)), this, SLOT(radioTuneReplied(QMap<QString,QString>)));
}

void LastGoWH::authGetMobileSessionReplied(QMap<QString, QString> reply)
{
    if(reply["status"] == "ok")
    {
        sk = reply["key"];

        if(reply["subscriber"] == "0")
        {
            isPlaying = false;
            ui->statusLabel->setText("Last.fm radio is only available to subscribers.");
            ui->playStopButton->setEnabled(false);
        }
        else
        {
            if(isTuning)
            {
                isGettingPlayList = true;
                isTuning = false;
                ac->radioTune(radioUrl, sk);
            }
            else
                ac->radioGetPlaylist(sk);
        }
    }
    else if(reply["status"] == "failed")
    {
        ui->statusLabel->setText(reply["message"]);
        qDebug() << reply["error"];
        qDebug() << reply["message"];
    }
}

void LastGoWH::setPlayButtonToPlay()
{
    ui->playStopButton->setIcon(QIcon(":/images/images/media-playback-start.png"));
    ui->nextTrackButton->setEnabled(false);
}

void LastGoWH::setPlayButtonToStop()
{
    ui->playStopButton->setIcon(QIcon(":/images/images/media-playback-stop.png"));
    ui->nextTrackButton->setEnabled(true);
}

void LastGoWH::playTrack(int i)
{
    setPlayButtonToStop();

    if(playlistTracks[i]->Image == "")
        ui->trackImage->setPixmap(QPixmap(":/images/images/vinile.jpg"));
    else
    {
        manager = new QNetworkAccessManager(this);
        connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(imageRequestReplied(QNetworkReply*)));

        QNetworkRequest request;
        request.setUrl(playlistTracks[i]->Image);
        request.setRawHeader("User-Agent", "LastGo 1.0");
        manager->get(request);
    }

    ui->titleDataLabel->setText(playlistTracks[i]->Title);
    ui->albumDataLabel->setText(playlistTracks[i]->Album);
    ui->artistDataLabel->setText(playlistTracks[i]->Creator);
    ui->progressBar->setMaximum(playlistTracks[i]->Duration.toInt());

    playlist->clear();
    playlist->addMedia(QUrl(playlistTracks[i]->Location));

    player->play();
    ui->nextTrackButton->setEnabled(true);
}

void LastGoWH::nextButtonClicked()
{
    ui->nextTrackButton->setEnabled(false);
    playListIndex += 1;

    if(playListIndex < playlistTracks.count())
    {
        playTrack(playListIndex);
    }
    else
    {
        ac->radioGetPlaylist(sk);
    }
}

void LastGoWH::playerStateChanged(QMediaPlayer::State state)
{
    // Automatically plays next song
    if(isPlaying)
        if(state == QMediaPlayer::StoppedState)
        {
            nextButtonClicked();
        }
}

void LastGoWH::trackPositionChanged(qint64 position)
{
    ui->progressBar->setValue(position);
}

void LastGoWH::imageRequestReplied(QNetworkReply *reply)
{
    QImage image;
    QByteArray data = reply->readAll();
    image.loadFromData(data);
    ui->trackImage->setPixmap(QPixmap::fromImage(image));
}

void LastGoWH::selectRadio()
{
    if(username != "")
    {
        RadioList dialog(username, this);
        dialog.setWindowTitle("Radio List");
        dialog.showNormal();
        dialog.exec();

        if(dialog.radioUrl != "")
        {
            if(sk == "")
            {
                isTuning = true;
                radioUrl = dialog.radioUrl;
                ac->authGetMobileSession(username, password);
                return;
            }
            else
            {
                ac->radioTune(dialog.radioUrl, sk);
            }
        }
    }
}

void LastGoWH::radioTuneReplied(QMap<QString, QString> reply)
{
    if(reply["status"] == "ok")
    {
        qDebug() << reply["status"];

        if(isGettingPlayList)
        {
            isGettingPlayList = false;
            ac->radioGetPlaylist(sk);
        }
    }
    else if(reply["status"] == "failed")
    {
        qDebug() << reply["error"];
        qDebug() << reply["message"];
        ui->statusLabel->setText("Error: " + reply["error"]);
    }
}

void LastGoWH::showAbout()
{
    AboutDialog dialog(this);
    dialog.setWindowTitle("About");
    dialog.showNormal();
    dialog.exec();
}
