/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	Jacob's Matrix Development
 *
 * NAME:        main.cpp ( Jacob's Matrix, C++ )
 *
 * COMMENTS:
 */

#include <libtcod.hpp>
#include <stdio.h>
#include <time.h>

#ifdef WIN32
#include <windows.h>
#include <process.h>
#define usleep(x) Sleep((x)*1000)
#pragma comment(lib, "SDL.lib")
#pragma comment(lib, "SDLmain.lib")

// I really hate windows.
#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif

#else
#include <unistd.h>
#endif

#include <fstream>
using namespace std;

int		 glbLevelStartMS = -1;

#include "rand.h"
#include "gfxengine.h"
#include "config.h"

#include <SDL.h>
#include <SDL_mixer.h>

Mix_Music		*glbTunes = 0;
bool			 glbMusicActive = false;

// Cannot be bool!
// libtcod does not accept bools properly.

volatile int		glbItemCache = -1;

void endOfMusic()
{
    // Keep playing.
    // glbMusicActive = false;
}

void
setMusicVolume(int volume)
{
    volume = BOUND(volume, 0, 10);

    glbConfig->myMusicVolume = volume;

    Mix_VolumeMusic((MIX_MAX_VOLUME * volume) / 10);
}

void
stopMusic(bool atoncedamnit = false)
{
    if (glbMusicActive)
    {
	if (atoncedamnit)
	    Mix_HaltMusic();
	else
	    Mix_FadeOutMusic(2000);
	glbMusicActive = false;
    }
}

void
startMusic()
{
    if (glbMusicActive)
	stopMusic();

    if (!glbTunes)
	glbTunes = Mix_LoadMUS(glbConfig->musicFile());
    if (!glbTunes)
    {
	printf("Failed to load %s, error %s\n", 
		glbConfig->musicFile(), 
		Mix_GetError());
    }
    else
    {
	glbMusicActive = true;
	glbLevelStartMS = TCOD_sys_elapsed_milli();

	if (glbConfig->musicEnable())
	{
	    Mix_PlayMusic(glbTunes, -1);		// Infinite
	    Mix_HookMusicFinished(endOfMusic);

	    setMusicVolume(glbConfig->myMusicVolume);
	}
    }
}


#ifdef main
#undef main
#endif

#include "mob.h"
#include "map.h"
#include "text.h"
#include "speed.h"
#include "display.h"
#include "engine.h"
#include "panel.h"
#include "msg.h"
#include "firefly.h"
#include "item.h"
#include "chooser.h"

PANEL		*glbMessagePanel = 0;
DISPLAY		*glbDisplay = 0;
MAP		*glbMap = 0;
ENGINE		*glbEngine = 0;
PANEL		*glbPopUp = 0;
PANEL		*glbJacob = 0;
PANEL		*glbWeaponInfo = 0;
PANEL		*glbVictoryInfo = 0;
bool		 glbPopUpActive = false;
bool		 glbRoleActive = false;
FIREFLY		*glbHealthFire = 0;
int		 glbLevel = 1;
int		 glbMaxLevel = 0;
PTRLIST<int>	 glbLevelTimes;
bool		 glbVeryFirstRun = true;

CHOOSER		*glbLevelBuildStatus = 0;

CHOOSER		*glbChooser = 0;
bool		 glbChooserActive = false;

int		 glbInvertX = -1, glbInvertY = -1;
int		 glbMeditateX = -1, glbMeditateY = -1;

#define DEATHTIME 15000


BUF
mstotime(int ms)
{
    int		sec, min;
    BUF		buf;

    sec = ms / 1000;
    ms -= sec * 1000;
    min = sec / 60;
    sec -= min * 60;

    if (min)
	buf.sprintf("%dm%d.%03ds", min, sec, ms);
    else if (sec)
	buf.sprintf("%d.%03ds", sec, ms);
    else
	buf.sprintf("0.%03ds", ms);

    return buf;
}

// Should call this in all of our loops.
void
redrawWorld()
{
    static MAP	*centermap = 0;
    static POS	 mapcenter;
    bool	 isblind = false;
    bool	 isvictory = false;

    gfx_updatepulsetime();

    if (glbMap)
	glbMap->decRef();
    glbMap = glbEngine->copyMap();

    glbWeaponInfo->clear();

    glbMeditateX = -1;
    glbMeditateY = -1;
    int			avataralive = 0;
    if (glbMap && glbMap->avatar())
    {
	MOB		*avatar = glbMap->avatar();
	int		 timems;

	avataralive = avatar->alive();

	if (avatar->alive() && avatar->isMeditating())
	{
	    glbMeditateX = glbDisplay->x() + glbDisplay->width()/2;
	    glbMeditateY = glbDisplay->y() + glbDisplay->height()/2;
	}

	timems = TCOD_sys_elapsed_milli();

	if (centermap)
	    centermap->decRef();
	centermap = glbMap;
	centermap->incRef();
	mapcenter = avatar->meditatePos();

	isblind = avatar->hasItem(ITEM_BLIND);

	if (avatar->hasItem(ITEM_INVULNERABLE))
	    glbHealthFire->setFireType(FIRE_MONO);
	else
	    glbHealthFire->setFireType(FIRE_ICE);

	glbHealthFire->setRatioLiving(avatar->getHP() / (float) avatar->defn().max_hp);

	BUF		buf;
	ITEM		*item;

	item = avatar->lookupWeapon();
	glbWeaponInfo->setTextAttr(ATTR_WHITE);
	if (item)
	{
	    glbWeaponInfo->appendText(gram_capitalize(item->getName()));
	    glbWeaponInfo->newLine();
	    glbWeaponInfo->setTextAttr(ATTR_NORMAL);
	    glbWeaponInfo->appendText(item->getDetailedDescription());
	}
	else
	    glbWeaponInfo->appendText("No weapon\n");
	glbWeaponInfo->newLine();

	glbWeaponInfo->setTextAttr(ATTR_WHITE);
	item = avatar->lookupWand();
	if (item)
	{
	    glbWeaponInfo->appendText(gram_capitalize(item->getName()));
	    glbWeaponInfo->newLine();
	    glbWeaponInfo->setTextAttr(ATTR_NORMAL);
	    glbWeaponInfo->appendText(item->getDetailedDescription());
	}
	else
	    glbWeaponInfo->appendText("No ranged weapon");
	glbWeaponInfo->newLine();

	isvictory = avatar->pos().tile() == TILE_UPSTAIRS;

	if (isvictory)
	{
	    glbVictoryInfo->clear();
	    glbVictoryInfo->appendText("Press 'v' to leave the dungeon!");
	}
    }
    
    glbDisplay->display(mapcenter, isblind);
    glbMessagePanel->redraw();

    glbJacob->clear();
    glbJacob->setTextAttr(ATTR_WHITE);
    BUF		buf;

    buf.sprintf(" Depth: %d\n\n Kobolds: %d", 
		glbMap ? glbMap->getDepth() : 0,
		glbMap ? glbMap->getNumMobs() - avataralive : 0);

    glbJacob->appendText(buf);

    glbJacob->redraw();

    glbWeaponInfo->redraw();

    if (isvictory)
	glbVictoryInfo->redraw();

    {
	FIRETEX		*tex = glbHealthFire->getTex();
	tex->redraw(0, 0);
	tex->decRef();
    }

    if (glbMeditateX >= 0 && glbMeditateY >= 0)
    {
	gfx_printattrback(glbMeditateX, glbMeditateY, ATTR_AVATARMEDITATE);
    }

    if (glbInvertX >= 0 && glbInvertY >= 0)
    {
	gfx_printattr(glbInvertX, glbInvertY, ATTR_HILITE);
    }


    if (glbItemCache >= 0)
    {
	glbLevelBuildStatus->clear();

	glbLevelBuildStatus->appendChoice("   Building Distance Caches   ", glbItemCache);
	glbLevelBuildStatus->redraw();
    }

    if (glbChooserActive)
    {
	glbChooser->redraw();
    }

    if (glbPopUpActive)
	glbPopUp->redraw();

    TCODConsole::flush();
}

// Pops up the given text onto the screen.
// The key used to dismiss the text is the return code.
int
popupText(const char *buf, int delayms = 0)
{
    int		key = 0;
    int		startms = TCOD_sys_elapsed_milli();

    glbPopUp->clear();
    glbPopUp->resize(50, 48);
    glbPopUp->move(15, 1);

    // A header matches the footer.
    glbPopUp->appendText("\n");

    // Must make active first because append text may trigger a 
    // redraw if it is long!
    glbPopUpActive = true;
    glbPopUp->appendText(buf);

    // In case we drew due to too long text
    glbPopUp->erase();

    // Determine the size of the popup and resize.
    glbPopUp->shrinkToFit();

    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();
	key = gfx_getKey(false);
	if (key)
	{
	    // Don't let them abort too soon.
	    if (((int)TCOD_sys_elapsed_milli() - startms) >= delayms)
		break;
	}
    }

    glbPopUpActive = false;
    glbPopUp->erase();

    glbPopUp->resize(50, 30);
    glbPopUp->move(15, 10);

    redrawWorld();

    return key;
}

int
popupText(BUF buf, int delayms = 0)
{
    return popupText(buf.buffer(), delayms);
}

//
// Freezes the game until a key is pressed.
// If key is not direction, false returned and it is eaten.
// else true returned with dx/dy set.  Can be 0,0.
bool
awaitDirection(int &dx, int &dy)
{
    int			key;

    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();

	key = gfx_getKey(false);

	if (gfx_cookDir(key, dx, dy))
	{
	    return true;
	}
	if (key)
	    return false;
    }

    return false;
}

void
doExamine()
{
    int		x = glbDisplay->x() + glbDisplay->width()/2;
    int		y = glbDisplay->y() + glbDisplay->height()/2;
    int		key;
    int		dx, dy;
    POS		p;
    BUF		buf;
    bool	blind;

    glbInvertX = x;
    glbInvertY = y;
    blind = false;
    if (glbMap && glbMap->avatar())
	blind = glbMap->avatar()->hasItem(ITEM_BLIND);
    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();

	key = gfx_getKey(false);

	if (gfx_cookDir(key, dx, dy))
	{
	    // Space we want to select.
	    if (!dx && !dy)
		break;
	    x += dx;
	    y += dy;
	    x = BOUND(x, glbDisplay->x(), glbDisplay->x()+glbDisplay->width()-1);
	    y = BOUND(y, glbDisplay->y(), glbDisplay->y()+glbDisplay->height()-1);

	    glbInvertX = x;
	    glbInvertY = y;
	    p = glbDisplay->lookup(x, y);

	    if (p.tile() != TILE_INVALID && p.isFOV())
	    {
		p.describeSquare(blind);
	    }

	    msg_newturn();
	}

	// Any other key stops the look.
	if (key)
	    break;
    }

    // Check if we left on a mob.
    // Must re look up as displayWorld may have updated the map.
    p = glbDisplay->lookup(x, y);
    if (p.mob())
    {
	popupText(text_lookup("mob", p.mob()->getRawName()));
    }
    else if (p.item())
    {
	popupText(text_lookup("item", p.item()->getRawName()));
    }

    glbInvertX = -1;
    glbInvertY = -1;
}

void
buildAction(ITEM *item, ACTION_NAMES *actions)
{
    glbChooser->clear();

    glbChooser->setTextAttr(ATTR_NORMAL);
    if (!item)
    {
	glbChooser->appendChoice("Nothing to do with nothing.");
	*actions++ = ACTION_NONE;
    }
    else
    {
	glbChooser->appendChoice("[ ] Do nothing");
	*actions++ = ACTION_NONE;

	glbChooser->appendChoice("[X] Examine");
	*actions++ = ACTION_EXAMINE;

	if (item->isRanged())
	{
	    glbChooser->appendChoice("[B] Break");
	    *actions++ = ACTION_BREAK;
	}
	if (item->isPotion())
	{
	    glbChooser->appendChoice("[Q] Quaff");
	    *actions++ = ACTION_QUAFF;
	}
	if (item->isFood())
	{
	    glbChooser->appendChoice("[E] Eat");
	    *actions++ = ACTION_EAT;
	}
	if (item->getDefinition() == ITEM_SEARCHRING)
	{
	    glbChooser->appendChoice("[S] Search");
	    *actions++ = ACTION_SEARCH;
	}
	if (item->getDefinition() == ITEM_AMULETTRUESIGHT)
	{
	    glbChooser->appendChoice("[M] Meditate");
	    *actions++ = ACTION_MEDITATE;
	}
	if (!item->defn().isflag)
	{
	    glbChooser->appendChoice("[D] Drop");
	    *actions++ = ACTION_DROP;
	}
    }
    glbChooser->setChoice(0);

    glbChooserActive = true;
}

bool
useItem(MOB *mob, ITEM *item, int itemno)
{
    ACTION_NAMES		actions[NUM_ACTIONS+1];
    bool			done = false;
    int				key;

    buildAction(item, actions);
    glbChooserActive = true;

    // Run the chooser...
    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();
	key = gfx_getKey(false);

	glbChooser->processKey(key);

	if (key)
	{
	    if (key == 'd' || key == 'D')
	    {
		glbEngine->queue().append(COMMAND(ACTION_DROP, itemno));
		done = true;
		break;
	    }
	    else if (key == 'q' || key == 'Q')
	    {
		glbEngine->queue().append(COMMAND(ACTION_QUAFF, itemno));
		done = true;
		break;
	    }
	    else if (key == 'm' || key == 'M')
	    {
		glbEngine->queue().append(COMMAND(ACTION_MEDITATE));
		done = true;
		break;
	    }
	    else if (key == 's' || key == 'S')
	    {
		glbEngine->queue().append(COMMAND(ACTION_SEARCH));
		done = true;
		break;
	    }
	    else if (key == 'e' || key == 'E')
	    {
		glbEngine->queue().append(COMMAND(ACTION_EAT, itemno));
		done = true;
		break;
	    }
	    else if (key == 'b' || key == 'B')
	    {
		glbEngine->queue().append(COMMAND(ACTION_BREAK, itemno));
		done = true;
		break;
	    }
	    else if (key == 'x' || key == 'X')
	    {
		popupText(text_lookup("item", item->getRawName()));
		done = false;
		break;
	    }
	    // User selects this?
	    else if (key == '\n' || key == ' ')
	    {
		ACTION_NAMES	action;

		action = actions[glbChooser->getChoice()];
		switch (action)
		{
		    case ACTION_DROP:
			glbEngine->queue().append(COMMAND(ACTION_DROP, itemno));
			done = true;
			break;
		    case ACTION_QUAFF:
			glbEngine->queue().append(COMMAND(ACTION_QUAFF, itemno));
			done = true;
			break;
		    case ACTION_BREAK:
			glbEngine->queue().append(COMMAND(ACTION_BREAK, itemno));
			done = true;
			break;
		    case ACTION_EAT:
			glbEngine->queue().append(COMMAND(ACTION_EAT, itemno));
			done = true;
			break;
		    case ACTION_SEARCH:
			glbEngine->queue().append(COMMAND(ACTION_SEARCH, itemno));
			done = true;
			break;
		    case ACTION_MEDITATE:
			glbEngine->queue().append(COMMAND(ACTION_MEDITATE, itemno));
			done = true;
			break;
		    case ACTION_EXAMINE:
			popupText(text_lookup("item", item->getRawName()));
			done = false;
			break;
		}
		break;
	    }
	    else if (key == '\x1b')
	    {
		// Esc on options is to go back to play.
		// Play...
		break;
	    }
	    else
	    {
		// Ignore other options.
	    }
	}
    }
    glbChooserActive = false;

    return done;
}

void
buildInventory(MOB *mob)
{
    glbChooser->clear();

    glbChooser->setTextAttr(ATTR_NORMAL);
    if (!mob)
    {
	glbChooser->appendChoice("The dead own nothing.");
    }
    else
    {
	if (!mob->inventory().entries())
	{
	    glbChooser->appendChoice("nothing");
	}
	else
	{
	    for (int i = 0; i < mob->inventory().entries(); i++)
	    {
		glbChooser->appendChoice(mob->inventory()(i)->getName());
	    }
	}
    }
    glbChooser->setChoice(0);

    glbChooserActive = true;
}

void
inventoryMenu()
{
    MOB		*avatar = (glbMap ? glbMap->avatar() : 0);
    int		key;
    buildInventory(avatar);

    // Run the chooser...
    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();
	key = gfx_getKey(false);

	glbChooser->processKey(key);

	if (key)
	{
	    int		itemno = glbChooser->getChoice();
	    bool	done = false;
	    // User selects this?
	    if (key == '\n' || key == ' ')
	    {
		done = true;

		if (itemno >= 0 && itemno < avatar->inventory().entries())
		{
		    done = useItem(avatar, avatar->inventory()(itemno), itemno);
		}
		// Finish the inventory menu.
		if (done)
		{
		    break;
		}
		else
		{
		    buildInventory(avatar);
		}
	    }
	    else if (key == '\x1b' || key == 'i')
	    {
		// Esc on options is to go back to play.
		// Play...
		break;
	    }
	    else if (itemno >= 0 && itemno < avatar->inventory().entries())
	    {
		if (key == 'd' || key == 'D')
		{
		    glbEngine->queue().append(COMMAND(ACTION_DROP, itemno));
		    done = true;
		}
		else if (key == 'q' || key == 'Q')
		{
		    glbEngine->queue().append(COMMAND(ACTION_QUAFF, itemno));
		    done = true;
		}
		else if (key == 'e' || key == 'E')
		{
		    glbEngine->queue().append(COMMAND(ACTION_EAT, itemno));
		    done = true;
		}
		else if (key == 'b' || key == 'B')
		{
		    glbEngine->queue().append(COMMAND(ACTION_BREAK, itemno));
		    done = true;
		}
		else if (key == 'm' || key == 'M')
		{
		    glbEngine->queue().append(COMMAND(ACTION_MEDITATE));
		    done = true;
		}
		else if (key == 's' || key == 'S')
		{
		    glbEngine->queue().append(COMMAND(ACTION_SEARCH));
		    done = true;
		}
		else if (key == 'x' || key == 'X')
		{
		    if (itemno >= 0 && itemno < avatar->inventory().entries())
		    {
			popupText(text_lookup("item", avatar->inventory()(itemno)->getRawName()));
		    }
		}
	    }
	    else
	    {
		// Ignore other options.
	    }

	    if (done)
		break;
	}
    }
    glbChooserActive = false;
}

void
buildOptionsMenu(OPTION_NAMES d)
{
    OPTION_NAMES	option;
    glbChooser->clear();

    glbChooser->setTextAttr(ATTR_NORMAL);
    FOREACH_OPTION(option)
    {
	glbChooser->appendChoice(glb_optiondefs[option].name);
    }
    glbChooser->setChoice(d);

    glbChooserActive = true;
}

bool
optionsMenu()
{
    int			key;
    bool		done = false;

    buildOptionsMenu(OPTION_INSTRUCTIONS);

    // Run the chooser...
    while (!TCODConsole::isWindowClosed())
    {
	redrawWorld();
	key = gfx_getKey(false);

	glbChooser->processKey(key);

	if (key)
	{
	    // User selects this?
	    if (key == '\n' || key == ' ')
	    {
		if (glbChooser->getChoice() == OPTION_INSTRUCTIONS)
		{
		    // Instructions.
		    popupText(text_lookup("game", "help"));
		}
		else if (glbChooser->getChoice() == OPTION_PLAY)
		{
		    // Play...
		    break;
		}
		else if (glbChooser->getChoice() == OPTION_VOLUME)
		{
		    int			i;
		    BUF			buf;
		    // Volume
		    glbChooser->clear();

		    for (i = 0; i <= 10; i++)
		    {
			buf.sprintf("%3d%% Volume", (10 - i) * 10);

			glbChooser->appendChoice(buf);
		    }
		    glbChooser->setChoice(10 - glbConfig->myMusicVolume);

		    while (!TCODConsole::isWindowClosed())
		    {
			redrawWorld();
			key = gfx_getKey(false);

			glbChooser->processKey(key);

			setMusicVolume(10 - glbChooser->getChoice());
			if (key)
			{
			    break;
			}
		    }
		    buildOptionsMenu(OPTION_VOLUME);
		}
		else if (glbChooser->getChoice() == OPTION_SPARKCOUNT)
		{
		    // Flame Quality
		    int			i;
		    BUF			buf;
		    const char 		*quality[6] =
		    {
			"No Sparks",
			"Low",
			"Middle",
			"High",
			"Extreme",
			"Just Silly"
		    };

		    int			counts[6] =
		    {	0,
			65,		// one per hp
			65*3,
			500,
			5000,
			50000
		    };

		    // Flame Quality
		    glbChooser->clear();

		    for (i = 0; i < 6; i++)
		    {
			glbChooser->appendChoice(quality[i]);
		    }

		    for (i = 0; i < 6; i++)
		    {
			if (glbHealthFire->particleCount() < counts[i])
			    break;
		    }
		    i--;
		    i = BOUND(i, 0, 6);
		    glbChooser->setChoice(i);

		    while (!TCODConsole::isWindowClosed())
		    {
			bool	waskey = false;
			redrawWorld();
			key = gfx_getKey(false);

			if (key)
			    waskey = true;

			glbChooser->processKey(key);

			if (key)
			{
			    break;
			}
			else if (waskey)
			{
			    // New selection likely, so update
			    glbConfig->mySparkCount = counts[glbChooser->getChoice()];
			    glbHealthFire->setParticleCount(glbConfig->mySparkCount);
			}
		    }
		    buildOptionsMenu(OPTION_SPARKCOUNT);
		}
		else if (glbChooser->getChoice() == OPTION_FULLSCREEN)
		{
		    glbConfig->myFullScreen = !glbConfig->myFullScreen;
		    // This is intentionally unrolled to work around a
		    // bool/int problem in libtcod
		    if (glbConfig->myFullScreen)
			TCODConsole::setFullscreen(true);
		    else
			TCODConsole::setFullscreen(false);
		}
		else if (glbChooser->getChoice() == OPTION_QUIT)
		{
		    // Quit
		    done = true;
		    break;
		}
	    }
	    else if (key == '\x1b')
	    {
		// Esc on options is to go back to play.
		// Play...
		break;
	    }
	    else
	    {
		// Ignore other options.
	    }
	}
    }
    glbChooserActive = false;

    return done;
}

bool
reloadLevel(bool alreadybuilt)
{
    if (TCODConsole::isWindowClosed())
	return true;

    glbVeryFirstRun = false;

    glbLevel = 1;
    glbMessagePanel->clear();
    if (!alreadybuilt)
	glbEngine->queue().append(COMMAND(ACTION_RESTART, glbLevel));

    startMusic();

    // Ideally we'd suppress this if you load from disk
    // but we don't know if we loaded from disk yet.
    popupText(text_lookup("welcome", "You1"));
    popupText(text_lookup("welcome", "You2"));

    // Wait for the map to finish.
    if (glbEngine->awaitRebuild())
    {
	popupText(text_lookup("welcome", "Back"));
    }
    else
    {
	popupText(text_lookup("welcome", "You3"));
    }

    // Redrawing the world updates our glbMap so everyone will see it.
    redrawWorld();

    return false;
}

// Let the avatar watch the end...
void
deathCoolDown()
{
    int		coolstartms = TCOD_sys_elapsed_milli();
    int		lastenginems = coolstartms;
    int		curtime;
    bool	done = false;
    double	ratio;

    while (!done)
    {
	curtime = TCOD_sys_elapsed_milli();

	// Queue wait events, one per 100 ms.
	while (lastenginems + 200 < curtime)
	{
	    glbEngine->queue().append(COMMAND(ACTION_WAIT));
	    lastenginems += 200;
	}

	// Set our fade.
	ratio = (curtime - coolstartms) / 5000.;
	ratio = 1.0 - ratio;
	if (ratio < 0)
	{
	    done = true;
	    ratio = 0;
	}

	TCOD_console_set_fade( (u8) (255 * ratio), TCOD_black);

	// Keep updating
	redrawWorld();
    }

    gfx_clearKeyBuf();

    TCOD_console_set_fade(255,  TCOD_black);
}

void
shutdownEverything()
{
    glbConfig->save("../kobold.cfg");

    // Save the game.  A no-op if dead.
    glbEngine->queue().append(COMMAND(ACTION_SAVE));
    glbEngine->awaitSave();

    stopMusic(true);
    if (glbTunes)
	Mix_FreeMusic(glbTunes);

    gfx_shutdown();

    SDL_QuitSubSystem(SDL_INIT_AUDIO);

    SDL_Quit();
}


#ifdef WIN32
int WINAPI
WinMain(HINSTANCE hINstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCMdSHow)
#else
int 
main(int argc, char **argv)
#endif
{
    bool		done = false;

    // Clamp frame rate.
    TCOD_sys_set_fps(60);

    glbConfig = new CONFIG();
    glbConfig->load("../kobold.cfg");

    // Dear Microsoft,
    // The following code in optimized is both a waste and seems designed
    // to ensure that if we call a bool where the callee pretends it is
    // an int we'll end up with garbage data.
    //
    // 0040BF4C  movzx       eax,byte ptr [eax+10h] 

    // TCODConsole::initRoot(SCR_WIDTH, SCR_HEIGHT, "Smart Kobold", myFullScreen);
    // 0040BF50  xor         ebp,ebp 
    // 0040BF52  cmp         eax,ebp 
    // 0040BF54  setne       cl   
    // 0040BF57  mov         dword ptr [esp+28h],eax 
    // 0040BF5B  push        ecx  

    // My work around is to constantify the fullscreen and hope that
    // the compiler doesn't catch on.
    if (glbConfig->myFullScreen)
	TCODConsole::initRoot(SCR_WIDTH, SCR_HEIGHT, "Smart Kobold", true, TCOD_RENDERER_SDL);
    else
	TCODConsole::initRoot(SCR_WIDTH, SCR_HEIGHT, "Smart Kobold", false, TCOD_RENDERER_SDL);

    // TCOD doesn't do audio.
    SDL_InitSubSystem(SDL_INIT_AUDIO);
    if (Mix_OpenAudio(22050, AUDIO_S16, 2, 4096))
    {
       printf("Unable to initialize audio: %s\n", Mix_GetError());
       // exit(1); //Not critical
    }

    SDL_InitSubSystem(SDL_INIT_JOYSTICK);

    setMusicVolume(glbConfig->musicVolume());

    rand_setseed((long) time(0));

    gfx_init();
    MAP::init();

    text_init();
    spd_init();

    glbDisplay = new DISPLAY(0, 0, 100, 40);

    glbMessagePanel = new PANEL(40, 10);
    msg_registerpanel(glbMessagePanel);
    glbMessagePanel->move(20, 40);

    glbPopUp = new PANEL(50, 30);
    glbPopUp->move(15, 10);
    glbPopUp->setBorder(true, ' ', ATTR_BORDER);
    // And so the mispelling is propagated...
    glbPopUp->setRigthMargin(1);
    glbPopUp->setIndent(1);

    glbJacob = new PANEL(15, 6);
    glbJacob->move(65, 1);
    glbJacob->setTransparent(true);

    glbWeaponInfo = new PANEL(20, 16);
    glbWeaponInfo->move(1, 1);
    glbWeaponInfo->setTransparent(true);

    glbVictoryInfo = new PANEL(26, 2);
    glbVictoryInfo->move(28, 7);
    glbVictoryInfo->setIndent(2);
    glbVictoryInfo->setBorder(true, ' ', ATTR_VICTORYBORDER);

    glbLevel = 1;

    glbChooser = new CHOOSER();
    glbChooser->move(40, 25, CHOOSER::JUSTCENTER, CHOOSER::JUSTCENTER);
    glbChooser->setBorder(true, ' ', ATTR_BORDER);
    glbChooser->setIndent(1);
    glbChooser->setRigthMargin(1);

    glbLevelBuildStatus = new CHOOSER();
    glbLevelBuildStatus->move(40, 25, CHOOSER::JUSTCENTER, CHOOSER::JUSTCENTER);
    glbLevelBuildStatus->setBorder(true, ' ', ATTR_BORDER);
    glbLevelBuildStatus->setIndent(1);
    glbLevelBuildStatus->setRigthMargin(1);
    glbLevelBuildStatus->setAttr(ATTR_NORMAL, ATTR_HILITE, ATTR_HILITE);

    glbEngine = new ENGINE(glbDisplay);

    glbHealthFire = new FIREFLY(glbConfig->sparkCount(), 20, 50, FIRE_ICE);

    // Run our first level immediately so it can be built or 
    // loaded from disk while people wait.
    glbEngine->queue().append(COMMAND(ACTION_RESTART, 1));

    done = optionsMenu();
    if (done)
    {
	shutdownEverything();
	return 0;
    }

    done = reloadLevel(true);
    if (done)
    {
	shutdownEverything();
	return 0;
    }

    do
    {
	int		key;
	int		dx, dy;

	redrawWorld();

	if (glbMap && glbMap->avatar() &&
	    !glbMap->avatar()->alive())
	{
	    // You have died...
	    deathCoolDown();

	    popupText(text_lookup("game", "lose"));

	    stopMusic();

	    done = optionsMenu();
	    if (done)
	    {
		shutdownEverything();
		return 0;
	    }
	    // Don't prep a reload.
	    done = reloadLevel(false);
	}

	key = gfx_getKey(false);

	if (gfx_cookDir(key, dx, dy))
	{
	    // Direction.
	    glbEngine->queue().append(COMMAND(ACTION_BUMP, dx, dy));
	}

	switch (key)
	{
	    case 'Q':
		done = true;
		break;

	    case 'R':
		done = reloadLevel(false);
		break;

	    case '/':
	    case 'e':
		glbEngine->queue().append(COMMAND(ACTION_ROTATE, 1));
		break;
	    case '*':
	    case 'r':
		glbEngine->queue().append(COMMAND(ACTION_ROTATE, 3));
		break;

	    case 'm':
		glbEngine->queue().append(COMMAND(ACTION_MEDITATE));
		break;

	    case 's':
		glbEngine->queue().append(COMMAND(ACTION_SEARCH));
		break;

	    case 'i':
		inventoryMenu();
		break;

	    case 'f':
	    case '0':
		msg_report("Cast spell in what direction?  ");
		if (awaitDirection(dx, dy) && (dx || dy))
		{
		    msg_report(rand_dirtoname(dx, dy));
		    msg_newturn();
		    glbEngine->queue().append(COMMAND(ACTION_FIRE, dx, dy));
		}
		else
		{
		    msg_report("Cancelled.  ");
		    msg_newturn();
		}
		break;

	    case 'a':
		popupText(text_lookup("game", "about"));
		break;

	    case 'z':
	    {
		popupText(text_lookup("welcome", "You1"));
		popupText(text_lookup("welcome", "You2"));
		popupText(text_lookup("welcome", "You3"));
		break;
	    }

	    case 'P':
		glbConfig->myFullScreen = !glbConfig->myFullScreen;
		// This is intentionally unrolled to work around a
		// bool/int problem in libtcod
		if (glbConfig->myFullScreen)
		    TCODConsole::setFullscreen(true);
		else
		    TCODConsole::setFullscreen(false);
		break;
	
	    case GFX_KEYF1:
	    case '?':
		popupText(text_lookup("game", "help"));
		break;

	    case 'v':
		if (glbMap && glbMap->avatar())
		{
		    if (glbMap->avatar()->alive() &&
			glbMap->avatar()->pos().tile() == TILE_UPSTAIRS)
		    {
			BUF		victitle;
			BUF		maintext;
			ITEM		*gold;
			int		numgold, profit;
			MOB		*avatar = glbMap->avatar();

			gold = avatar->lookupItem(ITEM_GOLD);

			numgold = gold ? gold->getStackCount() : 0;
			profit = numgold - 5893;

			if (profit > 0)
			    victitle.sprintf("victoryprofit");
			else if (profit < 0)
			    victitle.sprintf("victoryloss");
			else if (avatar->isFullHealth())
			    victitle.sprintf("victoryeven");
			else
			    victitle.sprintf("victoryevenhurt");

			if (profit < 0)
			    maintext.sprintf("You climb up the stairs into sunlight, posting a loss of %d gold.\n", -profit);
			else if (profit > 0)
			    maintext.sprintf("You climb up the stairs into sunlight, netting a profit of %d gold.\n", profit);
			else
			    maintext.sprintf("You climb up the stairs into sunlight with no more gold than you entered with.\n");
			maintext.strcat("\n");
			maintext.strcat(text_lookup("game", victitle));
			popupText(maintext);

			stopMusic();
			done = optionsMenu();
			if (!done)
			    done = reloadLevel(false);
		    }
		    else
		    {
			msg_report("You must be on the staircase, <, to exit.  ");
		    }
		}
		break;

	    case 'x':
		doExamine();
		break;

#if 0
	    case 'W':
		glbEngine->queue().append(COMMAND(ACTION_CREATEITEM, ITEM_BOW));

		break;
#endif

	    case '\x1b':
		// If meditating, abort.
		if (glbMap && glbMap->avatar() && 
		    glbMap->avatar()->alive() && glbMap->avatar()->isMeditating())
		{
		    glbEngine->queue().append(COMMAND(ACTION_MEDITATE));
		    break;
		}
		// FALL THROUGH
	    case 'O':
		done = optionsMenu();
		break;
	}
    } while (!done && !TCODConsole::isWindowClosed());

    shutdownEverything();

    return 0;
}
