using Gtk;
using Gdk;
using Hildon;

public class Discovery
{
	private HashTable<string,string> connection;
	private Knots knots;
	private string[] servers;
	private bool cache;
	private string last_url;
	private int timeout;
	private int default_timeout;
	private int last_file_size;
	private string _username;
	private string _password;
	
	public Discovery(Knots knots)
	{
		this.knots = knots;
		this.cache = true;
		this.timeout = 30;
		update_timeout();
	}
	
	public void update_timeout()
	{
		this.default_timeout = knots.get_settings().get_value_default_to("default_timeout", "30").to_int();
	}
	
	public int get_last_file_size()
	{
		return last_file_size;
	}
	
	public void set_url_cache(bool cache)
	{
		this.cache = cache;
	}
	
	public int server_count()
	{
		return servers.length;	
	}
	
	public void select_server(int selected)
	{
		if (selected < server_count())
			load_connection(servers[selected], false, false);
	}
	
	public string serverlist_xml()
	{
		string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?><root><items>";
		int index = 0;
		foreach(string server in servers)
		{
			string[] tokens = server.split("/");
			if (tokens[0].strip() == "")
				tokens[0] = tokens[3];
			xml += "<item><server>" + tokens[0] + "</server><id>" + index.to_string() + "</id><type>server</type></item>";
			index += 1;
		}
		return xml + "</items><pages><current>1</current><total>1</total></pages></root>";
	}
	
	public bool load_connection(string? address, bool save, bool test)
	{
		if (address != null && address.str("/") != null)
		{
			string[] tokens = address.split("/");
			string[] keys = {"name", "ssl", "port", "address", "username", "password"};
			if (tokens.length >= 4)
			{
				connection = new HashTable<string,string>(str_hash, str_equal);
				for (int i = 0; i < 6; i++)
				{
					if (i < tokens.length)
					{
						connection.insert(keys[i], tokens[i]);
					}
				}
				if (!test || test_connection())
				{
					if (save)
						knots.get_settings().set_value("last_server", address);
					return true;
				}
				else
				{
					connection = null;
					return false;
				}
			}
		}
		return false;
	}
	
	public void add_server(string server)
	{
		if (!server_added(server))
			servers += server;
	}
	
	public bool server_added(string server)
	{
		if (server != null)
		{
			foreach(string s in servers)
			{
				if (s.split("/")[2] == server.split("/")[2] && s.split("/")[3] == server.split("/")[3])
					return true;
			}
			return false;
		}
		return true;
	}
	
	public void discover_server_address()
	{
		this._username = knots.get_settings().get_value("username");
		this._password = knots.get_settings().get_value("password");
		this.connection = null;
		servers = new string[0];
		string[] address = new string[0];
		string discover = null;
		int sid = 0;
		bool save_last_server = true;
		while (knots.get_settings().get_value("sticky_server" + sid.to_string()) != null)
		{
			servers += knots.get_settings().get_value("sticky_server" + sid.to_string());
			sid += 1;
		}
		if (knots.get_settings().get_value("enable_discovery") == null || knots.get_settings().get_value("enable_discovery") != "false")
		{
			try
			{
				GLib.Process.spawn_command_line_sync("/usr/share/knots/discover" + (knots.get_settings().get_value("discover_wait") != null ? " " + knots.get_settings().get_value("discover_wait") : ""), out discover);
			}
			catch (GLib.SpawnError se)
			{
				print("ERROR: %s\n", se.message);
			}
			if (discover != null)
			{
				string[] serverlist = discover.strip().split("\n");
				foreach(string server in serverlist)
				{
					if (server.str("/") != null)
					{
						if (!server_added(server))
							address += server;
					}
				}
			}
		}
		/*
			Don't use dynknots if other servers are found
		*/
		if (knots.get_settings().get_value("dynknots_key") != null && (knots.get_settings().get_value_default_to("empty_remote_discovery", "false") == "true" || address.length == 0))
		{
			string server = knots.get_connection().fetch_from_address("/knots2/dynknots.php?action=resolve&code=" + knots.get_settings().get_value("dynknots_key"), "nakkiboso.com", null, null);
			if (server != null && server.str("/") != null)
			{
				if (!server_added(server.strip()))
				{
					address += server.strip();
					save_last_server = false;
				}
			}
		}
		if (!server_added(knots.get_settings().get_value("last_server")) && save_last_server)
		{
			address += knots.get_settings().get_value("last_server");
		}
		foreach(string server in address)
		{
			if (load_connection(server, false, true))
			{
				add_server(server);
			}
			else
			{
				if (knots.get_settings().get_value("last_server") != null && knots.get_settings().get_value("last_server") == server)
				{
					knots.get_settings().unset_value("last_server");
				}
			}
		}
		if (server_count() == 1)
		{
			load_connection(servers[0], save_last_server, false);
		}
	}
	
	public string? show_address_dialog()
	{
		MessageDialog dialog = new MessageDialog.with_markup(null, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.OK_CANCEL, "Server address:");
		dialog.get_image().hide();
		gtk_window_set_portrait_flags(dialog, Hildon.PortraitFlags.SUPPORT);
		Hildon.Entry server_address = new Hildon.Entry(SizeType.FINGER_HEIGHT);
		string last_address = "192.168.0.1:1978";
		bool use_ssl = false;
		if (knots.get_settings().get_value("last_server") != null)
		{
			string[] tokens = knots.get_settings().get_value("last_server").split("/");
			last_address = tokens[3] + ":" + tokens[2];
			if (tokens[1] == "1")
				use_ssl = true;
		}
		server_address.set_text(last_address);
		Hildon.CheckButton cb = new Hildon.CheckButton(SizeType.FINGER_HEIGHT);
		cb.set_active(use_ssl);
		cb.set_label("Server uses SSL");
		dialog.vbox.pack_start(server_address, false, true, 1);
		dialog.vbox.pack_start(cb, false, true, 1);
		server_address.show();
		cb.show();
		var response = dialog.run();
		if (response == ResponseType.OK)
		{
			string[] address_port = server_address.get_text().split(":");
			if (address_port.length == 1)
				address_port += "1978";
			if (address_port.length == 2)
			{
				string address = address_port[0] + "/" + (cb.get_active() ? "1" : "0") + "/" + address_port[1] + "/" + address_port[0];
				dialog.destroy();
				return address;
			}
		}
		dialog.destroy();
		return null;
	}
	
	public bool test_connection()
	{
		if (connected())
		{
			set_timeout(10);
			set_url_cache(false);
			string test = fetch("/external/test_connection");
			set_timeout(default_timeout);
			if (test != null && test.str("KNOTS_OK") != null)
			{
				return true;
			}
		}
		return false;
	}
	
	public string get_last_url()
	{
		return last_url;
	}
	
	public void disconnect()
	{
		this.connection = null;
	}
	
	public string fetch(string url)
	{
		return fetch_from_address(url, address(), null, null);
	}
	
	public void set_timeout(int timeout)
	{
		this.timeout = timeout;
	}
	
	public void authenticate(bool force)
	{
		if (force || username() == null || password() == null)
		{
			Hildon.LoginDialog login = new LoginDialog(knots.get_window());
			if (server_name() != null)
				login.set_message("Enter password for " + server_name());
			var response = login.run();
			if (response == ResponseType.OK)
			{
				_username = login.get_username();
				_password = login.get_password();
			}
			else
			{
				reset_credentials();
			}
			login.destroy();
		}
	}
	
	public string fetch_from_address(string? urli, string? server_address, string? use_username, string? use_password, bool use_ssl = false)
	{
		
		string url = urli.replace (" ", "%20").replace("<", "%3C").replace(">" , "%3E");
		string uri = (uses_ssl() || use_ssl ? "https" : "http") + "://" + server_address + url;
		last_url = uri;
		if (cache)
		{
			string content = knots.get_common().cached_url(uri);
			if (content != null && content != "")
			{
				return content;
			}
		}
		if (!Thread.supported())
		{
		    return "";
		}
		var session = new Soup.SessionSync ();
		session.timeout = timeout;
		var message = new Soup.Message ("GET", uri);
		session.authenticate.connect((sess, msg, auth, retrying) => {
		    if (use_username == null)
		    {
			    if (retrying)
			    {
				reset_credentials();
				knots.alert("Authentication failed");
			    }
			    authenticate(false);
			    if (username() == null)
				return;
			    auth.authenticate(username(),password());
		   }
		   else
		   {
		   	auth.authenticate(use_username, use_password);
		   }
		});
		session.send_message (message);
		string body = (string)message.response_body.data;
		last_file_size = (int)message.response_body.length;
		message=null;
		session=null;
		if (body != null)
		{
			body = xml_cleanup(body);
			if (cache && body.length < 10000)
			{
				knots.get_common().cache_url(uri, body);
			}
			else
				cache = true;
			return body;
		
		}
		cache = true;
		return "";
	}
	
	public void reset_credentials()
	{
		_username = null;
		_password = null;
	}
	
	public string xml_cleanup(string data)
	{
		if (data != null && data.str("<?xml") != null)
		{
			return data.split("</root>")[0] + "</root>";
		}
		else
		if (data != null && data.str("END_CONTENT") != null)
		{
			return data.split("END_CONTENT")[0];
		}
		return data;
	}
	
	public bool connected()
	{
		return connection != null;
	}
	
	public bool uses_ssl()
	{
		if (connection != null)
		{
			return ((string)connection.lookup("ssl")).to_int() == 1;
		}
		return false;
	}
	
	public string? username()
	{
		if (connection.lookup("username") != null)
			return (string)connection.lookup("username");			
		return _username;
	}
	
	public string? password()
	{
		if (connection.lookup("password") != null)
			return (string)connection.lookup("password");
		return _password;
	}
	
	public string? ip()
	{
		if (connected())
			return (string)connection.lookup("address");
		return null;
	}
	
	public int port()
	{
		if (connected())
			return ((string)connection.lookup("port")).to_int();
		return -1;
	}
	
	public string? address()
	{
		if (connected())
			return ip() + ":" + port().to_string();
		return null;
	}
	
	public string? server_name()
	{
		if (connected())
			return (string)connection.lookup("name");
		return null;
	}
	
	
}
