/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2010 Koos Vriezen <koos.vriezen@xs4all.nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include <config.h>
#include <math.h>
#include <sys/types.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <glib/gthread.h>
#include <gdk/gdkx.h>

static gchar *plugin;
static gchar *object_url;
static gchar *mimetype;
static GtkWidget *xembed;
static Window socket_id;
static long parent_id;
static gboolean plugin_embedded;

static void output (const char *format, ...) {
    static gboolean need_cr;
    va_list vl;
    if (*format == '\r') {
        need_cr = TRUE;
    } else if (need_cr) {
        printf ("\r");
        need_cr = FALSE;
    }
    va_start (vl, format);
    vprintf (format, vl);
    va_end (vl);
    fflush (stdout);
}

#include "npplayer.h"
#include "io.h"

static NSNotify ns_notify;
static GTree *npp_streams;
static GTree *npp_plugins;
static int streams_deleted;
static void *npdata;
static void *nsdata;
static int npp_argc;
static char **npp_argn;
static char **npp_argv;
static gboolean stream_processing;

using namespace KMPlayer;


class KMPLAYER_NO_EXPORT NpStream : public IOJobListener {
public:
    enum Reason {
        NoReason = -1,
        BecauseDone = 0, BecauseError = 1, BecauseStopped = 2
    };

    NpStream (uint32_t stream, const URL & url,
            unsigned post_len, const char *post);
    ~NpStream ();

    void open ();
    void close ();

    virtual void jobData (IOJob * job, ByteArray & data);
    virtual void jobResult (IOJob * job);
    virtual void redirected (IOJob * job, const String & uri);

    uint32_t stream;
    URL url;
    IOJob *job;
    uint32_t bytes;
    ByteArray post_data;
    Reason finish_reason;
};


static String evaluateScript (const String &script) {
    fprintf (stderr, "evaluateScript %s\n", (const char *) script);
    if (script == "window.location+\"__flashplugin_unique__\"" ||
            script == "top.location+\"__flashplugin_unique__\"")
        return "http://www.example.com/__flashplugin_unique__";
    return "undefined";
}

static void destroyStream (void *inst, uint32_t sid) {
    NpStream * ns = (NpStream *) g_tree_lookup (npp_streams, (gpointer) sid);
    if (ns) {
        nppStreamFinished (sid, NpStream::BecauseStopped, nsdata);
        g_tree_remove (npp_streams, (gpointer) sid);
        delete (ns);
    }
}

static gboolean openStreams (gpointer key, gpointer value, gpointer data) {
    int active = (int)(long)data;
    NpStream *ns = (NpStream *)value;
    (void) key;
    if (ns->finish_reason == NpStream::NoReason && !ns->job) {
        ++active;
        ns->open ();
    }
    return active > 4;
}

static gboolean findClosedStreams (gpointer key, gpointer value, gpointer data) {
    NpStream *ns = (NpStream *)value;
    (void) key;
    if (ns->finish_reason > NpStream::NoReason) {
        *(NpStream **)data = ns;
        return true;
    }
    return false;
}

static void processStreams () {
    if (!stream_processing) {
        stream_processing = TRUE;
        streams_deleted = 0;
        int active = 0;
        g_tree_traverse (npp_streams, openStreams, G_IN_ORDER, &active);
        while (streams_deleted > 0) {
            NpStream *ns = NULL;
            g_tree_traverse (npp_streams, findClosedStreams, G_IN_ORDER, &ns);
            if (ns) {
                g_tree_remove (npp_streams, (gpointer) ns->stream);
                delete ns;
            } else {
                fprintf (stderr, "ERROR findClosedStreams didn't find a stream\n");
                break;
            }
            streams_deleted--;
        }
        stream_processing = FALSE;
    }
}

static void streamFinished (NpStream *ns) {
    nppStreamFinished (ns->stream, ns->finish_reason, nsdata);
    if (!stream_processing) {
        g_tree_remove (npp_streams, (gpointer) ns->stream);
        delete ns;
        processStreams ();
    } else {
        streams_deleted++;
    }
}

static void requestStream (void *inst, uint32_t sid,
        const String & url, const String & target,
        unsigned post_len, const char *post) {
    URL uri (url);
    if (!target.isEmpty ()) {
        fprintf (stderr, "new page request %s\n", (const char *) target);
        if (url.startsWith ("javascript:")) {
            String result = evaluateScript (url.mid (11));
            if (result == "undefined")
                uri = URL ();
            else
                uri = URL (URL (object_url), result); // probably wrong ..
        }
        /*if (!uri.url ().isEmpty ())
            emit openUrl (uri, target);*/
        nppStreamFinished (sid, NpStream::BecauseDone, nsdata);
    } else if (url.startsWith ("javascript:")) {
        String result = evaluateScript (url.mid (11));
        const char *data = (const char *)result;
        nppStreamData (sid, data, strlen (data), nsdata);
        nppStreamFinished (sid, NpStream::BecauseDone, nsdata);
    } else {
        fprintf (stderr, "request %d '%s'\n", sid, (const char *) uri.url ());
        NpStream * ns = new NpStream (sid, uri, post_len, post);
        g_tree_insert (npp_streams, (gpointer) sid, ns);
        if (url != uri.url ())
            nppStreamRedirected (sid, (const char *)uri.url ());
        processStreams ();
    }
}


extern "C" {

static void npSetDimension (void *ndata, int w, int h)
{
    output ("ID_VIDEO_WIDTH=%d\n", w);
    output ("ID_VIDEO_HEIGHT=%d\n", h);
}

static void npGetUrl (void *inst, uint32_t stream,
        const char *url, const char *target,
        unsigned plen, const char *post, void *ndata) {
    requestStream (inst, stream, url, target, plen, post);
}

static char *nsEvaluate (const char *script, void *ndata) {
    String result = evaluateScript (script);
    if (result.isEmpty ())
        return g_strdup ("");
    return g_strdup ((const char *)result);
}

static void npFinish (void *inst, uint32_t stream, void *ndata) {
    destroyStream (inst, stream);
}

static gint streamCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

static gboolean deleteStreams (gpointer key, gpointer value, gpointer data) {
    (void) key;
    delete (NpStream *)value;
    return false; // continue
}

static gboolean copyParams (gpointer key, gpointer value, gpointer data) {
    char ***tuple = (char ***)data;
    tuple[0][0] = (char *)key;
    tuple[1][0] = (char *)value;
    tuple[0]++;
    tuple[1]++;
    return false; // continue
}

} // extern "C"


static gboolean isFlashMimeType (const String &mime)
{
    return mime == "application/x-shockwave-flash" ||
        mime == "application/futuresplash";
}

static void stop ()
{
    if (npdata) {
        g_tree_foreach (npp_streams, deleteStreams, NULL);
        g_tree_destroy (npp_streams);
        npp_streams = NULL;
        nppInstanceClose (npdata);
        npdata = NULL;
    }
    for (int i = 0; i < npp_argc; i++) {
        free (npp_argn[i]);
        free (npp_argv[i]);
    }
    if (npp_argn) {
        free (npp_argn);
        free (npp_argv);
        npp_argn = npp_argv = NULL;
    }
    npp_argc = 0;
}

static void quit ()
{
    stop ();
    nppInstanceWindowClose (npdata); //FIXME
    gtk_main_quit();
}

static bool play ()
{
    nppInstanceTopUrl (object_url);
    npp_streams = g_tree_new (streamCompare);

    fprintf (stderr, "play %s\n", object_url);

    npdata = nppInstanceOpen (mimetype, npp_argc, npp_argn, npp_argv, nsdata, &ns_notify);
    if (!npdata) {
        g_tree_destroy (npp_streams);
        npp_streams = NULL;
        fprintf (stderr, "WARNING failed to create new instance\n");
        return false;
    }
    return true;
}

static gboolean initPlayer (void * p)
{
    npp_plugins = g_tree_new ((GCompareFunc)::strcmp);
    ns_notify.setDimension = npSetDimension;
    ns_notify.getUrl = npGetUrl;
    ns_notify.evaluate = nsEvaluate;
    ns_notify.finishStream = npFinish;

    nsdata = g_strdup ("");

    // only flash for now
    if (!mimetype)
        mimetype = g_strdup ("application/x-shockwave-flash");
    const char *lib = "/usr/lib/browser/plugins/libflashplayer.so";
    NPPluginLib *nplib = (NPPluginLib *) g_tree_search (npp_plugins,
            (GCompareFunc)::strcmp, mimetype);
    if (!nplib) {
        nplib = nppPluginInit (lib);
        if (!nplib) {
            fprintf (stderr, "WARNING failed to initialize plugin lib\n");
            return false;
        }
        g_tree_insert (npp_plugins, (char *) mimetype, nplib);
    }
    nppInstanceWindowParent (npdata, (void *) socket_id);
    nppInstanceWindowDimension (npdata, 0, 0, 800, parent_id ? 480 : 424);

    if (!play ())
        quit ();

    return FALSE;
}


KDE_NO_CDTOR_EXPORT NpStream::NpStream (uint32_t id, const URL &u,
        unsigned post_len, const char *post)
 : stream (id), url (u), job (NULL), finish_reason (NoReason) {
    if (post_len && post) {
        post_data.resize (post_len);
        memcpy (post_data.data (), post, post_len);
    }
}

KDE_NO_CDTOR_EXPORT NpStream::~NpStream () {
    close ();
}

KMPLAYER_NO_EXPORT void NpStream::open () {
    if (url.url().startsWith ("javascript:")) {
        String result = evaluateScript (url.url().mid (11));
        const char *data = (const char *)result;
        nppStreamData (stream, data, strlen (data), NULL /*FIXME*/);
        finish_reason = BecauseDone;
        streamFinished (this);
    } else {
        bytes = 0;
        job = asyncGet (this, url.url ());
        if (post_data.size ()) {
            String buf;
            int data_pos = -1;
            bool nl = true;
            for (int i = 0; i < post_data.size () && data_pos < 0; ++i) {
                char c = post_data.data () [i];
                switch (c) {
                    case '\r':
                        break;
                    case '\n':
                        if (nl)
                            data_pos = i + 1;
                        else
                            job->setHeader (buf);
                        buf.truncate (0);
                        nl = true;
                        break;
                    default:
                        nl = false;
                        buf += Char (c);
                }
            }
            job->setHeader ("Expect:");
            if (data_pos > 0)
                job->setHttpPostData (ByteArray (post_data.data () + data_pos,
                            post_data.size () - data_pos));
        }
        job->start ();
    }
}

KMPLAYER_NO_EXPORT void NpStream::close () {
    if (job)
        job->kill ();
    if (NoReason == finish_reason)
        nppStreamFinished (stream, BecauseStopped, nsdata);
    job = NULL;
}

KMPLAYER_NO_EXPORT void NpStream::jobData (IOJob *jb, ByteArray & data) {
    //debugLog () << "NpStream::jobData " << url.url () << endl;
    if (!bytes)
        nppStreamInfo (stream, jb->contentType (), jb->contentLength ());
    bytes += data.size ();
    nppStreamData (stream, data.data (), data.size (), NULL /*FIXME*/);
}

KMPLAYER_NO_EXPORT void NpStream::jobResult (IOJob *jb) {
    finish_reason = jb->error () || !bytes ? BecauseError : BecauseDone;
    job = NULL;
    streamFinished (this);
}

KMPLAYER_NO_EXPORT void NpStream::redirected (IOJob *, const String & uri) {
    url = URL (url, uri);
    nppStreamRedirected (stream, (const char *)url.url ());
}


static void embeddedEvent (GtkPlug *plug, gpointer d) {
    (void)plug; (void)d;
    fprintf (stderr, "embeddedEvent\n");
}

static void pluginAdded (GtkSocket *socket, gpointer p)
{
    (void)p;
    if (!plugin_embedded) {
        plugin_embedded = TRUE;
        int w = 0, h;
        gdk_drawable_get_size (xembed->window, &w, &h);
        fprintf (stderr, "pluginAdded %dx%d wid:%d\n", w, h, socket->plug_window);
        if (socket->plug_window) {
            gpointer user_data = NULL;
            gdk_window_get_user_data (socket->plug_window, &user_data);
            if (!user_data) {
                /**
                 * GtkSocket resets plugins XSelectInput in
                 * _gtk_socket_add_window
                 *   _gtk_socket_windowing_select_plug_window_input
                 **/
                XSelectInput (gdk_x11_get_default_xdisplay (),
                        gdk_x11_drawable_get_xid (socket->plug_window),
                        KeyPressMask | KeyReleaseMask |
                        ButtonPressMask | ButtonReleaseMask |
                        KeymapStateMask |
                        ButtonMotionMask |
                        PointerMotionMask |
                        /*EnterWindowMask | LeaveWindowMask |*/
                        FocusChangeMask |
                        ExposureMask |
                        StructureNotifyMask | SubstructureNotifyMask |
                        /*SubstructureRedirectMask |*/
                        PropertyChangeMask
                        );
            }
        }
        output ("Start playing\n");
        output ("\rA:0.0\rA:0.0");
        nppInstanceStart (object_url, mimetype, npdata);
    }
}

static gboolean configureEvent(GtkWidget *w, GdkEventConfigure *e, gpointer d)
{
    (void)w; (void)d;
    nppInstanceWindowDimension (npdata, e->x, e->y, e->width, e->height);
}

static void windowCreatedEvent (GtkWidget *w, gpointer d) {
    (void)d;
    socket_id = gtk_socket_get_id (GTK_SOCKET (xembed));
    if (parent_id) {
        if (!GTK_PLUG (w)->socket_window)
            gtk_plug_construct (GTK_PLUG (w), parent_id);
        /*gdk_window_reparent( w->window,
                GTK_PLUG (w)->socket_window
                    ? GTK_PLUG (w)->socket_window
                    : gdk_window_foreign_new (parent_id),
                0, 0);*/
        gtk_widget_show_all (w);
        /*XReparentWindow (gdk_x11_drawable_get_xdisplay (w->window),
                gdk_x11_drawable_get_xid (w->window),
                parent_id,
                0, 0);*/
    }
    g_timeout_add (0, initPlayer, NULL);
}

static gboolean windowCloseEvent (GtkWidget *w, GdkEvent *e, gpointer d)
 {
    (void)w; (void)e; (void)d;
    quit();
    return FALSE;
}

static void windowDestroyEvent (GtkWidget *w, gpointer d)
 {
    (void)w; (void)d;
    gtk_main_quit();
}

static gboolean cb_quit (gpointer data) {
    (void)data;
    quit ();
    return FALSE;
}

static gboolean watchStdin (GIOChannel *src, GIOCondition cond, gpointer data) {
    (void)data;
    if (cond == G_IO_IN) {
        static char cmdbuf[256];
        static int bufpos;
        char buf[64];
        int i;
        gsize nr = 0;
        GIOStatus status = g_io_channel_read_chars (src, buf, sizeof (buf), &nr, 0L);
        if (status == G_IO_STATUS_EOF || status == G_IO_STATUS_ERROR) {
            g_idle_add (cb_quit, NULL);
            return FALSE;
        }
        if (nr > 0) {
            for (i = 0; i < nr; i++) {
                if (buf[i] == '\n') {
                    cmdbuf[bufpos] = 0;
                    fprintf (stderr, "read '%s'\n", cmdbuf);
                    bufpos = 0;
                    if (!strncmp (cmdbuf, "quit", 4)) {
                        g_idle_add (cb_quit, NULL);
                    }
                } else if (bufpos >= sizeof (cmdbuf)) {
                    fprintf (stderr, "read error\n");
                    g_idle_add (cb_quit, NULL);
                    return FALSE;
                } else {
                    cmdbuf[bufpos++] = buf[i];
                }
            }
        }
    }
    return TRUE;
}

#include <execinfo.h>
#include <signal.h>
static void sig_abort(int i) {
    void* trace[256];
    int n = backtrace(trace, 256);
    (void)i;
    if (!n)
        return;
    backtrace_symbols_fd (trace, n, 2);
    exit(0);
}


int main (int argc, char **argv) {
    GTree *params;
    GtkWidget *window;
    GdkColormap *color_map;
    GdkColor bg_color;
    GIOChannel *channel_in;
    GIOFlags flags;
    int watch_in;
    int i;

    setenv ("GDK_NATIVE_WINDOWS", "1", 1);
    g_thread_init (NULL);
    gtk_init (&argc, &argv);

    signal(SIGABRT, sig_abort);
    signal(SIGINT, sig_abort);
    signal(SIGSEGV, sig_abort);

    for (i = 1; i < argc; i++) {
        if (!strcmp (argv[i], "-p") && ++i < argc) {
            plugin = g_strdup (argv[i]);
        } else if (!strcmp (argv[i], "-m") && ++i < argc) {
            mimetype = g_strdup (argv[i]);
        } else if (!strcmp (argv [i], "-wid") && ++i < argc) {
            parent_id = strtol (argv[i], 0L, 10);
        } else if (!strcmp (argv [i], "-args")) {
            ++i;
            break;
        } else if (*argv [i] != '-') {
            object_url = g_strdup (argv[i]);
        }
    }
    if (!(object_url && plugin)) {
        fprintf (stderr, "Usage: %s -m mimetype -p plugin url <-wid id> <--args arg0=val0 [arg1=val1 ..]>\n", argv[0]);
        return 1;
    }

    params = g_tree_new ((GCompareFunc)::strcmp);
    if (i < argc) {
        for (i; i < argc; i++) {
            String arg (argv[i]);
            int p = arg.indexOf (Char ('='));
            String nm = arg.left (p);
            if (!g_tree_lookup (params, (const char *) nm))
                g_tree_insert (params,
                        strdup ((const char *) nm),
                        strdup ((const char *) arg.mid (p+1)));
        }
    } else if (isFlashMimeType (mimetype)) {
        g_tree_insert (params, strdup ("SRC"), strdup (object_url));
    } else {
        String flashvars = String ("file=") + URL::encode_string (object_url);
        char *url = g_strdup ("http://video.linuxfoundation.org/sites/all/modules/custom/os_video_player/mediaplayer.swf");
        g_tree_insert (params, strdup ("SRC"), object_url);
        g_tree_insert (params, strdup ("flashvars"), strdup (flashvars));
        object_url = url;
    }
    npp_argc = g_tree_nnodes (params);
    npp_argn = (char **) malloc (npp_argc * sizeof (char *));
    npp_argv = (char **) malloc (npp_argc * sizeof (char *));
    char **tuple[2] = { npp_argn, npp_argv };
    g_tree_traverse (params, copyParams, G_IN_ORDER, tuple);
    g_tree_destroy (params);

    channel_in = g_io_channel_unix_new (0);
    g_io_channel_set_encoding (channel_in, NULL, NULL);
    g_io_channel_set_buffered (channel_in, FALSE);
    flags = g_io_channel_get_flags (channel_in);
    g_io_channel_set_flags(channel_in,(GIOFlags)(flags|G_IO_FLAG_NONBLOCK), 0L);
    watch_in = g_io_add_watch (channel_in, G_IO_IN, watchStdin, NULL);

    window = parent_id
        ? gtk_plug_new (parent_id)
        : gtk_window_new (GTK_WINDOW_TOPLEVEL);
    g_signal_connect (G_OBJECT (window), "delete_event",
            G_CALLBACK (windowCloseEvent), NULL);
    g_signal_connect (G_OBJECT (window), "destroy",
            G_CALLBACK (windowDestroyEvent), NULL);
    g_signal_connect_after (G_OBJECT (window), "realize",
            GTK_SIGNAL_FUNC (windowCreatedEvent), NULL);
    g_signal_connect (G_OBJECT (window), "configure-event",
            GTK_SIGNAL_FUNC (configureEvent), NULL);

    xembed = gtk_socket_new();
    g_signal_connect (G_OBJECT (xembed), "plug-added",
            GTK_SIGNAL_FUNC (pluginAdded), NULL);

    color_map = gdk_colormap_get_system();
    gdk_colormap_query_color (color_map, 0, &bg_color);
    gtk_widget_modify_bg (xembed, GTK_STATE_NORMAL, &bg_color);

    gtk_container_add (GTK_CONTAINER (window), xembed);

    if (!parent_id) {
        gtk_widget_set_size_request (window, 800, parent_id ? 480 : 424);
        gtk_widget_show_all (window);
    } else {
        g_signal_connect (G_OBJECT (window), "embedded",
                GTK_SIGNAL_FUNC (embeddedEvent), NULL);
        gtk_widget_set_size_request (window, 800, parent_id ? 480 : 424);
        gtk_widget_realize (window);
    }

    fprintf (stderr, "entering gtk_main\n");

    gtk_main();

    g_source_remove (watch_in);
    g_io_channel_shutdown (channel_in, TRUE, 0L);
    g_io_channel_unref (channel_in);

    fprintf (stderr, "done\n");

    return 0;
}
