/**************************************************************************
 *  Kchmviewer for Maemo5 - a CHM file viewer with broad language support *
 *  Port of KChmViewer ( www.kchmviewer.net ) to mobile platforms         *
 *  Copyright (C) 2009-2010 Max Kiva , max.kiva@gmail.com                 *
 *                                                                        *
 *  This program is free software: you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation, either version 3 of the License, or     *
 *  (at your option) any later version.                                   *
 *															                                  	      *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 **************************************************************************/
//#include <iostream>

#include <QtGui>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QApplication>
#include <qfiledialog.h>
#include <qtimer.h>

#include "mainwindow.h"
#include "pageviewwidget.h"
#include "toolbarwidget.h"
#include "contentswidget.h"
#include "finddialog.h"
#include "bookmarksdialog.h"
#include "recentdocsdialog.h"
#include "resources.h"
#include "chm.h"

/***************************************************************************
 */
MainWindow::MainWindow(QWidget *parent, Qt::WindowFlags flags) 
  : QMainWindow(parent,flags),
    pageView_( new PageViewWidget(this) ),
    toolBar_( new ToolBarWidget(this) ),
    toolBarSize_( screenWidth(),toolbarHeight()),
    pageSize_( screenWidth(), toolbarY() ),
    contentsSize_( screenWidth(), toolbarY() ),
    fullScreenOff_( new QPushButton( QPixmap( iconFullScreen() ),0,this) ),
    nextPage_( new QPushButton( QPixmap( iconForward() ),0,this) ),
    prevPage_( new QPushButton( QPixmap( iconBack() ),0,this) ),


    bookmarkPage_( new QAction(tr("Bookmark Page"), this) ), //QAction* bookmarkPage_;
    findInPage_( new QAction(tr("Find In Page"),this) ),//QAcion* findInPage_;

    fullScreenMode_(false)
{
#ifndef MAEMO  
  resize( screenWidth(), screenHeight() );
#endif  
  toolBar_->move( toolbarX(),toolbarY());
  toolBar_->resize( toolBarSize_ ); 
  toolBar_->show();

  pageView_->move(0,notificationAreaHeight());
  pageView_->setMinimumSize( pageSize_ );
  pageView_->setMaximumSize( pageSize_ );

  connect( pageView_,SIGNAL(cursorPositionChanged()),this,SLOT(showFullScreenControls()));

  // setup the full-screen-off button
  fullScreenOff_->move( screenWidth()-48-4,screenHeight()-48);
  fullScreenOff_->resize(48,48);
  fullScreenOff_->setMinimumSize( 48,48 );
  fullScreenOff_->setMaximumSize( 48,48 );
  fullScreenOff_->setFlat(true);
  fullScreenOff_->hide();
  connect( fullScreenOff_,SIGNAL(clicked()), this, SLOT(doFullScreenOff()) );
  
  prevPage_->move( 4, (screenHeight() - 48 )/2 );
  prevPage_->resize(48,48);
  prevPage_->setMinimumSize( 48,48 );
  prevPage_->setMaximumSize( 48,48 );
  prevPage_->setFlat(true);
  prevPage_->hide();
  connect( prevPage_,SIGNAL(clicked()), this, SLOT(doPrevPage()) );

  nextPage_->move( screenWidth() - 48 -4, (screenHeight() - 48 )/2 );
  nextPage_->resize(48,48);
  nextPage_->setMinimumSize( 48,48 );
  nextPage_->setMaximumSize( 48,48 );
  nextPage_->setFlat(true);
  nextPage_->hide();
  connect( nextPage_,SIGNAL(clicked()), this, SLOT(doNextPage()) );

  connect(bookmarkPage_, SIGNAL(triggered()) , this, SLOT(bookmarkCurrentPage()) );
  connect(findInPage_, SIGNAL(triggered()) , this, SLOT(findInCurrentPage()) );

  bookmarkPage_->setEnabled( false );
  findInPage_->setEnabled( false );

  menuBar()->addAction( bookmarkPage_ );
  menuBar()->addAction( findInPage_ );

  setCentralWidget(pageView_);
  grabZoomKeys();
}

MainWindow::~MainWindow()
{
  delete pageView_;

  delete toolBar_;
  delete fullScreenOff_;
  delete nextPage_;
  delete prevPage_;

  delete bookmarkPage_;
  delete findInPage_;
}

void MainWindow::grabZoomKeys(bool grab)
{

  if (!winId()) 
  {
    qWarning("Can't grab keys unless we have a window id");
    return;
  }

  unsigned long val = (grab) ? 1 : 0;
  Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
  if (!atom) 
  {
    qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM.");
    return;
  }

  XChangeProperty (QX11Info::display(),
      winId(),
      atom,
      XA_INTEGER,
      32,
      PropModeReplace,
      reinterpret_cast<unsigned char *>(&val),
      1);
}


void MainWindow::keyPressEvent(QKeyEvent* event)
{
  int key = event->key();
  Qt::KeyboardModifiers mods = event->modifiers();
  switch ( key ) 
  {
    case Qt::Key_F7:
      doZoomIn();
      break;

    case Qt::Key_F8:
      doZoomOut();
      break;
    default:
      QMainWindow::keyPressEvent(event);
      break;
  }
}



void MainWindow::openFile( const QString& fileName )
{
  if (  pageView_->loadChmFile( fileName ) )
  {
    bookmarkPage_->setEnabled( true );
    //findInPage_->setEnabled( true );
    toolBar_->enableButtons();
  }
}

void MainWindow::doOpenFile()
{

  QDir dir( pageView_->chm()->lastDocDir() );
  
  QString openDir = dir.absolutePath();

  if( !dir.exists() ) 
  {
    openDir = QDir::home().dirName();
  }

  QString fileName = QFileDialog::getOpenFileName( this, 
      tr("Open Chm file"), openDir, tr("Chm Files (*.chm)"));

  //TODO: check validity of the file name and if 
  //TODO: it is not the one active already
  if ( fileName.length() )
  {
    openFile( fileName );  
  }

}

void MainWindow::doOpenTableOfContents()
{
 QPoint pos( toolbarX(),toolbarY()-64 );
 int index = TocDialog::doToc( *(pageView_->chm()), pos,contentsSize_,this );

#ifdef KCHMVIEWER_DEBUG
 qDebug( "Chose toc item: %d", index );
#endif
 if( index >= 0 )
 {
   pageView_->openPageNum( index );
 }
}

void MainWindow::doBookmarks()
{
 
 QPoint pos( toolbarX(),toolbarY()-64 );
 int index = BookmarksDialog::doBookmarks( *(pageView_->chm()), pos,contentsSize_,this );

#ifdef KCHMVIEWER_DEBUG
 qDebug( "Chose bookmarked page: %d", index );
#endif
 if( index >= 0 )
 {
   const QStringList& rl = pageView_->chm()->getBookmarks();

#ifdef KCHMVIEWER_DEBUG
   qDebug("opening bookmarked page: %s", rl[index].toAscii().constData() );
#endif  

   

   pageView_->openPageNum( pageView_->chm()->getUrlPageNum(rl[index]) );
 }

}


void MainWindow::doRecentDocs()
{
 
 QPoint pos( toolbarX(),toolbarY()-64 );
 int index = RecentDocsDialog::doRecentDocs( *(pageView_->chm()), pos,contentsSize_,this );

#ifdef KCHMVIEWER_DEBUG
 qDebug( "Chose bookmarked page: %d", index );
#endif
 if( index >= 0 )
 {
   const QStringList& rl = pageView_->chm()->getRecentDocs();

#ifdef KCHMVIEWER_DEBUG
   qDebug("opening bookmarked page: %s", rl[index].toAscii().constData() );
#endif   
   openFile( rl[index] );
 }

}


void MainWindow::doFind()
{
 
 QPoint pos( toolbarX(),toolbarY()-64 );
 int index = FindDialog::doFind( *(pageView_->chm()),  pos,contentsSize_, this );

#ifdef KCHMVIEWER_DEBUG
 qDebug( "Find result: %d", index );
#endif 
 if( index >= 0 )
 {
   const QStringList& rl = pageView_->chm()->getSearchResults();

   pageView_->openPageNum( pageView_->chm()->getUrlPageNum(rl[index]) );
 }

}

void MainWindow:: doFullScreenOn()
{
  if( !fullScreenMode_ )
  {
    
#ifdef STD_TRACING
    std::cout << __PRETTY_FUNCTION__ 
              << ": full screen ON"
              << std::endl;
#endif
    
    toolBar_->hide();

    pageView_->move(0,0);
    pageView_->setMinimumSize( screenWidth(),screenHeight() );
    pageView_->setMaximumSize(  screenWidth(),screenHeight() );


    showFullScreen();
    // must be set before showing controls
    fullScreenMode_ = true;
    
    showFullScreenControls();


  }
}

void MainWindow:: doFullScreenOff()
{
  if( fullScreenMode_ )
  {
#ifdef STD_TRACING
    std::cout << __PRETTY_FUNCTION__ 
              << ": full screen OFF"
              << std::endl;
#endif

    pageView_->move(0,notificationAreaHeight());
    pageView_->setMinimumSize( pageSize_ );
    pageView_->setMaximumSize( pageSize_ );
  
    hideFullScreenControls();
    
    toolBar_->show();

    showNormal();
    

    fullScreenMode_ = false;
  }
}

void MainWindow:: doPrevPage()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  toolBar_->enableButtons( false );
  pageView_->prevPage();
  toolBar_->enableButtons( true );
}

void MainWindow:: doNextPage()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  toolBar_->enableButtons( false );
  pageView_->nextPage();
  toolBar_->enableButtons( true );
}

void MainWindow:: doZoomIn()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  pageView_->zoomIn();
}

//////////////////////////////////////////////////////////////////////////////
void MainWindow:: doZoomOut()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif
  pageView_->zoomOut();
}

//////////////////////////////////////////////////////////////////////////////
void MainWindow::bookmarkCurrentPage()
{
  pageView_->bookmarkCurrentPage();
}


//////////////////////////////////////////////////////////////////////////////
void MainWindow::findInCurrentPage()
{
}


void MainWindow::minimizeApplication()
{
    setWindowState(Qt::WindowMinimized);
}


void MainWindow::showFullScreenControls()
{
    if( fullScreenMode_ )
    {
      prevPage_->show();
      nextPage_->show();

      fullScreenOff_->show();

      QTimer::singleShot (5000, this, SLOT(hideFullScreenControls()) );
    }
}

void MainWindow::hideFullScreenControls()
{
    fullScreenOff_->hide();
    nextPage_->hide();
    prevPage_->hide();
}


