/**************************************************************************
 *  Kchmviewer for Maemo5 - a CHM file viewer with broad language support *
 *  Port of KChmViewer ( www.kchmviewer.net ) to mobile platforms         *
 *  Copyright (C) 2009-2010 Max Kiva , max.kiva@gmail.com                 *
 *                                                                        *
 *  This program is free software: you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation, either version 3 of the License, or     *
 *  (at your option) any later version.                                   *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 **************************************************************************/
#include <QDir>
#include <QTextStream>
#include <libchmurlfactory.h>
#include <libchmsearchengine.h>
#include "resources.h"
#include "chm.h"

//////////////////////////////////////////////////////////////////////////////
Chm::Chm()
  : chm_(),
    fileInfo_(),
    docDir_(),
    se_(),
    tableOfContents_(),
    resultList_(),
    bookmarks_(),
    recentDocs_(),
    lastDocDir_(),
    searchText_()
{
// TODO: provide signal to emit the progress notifications
//  connect( se_, SIGNAL( progressStep( int, const QString& ) ), this, SLOT( onProgressStep( int, const QString& ) ) );
    loadRecentDocs();
    loadLastDocDir();
}

//////////////////////////////////////////////////////////////////////////////
Chm::~Chm()
{
}

//////////////////////////////////////////////////////////////////////////////
bool Chm::loadFile( const QString& name )
{ 
  bool res = chm_.loadFile( name ); 

  if ( res )
  {

    tableOfContents_.clear(); 
    chm_.parseTableOfContents( &tableOfContents_ );

    urlMap_.clear();

    for( int i = 0; i < tableOfContents_.count(); ++i )
    { 

      QString url = tableOfContents_[i].urls.join("|");
      urlMap_.insert( url, i );
    }


    fileInfo_.setFile( name );

    lastDocDir_ = fileInfo_.absoluteDir().absolutePath();

    storeLastDocDir();

    addRecentDoc( name );
    storeRecentDocs();


    createDataDir();
    resultList_.clear();
    bookmarks_.clear();
  
    searchText_.clear();

    loadSearchResults();
    loadBookmarks();
  }

  return res;
}

//////////////////////////////////////////////////////////////////////////////
bool Chm::addRecentDoc( const QString& doc )
{
  bool res = true;

  // add only of it is not in the list
  if(  recentDocs_.indexOf(doc) == -1 )
  {
#ifdef KCHMVIEWER_DEBUG
    qDebug("adding recent doc: \'%s\'", doc.toAscii().constData());
#endif    
    if ( recentDocs_.size() > 16 )
    {
      recentDocs_.removeFirst();
    }
    recentDocs_.push_front( doc );
  }

  return res;
}

//////////////////////////////////////////////////////////////////////////////
void Chm::createDataDir()
{
  //get the file name without the path and extension
  QString justName;
  //get user home dir
  //QString home = QDir::home().dirName();
  QDir appDir( QDir::homePath() + "/." + appName() );

  // ensure app directory exists
  if (!appDir.exists() )
  {
    appDir.mkdir( appDir.absolutePath() );
  }
#ifdef KCHMVIEWER_DEBUG
  qDebug( "app dir: %s", appDir.absolutePath().toAscii().constData() );
#endif
  // ensure this document directory exists
  QString doc_dir = appDir.absolutePath() + "/" + fileInfo_.completeBaseName();
  docDir_.setPath( doc_dir );
#ifdef KCHMVIEWER_DEBUG
  qDebug( "document dir: %s", docDir_.absolutePath().toAscii().constData() );
#endif  
  if( !docDir_.exists() )
  {
    docDir_.mkdir( docDir_.absolutePath() );
  }
  
}
/////////////////////////////////////////////////////////////////////////////
// load the search index for the document or if that fails
// generate fresh one
bool Chm::loadSearchIndex()
{
  bool result = false;
  
  QFile file( docDir_.absoluteFilePath("index") );
  result = file.open( QIODevice::ReadOnly );

  if ( !result )
  {
    result = generateSearchIndex();
    if ( result )
    {
      result = file.open( QIODevice::ReadOnly );
    }
  }

  if ( result )
  {
      QDataStream stream( &file );
      result = se_.loadIndex( stream );

      file.close();
  }
  
  return true;
}
///////////////////////////////////////////////////////////////////
//
// generate search index and return 'true' if successfull
bool Chm::generateSearchIndex()
{
  QFile file( docDir_.absoluteFilePath("index") );
  bool result = file.open( QIODevice::WriteOnly );

  if ( result )
  {
      // file opened ok, generate the index and store result
      QDataStream stream( &file );
      result = se_.generateIndex( &chm_, stream );
      file.close();
  }

  return result;
}
////////////////////////////////////////////////////////////////////////
bool Chm::search( const QString& text  )
{
  // load ( create first if required ) the search index
  bool result = loadSearchIndex();

  if( result )
  {
      resultList_.clear();

      result = se_.searchQuery( text , &resultList_, &chm_ );

      if ( result )
      {
        searchText_ = text;
        storeSearchResults( text );
      }
  }

  return result;
}

///////////////////////////////////////////////////////////////////////
bool Chm::storeSearchResults( const QString& text )
{
  bool result = false;

  QString file( docDir_.absoluteFilePath("search.results") );
  resultList_.push_front( text );
  result = storeStringList( file, resultList_ );
  resultList_.pop_front();

  return result;
}

///////////////////////////////////////////////////////////////////////
bool Chm::loadSearchResults()
{
  bool result = false;

  QString file( docDir_.absoluteFilePath("search.results") );
  result = loadStringList( file, resultList_ );

  if ( result && resultList_.size() )
  {
    searchText_ = resultList_[0]; 
    resultList_.pop_front();
  }
  return result;
}

//////////////////////////////////////////////////////////////////////////////
void Chm::bookmarkPage( const QString& url )
{
  QString topic = chm_.getTopicByUrl( url );

  if( topic != QString::null && bookmarks_.indexOf(url) == -1 )
  {
#ifdef KCHMVIEWER_DEBUG
    qDebug( "bookmarking : %s/%s", url.toAscii().constData(), topic.toAscii().constData() );
#endif
    bookmarks_.append( url );
    storeBookmarks();
  }
  else
  {
#ifdef KCHMVIEWER_DEBUG
    qDebug( "could not bookmark : %s/%s", url.toAscii().constData(), topic.toAscii().constData() );
#endif
  }

}

void Chm::deleteBookmark( size_t i )
{
  if( i < static_cast<size_t>(bookmarks_.size()) )
  {
    bookmarks_.removeAt(i);
  }
}

///////////////////////////////////////////////////////////////////////
bool Chm::storeRecentDocs()
{
  bool result = false;

  QDir appDir( QDir::homePath() + "/." + appName() );
  QString file( appDir.absoluteFilePath("recentDocs") );

  result = storeStringList( file, recentDocs_ );

  return result;
}

///////////////////////////////////////////////////////////////////////
bool Chm::loadRecentDocs()
{
  bool result = false;

  QDir appDir( QDir::homePath() + "/." + appName() );
  QString file( appDir.absoluteFilePath("recentDocs") );

  result = loadStringList( file, recentDocs_ );

  return result;
}


///////////////////////////////////////////////////////////////////////
bool Chm::loadBookmarks()
{
  bool result = false;

  QString name = docDir_.absoluteFilePath("bookmarks");
  result = loadStringList( name, bookmarks_ );

  return result;
}
///////////////////////////////////////////////////////////////////////
bool Chm::storeBookmarks()
{
  bool result = false;

  QString name = docDir_.absoluteFilePath("bookmarks");
  result = storeStringList( name, bookmarks_ );

  return result;
}


///////////////////////////////////////////////////////////////////////
bool Chm::loadLastDocDir()
{
  bool result = false;

  QStringList list;

  QString name = docDir_.absoluteFilePath("lastdocdir");
  result = loadStringList( name, list );

  if ( list.size() )
  {
    lastDocDir_ = list[0];
  }
  else
  {
    lastDocDir_.clear();
  }

  return result;
}
///////////////////////////////////////////////////////////////////////
bool Chm::storeLastDocDir()
{
  bool result = false;

  QStringList list;
  list.append( lastDocDir_ );

  QString name = docDir_.absoluteFilePath("lastdocdir");
  result = storeStringList( name, list );

  return result;
}

///////////////////////////////////////////////////////////////////////////////
bool Chm::loadStringList( const QString& fileName , QStringList& list )
{
  bool result = false;

  QFile file( fileName );

  result = file.open( QFile::ReadOnly );

  if ( result )
  {
    list.clear();


#if 0    
    QDataStream stream( &file );

    stream >> list;
#else    
    QTextStream in(&file);
    QString line = in.readLine();
    for( ; !line.isNull(); line = in.readLine())
    {
#ifdef KCHMVIEWER_DEBUG
      qDebug("loading string: %s", line.toAscii().constData() );
#endif      
      list.append( line );
    }
#endif    
  }
  else
  {
#ifdef KCHMVIEWER_DEBUG
    qDebug("could not open file \'%s\' to read.",  fileName.toAscii().constData() );
#endif      
  }
  return result;
}

///////////////////////////////////////////////////////////////////////
bool Chm::storeStringList( const QString& fileName, const QStringList& list )
{
  bool result = false;

  if( !list.empty() )
  {
    QFile file( fileName );
    result = file.open( QFile::WriteOnly | QFile::Truncate );
    if ( result )
    {
#if 0 
      QDataStream stream( &file );
      stream << list;
#else      
      QTextStream out( &file );
      for( int i = 0; i < list.count(); ++i )
      {
        out << list[i];
        endl( out );
#ifdef KCHMVIEWER_DEBUG
        qDebug("storing bookmark: %s", list[i].toAscii().constData() );
#endif        
      }
#endif      
      file.close();
    }
    else
    {
#ifdef KCHMVIEWER_DEBUG
      qDebug("could not open file \'%s\' to write.",  fileName.toAscii().constData() );
#endif      
    }
  }
  else
  {
#ifdef KCHMVIEWER_DEBUG
    qDebug("no strings to store.");
#endif
  }

  return result;
}


///////////////////////////////////////////////////////////////////////////////
size_t Chm::getUrlPageNum( const QString& url ) const
{
  return urlMap_.value( url, 0 );
}

