/**************************************************************************
 *  Kchmviewer for Maemo5 - a CHM file viewer with broad language support *
 *  Copyright (C) 2009-2010 Max Kiva , max.kiva@gmail.com                 *
 *                                                                        *
 *  This program is free software: you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation, either version 3 of the License, or     *
 *  (at your option) any later version.                                   *
 *															                                  	      *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 **************************************************************************/

#include <QWidget>
#include <QString>
#include <QPalette>
#include <QColor>
#include <QFile>
#include <QDataStream>
#include <libchmurlfactory.h>
#include <libchmsearchengine.h>
#include "pageviewwidget.h"
#include "mainwindow.h"
#include "chm.h"


static QString decodeUrl( const QString &input );

/****************************************************************************
 *
 *
 */
PageViewWidget::PageViewWidget( QWidget* parent )
  : QTextBrowser( parent ),
    chm_(  new Chm() ),
    tableOfContents_(),
    baseUrl_("/"),
    page_(0)
{
  setWordWrapMode(QTextOption::WordWrap);
}

/****************************************************************************
 *
 *
 */
PageViewWidget::~PageViewWidget()
{
  delete chm_;
}

void PageViewWidget::openPageNum( size_t page )
{
  if ( page && page < static_cast<size_t>(tableOfContents_.size()) )
  {
    QString url = tableOfContents_[page].urls.join("|");
    
    //MainWindow* wnd = dynamic_cast<MainWindow*>(parentWidget()); 
    //if ( wnd )
    //{
    //  wnd->setWindowTitle( QString(tableOfContents_[page].name) );
    ////}
    // setTitle( "KChmViewer @" +tableOfContents_[page].name );
    
    openPage( url );
    page_ = page;

  }
}

/****************************************************************************
 *
 *
 */
bool PageViewWidget::openPage( const QString& url )
{
  QString topic = chm_->getTopicByUrl(url);
#ifdef KCHMVIEWER_DEBUG  
  qDebug("PageViewWidget::openPage \'%s\'", topic.toAscii().constData() );
#endif
  bool result = true;
  
  MainWindow* wnd = dynamic_cast<MainWindow*>(parentWidget()); 
  if ( wnd )
  {
      wnd->setWindowTitle( topic );
  }
    
  makeURLabsolute( url, true );
  QString decodedUrl = decodeUrl( url );

  
  // call into baseclass: QTextBrowsr
  setSource( decodedUrl );
  

  reload();
  

    
  return result;
}



static QString decodeUrl( const QString &input )
{
	QString temp;
	int len = input.length();
	for( int i = 0 ; i < len; ++i )
	{
		QChar c = input[i];
		if (c == '%' && i + 2 < len)
		{
			int a = input[++i].unicode();
			int b = input[++i].unicode();

			if (a >= '0' && a <= '9')
				a -= '0';
			else if (a >= 'a' && a <= 'f')
				a = a - 'a' + 10;
			else if (a >= 'A' && a <= 'F')
				a = a - 'A' + 10;

			if (b >= '0' && b <= '9')
				b -= '0';
			else if (b >= 'a' && b <= 'f')
				b  = b - 'a' + 10;
			else if (b >= 'A' && b <= 'F')
				b  = b - 'A' + 10;

			temp.append( (QChar)((a << 4) | b ) );
		}
		else
		{
			temp.append( c );
		}
	}

   return temp;
}


bool PageViewWidget::loadChmFile( const QString& filename )
{
  

  bool loaded = chm_->loadFile( filename );

  if ( loaded )
  {
    baseUrl_ = "/";

    filename_ = filename;

    tableOfContents_.clear(); 
    chm_->parseTableOfContents( &tableOfContents_ );


    if ( tableOfContents_.size()  )
    {
      MainWindow* wnd = dynamic_cast<MainWindow*>(parentWidget()); 
      if ( wnd )
      {
        wnd->setWindowTitle( QString(tableOfContents_[0].name) );
      }

      openPage( tableOfContents_[0].urls.join("|") );
    }

  }

  return loaded;
  
}


QVariant PageViewWidget::loadResource(int type, const QUrl & name)
{
	QString data, file, path = name.toString( QUrl::StripTrailingSlash );

#ifdef STD_TRACING  
  std::cout << "loadResource :" << name.toString().toStdString() << std::endl;
#endif  
	// Retreive the data from chm file

	if ( !chm_ )
		return 0;

	int pos = path.indexOf('#');
	if ( pos != -1 )
		path = path.left (pos);
	
	path = makeURLabsolute( path, false );

#ifdef STD_TRACING  
  std::cout << "loadResource : abs url:" << path.toStdString() << std::endl;
#endif  
	// To handle a single-image pages, we need to generate the HTML page to show 
	// this image. We did it in KCHMViewWindow::handleStartPageAsImage; now we need
	// to generate the HTML page, and set it.
	if ( LCHMUrlFactory::handleFileType( path, data ) )
		return QVariant( QString( data ) );
	
#ifdef STD_TRACING  
  std::cout << "loadResource : type" << type << std::endl;
#endif  

	if ( type == QTextDocument::HtmlResource || type == QTextDocument::StyleSheetResource )
	{

#ifdef STD_TRACING    
    std::cout << "loadResource : html or css , path:" << path.toStdString() << std::endl;
#endif

		if ( !chm_->getFileContentAsString( &data, path ) )
			qWarning( "Needed file %s is not present in this CHM archive\n", qPrintable( path ) );

		// A "fix" (actually hack) for CHM files people sent to me. I have no idea why QTextBrowser cannot show it.
		if ( type == QTextDocument::HtmlResource )
			data.remove( "<META http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">" );

		return QVariant( QString( data ) );
	}
	else if ( type == QTextDocument::ImageResource )
	{
		QImage img;
		QByteArray buf;
		
		QString fpath = decodeUrl( path );
		
#ifdef STD_TRACING
    std::cout << "loadResource : image , path:" << fpath.toStdString() << std::endl;
#endif    
		if ( chm_->getFileContentAsBinary( &buf, fpath ) )
		{
			if ( !img.loadFromData ( (const uchar *) buf.data(), buf.size() ) )
      {
				qWarning( "Needed file %s is not present in this CHM archive\n", qPrintable( path ) );
      }
		}
    else
    {
#ifdef STD_TRACING
      std::cout << "loadResource : image , binary load failed." << std::endl;
#endif      
    }
		
		return QVariant( img );
	}
	
//	qWarning("loadResource: Unknown type %d", type);
	return QVariant();
}
//////////////////////////////////////////////////////////////////////////////////////////////////
QString PageViewWidget::makeURLabsolute ( const QString & url, bool set_as_base )
{
	QString p1, p2, newurl = url;

	if ( !LCHMUrlFactory::isRemoteURL (url, p1)
	&& !LCHMUrlFactory::isJavascriptURL (url)
	&& !LCHMUrlFactory::isNewChmURL (url, p1, p2) )
	{
		newurl = QDir::cleanPath (url);

		// Normalize url, so it becomes absolute
		if ( newurl[0] != '/' )
			newurl = baseUrl_ + "/" + newurl;
	
		newurl = QDir::cleanPath (newurl);


		if ( set_as_base  )
		{
			baseUrl_ = newurl;
		
			// and set up new baseurl
			int i = newurl.lastIndexOf('/');
			if ( i != -1 )
				baseUrl_ = QDir::cleanPath (newurl.left (i + 1));
		}
	}

	return newurl;
}


void PageViewWidget::nextPage()
{
  size_t pg = page_+1;
  openPageNum( pg );
}

void PageViewWidget::prevPage()
{
  size_t pg = page_-1;
  openPageNum( pg );
}

void PageViewWidget::zoomOut()
{
  QTextBrowser::zoomOut();
}

void PageViewWidget::zoomIn()
{
  QTextBrowser::zoomIn();
}

void PageViewWidget::bookmarkCurrentPage()
{
  if ( tableOfContents_.size() )
  {
    QString url = tableOfContents_[page_].urls.join("|");
    chm_->bookmarkPage( url );
  }
}








