/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtSql/QSqlQuery>
#include <QtCore/QVariant>
#include <QtCore/QDebug>
#include <QtCore/QTranslator>
#include <QtCore/QLocale>
#include <gconf/gconf-client.h>
#include "settings.h"
#include "db.h"

namespace
{
    static int const LANGUAGE_COUNT = 5;

    static QString const LANGUAGE_NAMES[LANGUAGE_COUNT] = {
       "English",
       "Deutsch",
       "Norsk",
       "Suomi",
       "Svenska"
    };

    static QString const LANGUAGE_IDS[LANGUAGE_COUNT] = {
       "en_US",
       "de_DE",
       "nb_NO",
       "fi_FI",
       "sv_SE"
    };
}

Settings* Settings::instance_ = 0;

Settings* Settings::instance()
{
    if(!instance_)
    {
        instance_ = new Settings;
    }

    return instance_;
}

void Settings::getLanguages(QList<Settings::Language>& languages)
{
    for(int i = 0; i < LANGUAGE_COUNT; i++)
    {
        Language lang;
        lang.name = LANGUAGE_NAMES[i];
        lang.id = LANGUAGE_IDS[i];
        languages.push_back(lang);
    }
}

void Settings::loadLanguage(QApplication& app)
{
    QString language = get("language");

    if(language.isEmpty())
    {
       language = QLocale::system().name();
    }

    QTranslator* translator = new QTranslator(&app);
    translator->load(":/translations/" + language);
    app.installTranslator(translator);
}

void Settings::close()
{
    delete instance_;
    instance_ = 0;
}

void Settings::startEdit()
{
    if(!editing_ && !DB::connected())
    {
        editing_ = DB::connect();
    }
}

void Settings::endEdit()
{
    if(editing_)
    {
        DB::disconnect();
        editing_ = false;
    }
}

bool Settings::set(QString const& name, QString const& value)
{
    bool close = !editing_;

    startEdit();

    QSqlQuery deleteQuery;
    deleteQuery.prepare("DELETE FROM settings WHERE name = :name");
    deleteQuery.bindValue(":name", QVariant(name));
    deleteQuery.exec();

    QSqlQuery query;
    query.prepare("INSERT INTO settings(name, value) VALUES(:name, :value)");
    query.bindValue(":name", QVariant(name));
    query.bindValue(":value", QVariant(value));

    bool returnValue = query.exec();

    if(close)
    {
        endEdit();
    }

    settings_[name] = value;

    return returnValue;
}

QString Settings::get(QString const& name)
{
    QMap<QString, QString>::iterator it;

    if((it = settings_.find(name)) != settings_.end())
    {
        return it.value();
    }

    QString result = "";

    bool close = !editing_;

    startEdit();

    QSqlQuery query;

    query.prepare("SELECT value FROM settings WHERE name = :name");
    query.bindValue(":name", name);

    if(query.exec() && query.next())
    {
        result = query.value(0).toString();
    }
    else
    {
        result = getDefaultValue(name);
    }

    if(close)
    {
        endEdit();
    }

    settings_[name] = result;

    return result;

}

QString Settings::getDefaultValue(QString const& name)
{
    static bool defaultValuesLoaded = false;
    static QMap <QString, QString> defaultValues;

    if(!defaultValuesLoaded)
    {
        defaultValues["autostart"] = "1";
        defaultValues["eniro_site"] = tr("fi");
        defaultValues["cache_size"] = "200";
        defaultValues["connection"] = "global";
        defaultValuesLoaded = true;
    }

    QMap<QString, QString>::const_iterator it = defaultValues.find(name);

    if(it != defaultValues.end())
    {
        return it.value();
    }

    return "";

}

bool Settings::reset()
{
    bool close = !editing_;

    startEdit();

    QSqlQuery query;

    bool ret = query.exec("DELETE FROM settings");

    if(close)
    {
        endEdit();
    }

    return ret;
}

Settings::ConnectionType Settings::getConnectionType()
{
    QString value = get("connection");

    if(value == "any")
    {
        return ANY;
    }
    else if(value == "wlan")
    {
        return WLAN;
    }
    else if(value == "gprs")
    {
        return GPRS;
    }
    else
    {
        if(value != "global")
        {
            qDebug() << "Unknown connection type in settings, using default";
        }

        QList<QString> values;

        GConfClient* gcClient = NULL;
        gcClient = gconf_client_get_default();

        g_assert(GCONF_IS_CLIENT(gcClient));

        GError* error = NULL;
        GSList* list = NULL;
        list = gconf_client_get_list(gcClient,
                                     "/system/osso/connectivity/network_type/auto_connect",
                                     GCONF_VALUE_STRING,
                                     &error);

        if(error)
        {
            qDebug() << "Error: " << error->message;
            g_error_free(error);
        }
        else
        {
            while(list)
            {
                values.push_back((char *)list->data);
                list = list->next;
            }
        }

        g_object_unref(gcClient);

        if(values.size() == 0)
        {
            return ALWAYS_ASK;
        }
        else
        {
            QString value = values.at(0);

            if(value == "*")
            {
                return ANY;
            }
            else if(value == "GPRS")
            {
                return GPRS;
            }
            else if(value == "WLAN_INFRA" || value == "WLAN_ADHOC" || value == "WLAN")
            {
                return WLAN;
            }
            else
            {
                qDebug() << "Unknown connection type: " << value;
                return ALWAYS_ASK;
            }
        }

    }

    return ALWAYS_ASK;
}

Settings::Settings(): editing_(false)
{
}

Settings::~Settings()
{
    DB::removeDatabase();
}
