/*
 * jammo-track-view.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-track-view.h"
#include "jammo-sample-button.h"
#include "../meam/jammo-sample.h"
#include "../gems/gems.h"
G_DEFINE_TYPE(JammoTrackView, jammo_track_view, TANGLE_TYPE_DROPPABLE_ACTOR);

enum {
	PROP_0,
	PROP_TRACK,
	PROP_N_SLOTS,
	PROP_SLOT_DURATION,
	PROP_SLOT_HEIGHT,
	PROP_SLOT_WIDTH,
	PROP_LINE_EVERY_NTH_SLOT,
	PROP_DISABLED_SLOTS_BEGIN,
	PROP_DISABLED_SLOTS_END,
	PROP_EDITING_ENABLED
};

struct _JammoTrackViewPrivate {
	JammoEditingTrack* track;
	guint n_slots;
	guint64 slot_duration;
	gfloat slot_width;
	gfloat slot_height;
	guint line_every_nth_slot;
	guint disabled_slots_begin;
	guint disabled_slots_end;
	ClutterColor hilight_color;
	
	gfloat real_slot_width;
	gfloat hilighted_start;
	gfloat hilighted_end;
	TangleGrid* grid;
	
	guint editing_enabled : 1;
};

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data);
static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data);
static gboolean check_if_overlapping(JammoTrackView* track_view, ClutterActor* sample_button, guint slot_index);

ClutterActor* jammo_track_view_new(JammoEditingTrack* track, guint n_slots, guint64 slot_duration, gfloat slot_width, gfloat slot_height) {

	//printf ("JAMMO_TYPE_TRACK_VIEW, n-slots:%d, slot-duration:%Lu,  slot-width:%f, slot-height:%f\n", n_slots,slot_duration, slot_width, slot_height);
	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_TRACK_VIEW, "track", track, "n-slots", n_slots, "slot-duration", slot_duration, "slot-width", slot_width, "slot-height", slot_height, NULL));
}

/* Note! This function does not check if the given sample will be overlapping some other sample. */
void jammo_track_view_add_jammo_sample_button(JammoTrackView* track_view, JammoSampleButton* sample_button, guint slot) {
	TangleWidget* widget;

	jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button), slot * track_view->priv->slot_duration);

	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));

	clutter_actor_set_x(CLUTTER_ACTOR(sample_button),
		slot * (track_view->priv->real_slot_width ? track_view->priv->real_slot_width : track_view->priv->slot_width));
	clutter_actor_set_y(CLUTTER_ACTOR(sample_button), 0.0);
	clutter_actor_set_height(CLUTTER_ACTOR(sample_button), track_view->priv->slot_height);
	tangle_widget_add(TANGLE_WIDGET(widget), CLUTTER_ACTOR(sample_button), NULL);
}

/*
Used when removing over network,
*/
void jammo_track_view_remove_jammo_sample_button_from_slot(JammoTrackView* track_view, guint asked_slot) {
	gfloat asked_x = asked_slot * track_view->priv->slot_width;
	//printf("looking sample-button from x-position '%f'\n",asked_x);
	TangleWidget* widget;
	GList* children;
	ClutterActor* actor = NULL;

	gfloat x;  //left side of actor
	gfloat x2; //x2=x+width, right side of actor
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));

	for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
		actor = CLUTTER_ACTOR(children->data);
		x = clutter_actor_get_x(actor);
		x2 = x + clutter_actor_get_width(actor);
		//printf("this sample-button covers '%f' ... '%f'\n",x,x2);
		if (x<= asked_x && asked_x < x2) {
			//printf("found\n");
			clutter_container_remove_actor(CLUTTER_CONTAINER(clutter_actor_get_parent(actor)), actor);
			break;
		}
	}
}

guint jammo_track_view_get_sample_button_slot(JammoTrackView* track_view, JammoSampleButton* sample_button) {
	g_return_val_if_fail(clutter_actor_get_parent(CLUTTER_ACTOR(sample_button)) == tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)), (guint)-1);

	return (guint)(clutter_actor_get_x(CLUTTER_ACTOR(sample_button)) / track_view->priv->real_slot_width);
}

gboolean jammo_track_view_get_editing_enabled(JammoTrackView* track_view) {

	return track_view->priv->editing_enabled;
}

void jammo_track_view_set_editing_enabled(JammoTrackView* track_view, gboolean editing_enabled) {
	GList* children;

	if (track_view->priv->editing_enabled != editing_enabled) {
		track_view->priv->editing_enabled = editing_enabled;
		
		tangle_droppable_actor_set_dropping_enabled(TANGLE_DROPPABLE_ACTOR(track_view), editing_enabled);
		for (children = tangle_widget_get_children_readonly(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)))); children; children = children->next) {
			tangle_draggable_actor_set_dragging_enabled(TANGLE_DRAGGABLE_ACTOR(children->data), editing_enabled);
		}
		
		g_object_notify(G_OBJECT(track_view), "editing-enabled");
	}
}

static gboolean jammo_track_view_is_dragging_droppable(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {
	gboolean droppable = FALSE;
	JammoTrackView* track_view;
	gfloat center_x, center_y;
	guint slot;
	guint n_slots;
	gfloat start;
	
	track_view = JAMMO_TRACK_VIEW(droppable_actor);
	
	if (JAMMO_IS_SAMPLE_BUTTON(dragging->draggable_actor) &&
	    clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->center_x, dragging->center_y, &center_x, &center_y)) {
		n_slots = (guint)(clutter_actor_get_width(CLUTTER_ACTOR(dragging->draggable_actor)) / track_view->priv->real_slot_width + 0.5);
		center_x -= (n_slots - 1) * track_view->priv->real_slot_width / 2;
		slot = center_x / track_view->priv->real_slot_width;

		if (slot >= track_view->priv->disabled_slots_begin &&
		    slot + n_slots - 1 < track_view->priv->n_slots - track_view->priv->disabled_slots_end &&
		    !check_if_overlapping(track_view, CLUTTER_ACTOR(dragging->draggable_actor), slot)) {
			droppable = TRUE;
			start = slot * track_view->priv->real_slot_width;
			if (start != track_view->priv->hilighted_start) {
				track_view->priv->hilighted_start = start;
				track_view->priv->hilighted_end = start + n_slots * track_view->priv->real_slot_width;
				clutter_actor_queue_redraw(CLUTTER_ACTOR(track_view));
			}
		}
	}
	
	if (!droppable && track_view->priv->hilighted_start != track_view->priv->hilighted_end) {
		track_view->priv->hilighted_start = track_view->priv->hilighted_end = 0.0;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(track_view));
	}
	    		
	return droppable;
}

static gboolean jammo_track_view_handle_dragging(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {

	return FALSE;
}

static void jammo_track_view_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	JammoTrackView* track_view;
	
	track_view = JAMMO_TRACK_VIEW(actor);
	
	if (min_width_p) {
		*min_width_p = track_view->priv->n_slots * track_view->priv->slot_width;
	}
	if (natural_width_p) {
		*natural_width_p = track_view->priv->n_slots * track_view->priv->slot_width;
	}
	if (max_width_p) {
		*max_width_p = 0.0;
	}
}

static void jammo_track_view_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	JammoTrackView* track_view;

	track_view = JAMMO_TRACK_VIEW(actor);

	if (min_height_p) {
		*min_height_p = track_view->priv->slot_height;
	}
	if (natural_height_p) {
		*natural_height_p = track_view->priv->slot_height;
	}
	if (max_height_p) {
		*max_height_p = track_view->priv->slot_height;
	}
}

static void jammo_track_view_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	JammoTrackView* track_view;

	track_view = JAMMO_TRACK_VIEW(wrapper_actor);

	TANGLE_WRAPPER_ACTOR_CLASS(jammo_track_view_parent_class)->allocate_wrapped(wrapper_actor, wrapped, box, flags);
	
	track_view->priv->real_slot_width = (box->x2 - box->x1) / track_view->priv->n_slots;

	if (track_view->priv->line_every_nth_slot) {
		tangle_grid_set_grid_spacing_x(track_view->priv->grid, track_view->priv->real_slot_width * track_view->priv->line_every_nth_slot - (1.0 / track_view->priv->n_slots));
		tangle_grid_set_grid_spacing_y(track_view->priv->grid, box->y2 - box->y1 - 1.0);
	} else {
		tangle_grid_set_grid_spacing_x(track_view->priv->grid, 0.0);
		tangle_grid_set_grid_spacing_y(track_view->priv->grid, 0.0);
	}
}

static gboolean jammo_track_view_handle_dropping(TangleDroppableActor* droppable_actor, TangleDragging* dragging, gboolean accepted) {
	JammoTrackView* track_view;
	TangleWidget* widget;
	gfloat x, y, center_x, center_y;
	JammoSampleButton* sample_button;
	ClutterActor* copy_of_sample_button;
	guint slot;
	guint n_slots;
	
	track_view = JAMMO_TRACK_VIEW(droppable_actor);
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));
	
	TANGLE_DROPPABLE_ACTOR_CLASS(jammo_track_view_parent_class)->handle_dropping(droppable_actor, dragging, accepted);

	if (accepted && clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->x, dragging->y, &x, &y) &&
	    clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->center_x, dragging->center_y, &center_x, &center_y)) {
		sample_button = JAMMO_SAMPLE_BUTTON(dragging->draggable_actor);
		copy_of_sample_button = jammo_sample_button_new_from_existing(sample_button);

		n_slots = (guint)(clutter_actor_get_width(CLUTTER_ACTOR(dragging->draggable_actor)) / track_view->priv->real_slot_width + 0.5);
		center_x -= (n_slots - 1) * track_view->priv->real_slot_width / 2;
		slot = center_x / track_view->priv->real_slot_width;

		jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(JAMMO_SAMPLE_BUTTON(copy_of_sample_button)), slot * track_view->priv->slot_duration);

		/* tangle_draggable_actor_set_floating_mode_dragged(TANGLE_DRAGGABLE_ACTOR(copy_of_sample_button), TANGLE_FLOATING_MODE_COLLAPSE); */
		clutter_actor_set_x(copy_of_sample_button, x - dragging->anchor_x);
		clutter_actor_set_y(copy_of_sample_button, y - dragging->anchor_y);
		clutter_actor_set_height(copy_of_sample_button, track_view->priv->slot_height);
		tangle_widget_add(widget, copy_of_sample_button, NULL);

		clutter_actor_animate(copy_of_sample_button, CLUTTER_EASE_OUT_CUBIC, 300, "x", slot * track_view->priv->real_slot_width, "y", 0.0, NULL);

		gems_add_new_sample_to_track(12, slot); //TODO loop_id
	}
	
	track_view->priv->hilighted_start = track_view->priv->hilighted_end = -1.0;

	return FALSE;
}

static void jammo_track_view_paint(ClutterActor* actor) {
	JammoTrackView* track_view;
	gdouble scale_x;
	gdouble scale_y;
	guint8 alpha;
	ClutterActorBox box;

	track_view = JAMMO_TRACK_VIEW(actor);

	if (CLUTTER_ACTOR_IS_MAPPED(actor)) {
		CLUTTER_ACTOR_CLASS(jammo_track_view_parent_class)->paint(actor);

		if (track_view->priv->hilighted_start != track_view->priv->hilighted_end) {
			cogl_push_matrix();

			g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
			cogl_scale(scale_x, scale_y, 0);

			alpha = clutter_actor_get_paint_opacity(actor) * track_view->priv->hilight_color.alpha / 255;
			cogl_set_source_color4ub(track_view->priv->hilight_color.red,
			 			 track_view->priv->hilight_color.green,
			 			 track_view->priv->hilight_color.blue,
						 alpha);
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
			cogl_rectangle(box.x1 + track_view->priv->hilighted_start, box.y1, box.x1 + track_view->priv->hilighted_end, box.y2);

			cogl_pop_matrix();
		}
	}
}

static GObject* jammo_track_view_constructor(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties) {
	GObject* object;
	JammoTrack* track;
	ClutterActor* widget;

	g_return_val_if_fail(tangle_lookup_construct_properties(type, n_construct_properties, construct_properties, "track", &track, NULL), NULL);
	g_return_val_if_fail(track != NULL, NULL);

	widget = tangle_widget_new();

	object = tangle_construct_with_extra_properties(G_OBJECT_CLASS(jammo_track_view_parent_class)->constructor, type, n_construct_properties, construct_properties, "wrapped", widget, NULL);

	g_signal_connect(widget, "actor-added", G_CALLBACK(on_actor_added), object);
	g_signal_connect(widget, "actor-removed", G_CALLBACK(on_actor_removed), object);
	
	JAMMO_TRACK_VIEW(object)->priv->grid = TANGLE_GRID(tangle_grid_new(0.0, 0.0));
	tangle_widget_set_background_actor(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(object))), CLUTTER_ACTOR(JAMMO_TRACK_VIEW(object)->priv->grid));

	return object;
}

static void jammo_track_view_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoTrackView* track_view;
	
	track_view = JAMMO_TRACK_VIEW(object);

	switch (prop_id) {
		case PROP_TRACK:
			track_view->priv->track = JAMMO_EDITING_TRACK(g_value_get_object(value));
			g_object_ref(track_view->priv->track);
			break;
		case PROP_N_SLOTS:
			track_view->priv->n_slots = g_value_get_uint(value);
			break;
		case PROP_SLOT_DURATION:
			track_view->priv->slot_duration = g_value_get_uint64(value);
			break;
		case PROP_SLOT_HEIGHT:
			track_view->priv->slot_height = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(track_view));
			break;
		case PROP_SLOT_WIDTH:
			track_view->priv->slot_width = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(track_view));
			break;
		case PROP_LINE_EVERY_NTH_SLOT:
			track_view->priv->line_every_nth_slot = g_value_get_uint(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(track_view));
			break;
		case PROP_DISABLED_SLOTS_BEGIN:
			track_view->priv->disabled_slots_begin = g_value_get_uint(value);
			break;
		case PROP_DISABLED_SLOTS_END:
			track_view->priv->disabled_slots_end = g_value_get_uint(value);
			break;
		case PROP_EDITING_ENABLED:
			jammo_track_view_set_editing_enabled(track_view, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_track_view_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoTrackView* track_view;

	track_view = JAMMO_TRACK_VIEW(object);

        switch (prop_id) {
		case PROP_TRACK:
			g_value_set_object(value, track_view->priv->track);
			break;
		case PROP_N_SLOTS:
			g_value_set_uint(value, track_view->priv->n_slots);
			break;
		case PROP_SLOT_DURATION:
			g_value_set_uint64(value, track_view->priv->slot_duration);
			break;
		case PROP_SLOT_HEIGHT:
			g_value_set_float(value, track_view->priv->slot_height);
			break;
		case PROP_SLOT_WIDTH:
			g_value_set_float(value, (track_view->priv->real_slot_width ? track_view->priv->real_slot_width : track_view->priv->slot_width));
			break;
		case PROP_LINE_EVERY_NTH_SLOT:
			g_value_set_uint(value, track_view->priv->line_every_nth_slot);
			break;
		case PROP_DISABLED_SLOTS_BEGIN:
			g_value_set_uint(value, track_view->priv->disabled_slots_begin);
			break;
		case PROP_DISABLED_SLOTS_END:
			g_value_set_uint(value, track_view->priv->disabled_slots_end);
			break;
		case PROP_EDITING_ENABLED:
			g_value_set_boolean(value, track_view->priv->editing_enabled);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_track_view_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->finalize(object);
}

static void jammo_track_view_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->dispose(object);
}

static void jammo_track_view_class_init(JammoTrackViewClass* track_view_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(track_view_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(track_view_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(track_view_class);
	TangleWrapperActorClass* wrapper_actor_class = TANGLE_WRAPPER_ACTOR_CLASS(track_view_class);
	TangleDroppableActorClass* droppable_actor_class = TANGLE_DROPPABLE_ACTOR_CLASS(track_view_class);

	gobject_class->constructor = jammo_track_view_constructor;
	gobject_class->finalize = jammo_track_view_finalize;
	gobject_class->dispose = jammo_track_view_dispose;
	gobject_class->set_property = jammo_track_view_set_property;
	gobject_class->get_property = jammo_track_view_get_property;
	
	clutter_actor_class->paint = jammo_track_view_paint;
	
	actor_class->get_preferred_width = jammo_track_view_get_preferred_width;
	actor_class->get_preferred_height = jammo_track_view_get_preferred_height;
	
	wrapper_actor_class->allocate_wrapped = jammo_track_view_allocate_wrapped;
	
	droppable_actor_class->is_dragging_droppable = jammo_track_view_is_dragging_droppable;
	droppable_actor_class->handle_dragging = jammo_track_view_handle_dragging;
	droppable_actor_class->handle_dropping = jammo_track_view_handle_dropping;

	/**
	 * JammoTrackView:track:
	 *
	 * The track of this view.
	 */
	g_object_class_install_property(gobject_class, PROP_TRACK,
	                                g_param_spec_object("track",
	                                "Track",
	                                "The track of this view",
	                                JAMMO_TYPE_EDITING_TRACK,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:n-slots:
	 *
	 * The number of slots.
	 */
	g_object_class_install_property(gobject_class, PROP_N_SLOTS,
	                                g_param_spec_uint("n-slots",
	                                "N slots",
	                                "The number of slots",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-duration:
	 *
	 * The duration of a slot in nanoseconds.
	 */
	g_object_class_install_property(gobject_class, PROP_SLOT_DURATION,
	                                g_param_spec_uint64("slot-duration",
	                                "Slot duration",
	                                "The duration of a slot in nanoseconds",
	                                0, G_MAXUINT64, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-height:
	 *
	 * The height of the slot in the screen (pixels, Clutter units).
	 */
	g_object_class_install_property(gobject_class, PROP_SLOT_HEIGHT,
	                                g_param_spec_float("slot-height",
	                                "Slot height",
	                                "The height of the slot in the screen (pixels, Clutter units)",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-width:
	 *
	 * The width of the slot in the screen (pixels, Clutter units).
	 */
	g_object_class_install_property(gobject_class, PROP_SLOT_WIDTH,
	                                g_param_spec_float("slot-width",
	                                "Slot width",
	                                "The width of the slot in the screen (pixels, Clutter units)",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:line-every-nth-slot:
	 *
	 * The number of slots after a separator line is drawn (0 to disable).
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_EVERY_NTH_SLOT,
	                                g_param_spec_uint("line-every-nth-slot",
	                                "Line every nth slot",
	                                "The number of slots after a separator line is drawn (0 to disable)",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:disabled-slots-begin:
	 *
	 * The number of slots that are disabled (in terms of dropping) from the beginning of the track.
	 */
	g_object_class_install_property(gobject_class, PROP_DISABLED_SLOTS_BEGIN,
	                                g_param_spec_uint("disabled-slots-begin",
	                                "Disabled slots begin",
	                                "The number of slots that are disabled (in terms of dropping) from the beginning of the track",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:disabled-slots-end:
	 *
	 * The number of slots that are disabled (in terms of dropping) from the end of the track.
	 */
	g_object_class_install_property(gobject_class, PROP_DISABLED_SLOTS_END,
	                                g_param_spec_uint("disabled-slots-end",
	                                "Disabled slots end",
	                                "The number of slots that are disabled (in terms of dropping) from the end of the track",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:editing enabled:
	 *
	 * Whether the track can be edited or not.
	 */
	g_object_class_install_property(gobject_class, PROP_EDITING_ENABLED,
	                                g_param_spec_boolean("editing-enabled",
	                                "Editing enabled",
	                                "Whether the track can be edited or not",
	                                TRUE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (JammoTrackViewPrivate));
}

static void jammo_track_view_init(JammoTrackView* track_view) {
	track_view->priv = G_TYPE_INSTANCE_GET_PRIVATE(track_view, JAMMO_TYPE_TRACK_VIEW, JammoTrackViewPrivate);
	track_view->priv->editing_enabled = TRUE;

	track_view->priv->hilight_color.red = 255;
	track_view->priv->hilight_color.green = 255;
	track_view->priv->hilight_color.blue = 255;
	track_view->priv->hilight_color.alpha = 100;

	track_view->priv->hilighted_start = track_view->priv->hilighted_end = -1.0;
}

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;
	JammoSample* sample;

	g_return_if_fail(JAMMO_IS_SAMPLE_BUTTON(actor));
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	sample = jammo_sample_button_get_sample(sample_button);
	
	g_assert(jammo_sample_get_editing_track(sample));
}

static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;

	g_return_if_fail(JAMMO_IS_SAMPLE_BUTTON(actor));
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	
	jammo_editing_track_remove_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button));

	guint slot = clutter_actor_get_x(actor) / track_view->priv->slot_width;
	gems_remove_sample_from_slot(slot);
}

static gboolean check_if_overlapping(JammoTrackView* track_view, ClutterActor* sample_button, guint slot_index) {
	TangleWidget* widget;
	guint n_slots;
	GList* children;
	ClutterActor* actor = NULL;
	gfloat x;
	
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));
	n_slots = (guint)(clutter_actor_get_width(sample_button) / track_view->priv->real_slot_width + 0.5);
	for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
		actor = CLUTTER_ACTOR(children->data);
		x = clutter_actor_get_x(actor);
		if (actor != sample_button &&
		    x < slot_index * track_view->priv->real_slot_width &&
		    x + clutter_actor_get_width(actor) > slot_index * track_view->priv->real_slot_width) {
		    n_slots += slot_index - (guint)(x / track_view->priv->real_slot_width);
			slot_index = (guint)(x / track_view->priv->real_slot_width);
			break;
		}
		actor = NULL;
	}
	if (!actor) {
		for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
			actor = CLUTTER_ACTOR(children->data);
			x = clutter_actor_get_x(actor);
			if (actor != sample_button &&
			    x >= slot_index * track_view->priv->real_slot_width &&
			    x <= (slot_index + n_slots - 1) * track_view->priv->real_slot_width) {
				n_slots -= (guint)(x / track_view->priv->real_slot_width) - slot_index;
				break;
			}
			actor = NULL;
		}
	}
	
	return actor != NULL;
}
