/*
 * jammo-recording-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-recording-track.h"
#include "jammo-meam-private.h"
#include <string.h>

G_DEFINE_TYPE(JammoRecordingTrack, jammo_recording_track, JAMMO_TYPE_TRACK);

enum {
	PROP_0,
	PROP_PITCH_DETECT,
	PROP_FILENAME
};

enum {
	PITCH_DETECTED,
	LAST_SIGNAL
};

struct _JammoRecordingTrackPrivate {
	GstElement* bin;
	gboolean pitch_detect;
	gchar* filename;
};

static guint signals[LAST_SIGNAL] = { 0 };

JammoRecordingTrack* jammo_recording_track_new(const gchar* filename) {

	return JAMMO_RECORDING_TRACK(g_object_new(JAMMO_TYPE_RECORDING_TRACK, "filename", filename, NULL));
}

JammoRecordingTrack* jammo_recording_track_new_with_pitch_detect(const gchar* filename) {

	return JAMMO_RECORDING_TRACK(g_object_new(JAMMO_TYPE_RECORDING_TRACK, "filename", filename, "pitch-detect", TRUE, NULL));
}

static void jammo_recording_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoRecordingTrack* recording_track;
	
	recording_track = JAMMO_RECORDING_TRACK(object);

	switch (prop_id) {
		case PROP_PITCH_DETECT:
			recording_track->priv->pitch_detect = g_value_get_boolean(value);
			break;
		case PROP_FILENAME:
			g_assert(!recording_track->priv->filename);
			recording_track->priv->filename = g_strdup(g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_recording_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoRecordingTrack* recording_track;

	recording_track = JAMMO_RECORDING_TRACK(object);

        switch (prop_id) {
		case PROP_PITCH_DETECT:
			g_value_set_boolean(value, recording_track->priv->pitch_detect);
			break;
		case PROP_FILENAME:
			g_value_set_string(value, recording_track->priv->filename);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static guint64 jammo_recording_track_get_duration(JammoTrack* track) {
	guint64 duration = 0;

	/* TODO: Get real duration. */

	return duration;
}

static GstElement* jammo_recording_track_get_element(JammoTrack* track) {

	return JAMMO_RECORDING_TRACK(track)->priv->bin;
}

static void jammo_recording_track_receive_message(JammoTrack* track, GstMessage* message) {
	const GstStructure* structure;
	gfloat frequency;
	
	structure = gst_message_get_structure(message);
	if (structure && !strcmp(gst_structure_get_name(structure), "pitchdetect")) {
		frequency = g_value_get_float(gst_structure_get_value(structure, "frequency"));
		g_signal_emit(track, signals[PITCH_DETECTED], 0, frequency);
	}
}

static GObject* jammo_recording_track_constructor(GType type, guint n_properties, GObjectConstructParam* properties) {
	GObject* object;
	JammoRecordingTrack* recording_track;
	GstElement* source;
	GstElement* convert;
	GstElement* pitch = NULL;
	GstElement* pitch_convert = NULL;
	GstElement* encoder;
	GstElement* sink;
	
	object = G_OBJECT_CLASS(jammo_recording_track_parent_class)->constructor(type, n_properties, properties);

	recording_track = JAMMO_RECORDING_TRACK(object);
	recording_track->priv->bin = gst_bin_new(NULL);
	source = gst_element_factory_make("autoaudiosrc", NULL);
	convert = gst_element_factory_make("audioconvert", NULL);
	if (recording_track->priv->pitch_detect) {
		if (!(pitch = gst_element_factory_make("pitchdetect", NULL))) {
			g_warning("Element type 'pitchdetect' is not available. Disabling pitch detecting.");
			recording_track->priv->pitch_detect = FALSE;
		} else {
			pitch_convert = gst_element_factory_make("audioconvert", NULL);
		}
	}
	encoder = gst_element_factory_make("wavenc", NULL);
	sink = gst_element_factory_make("filesink", NULL);
	g_assert(recording_track->priv->filename);
	g_object_set(sink, "location", recording_track->priv->filename, NULL);
	if (recording_track->priv->pitch_detect) {
		gst_bin_add_many(GST_BIN(recording_track->priv->bin), source, convert, pitch, pitch_convert, encoder, sink, NULL);
		gst_element_link_many(source, convert, pitch, pitch_convert, encoder, sink, NULL);
	} else {
		gst_bin_add_many(GST_BIN(recording_track->priv->bin), source, convert, encoder, sink, NULL);
		gst_element_link_many(source, convert, encoder, sink, NULL);
	}	

	return object;
}


static void jammo_recording_track_finalize(GObject* object) {
	JammoRecordingTrack* recording_track;
	
	recording_track = JAMMO_RECORDING_TRACK(object);
	g_free(recording_track->priv->filename);

	G_OBJECT_CLASS(jammo_recording_track_parent_class)->finalize(object);
}

static void jammo_recording_track_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_recording_track_parent_class)->dispose(object);
}

static void jammo_recording_track_class_init(JammoRecordingTrackClass* recording_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(recording_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(recording_track_class);

	track_class->get_duration = jammo_recording_track_get_duration;
	track_class->get_element = jammo_recording_track_get_element;
	track_class->receive_message = jammo_recording_track_receive_message;
	gobject_class->constructor = jammo_recording_track_constructor;
	gobject_class->finalize = jammo_recording_track_finalize;
	gobject_class->dispose = jammo_recording_track_dispose;
	gobject_class->set_property = jammo_recording_track_set_property;
	gobject_class->get_property = jammo_recording_track_get_property;

	/**
	 * JammoRecordingTrack:pitch-detect:
	 */
	g_object_class_install_property(gobject_class, PROP_PITCH_DETECT,
	                                g_param_spec_boolean("pitch-detect",
	                                "Pitch detect",
	                                "Specifies whether pitch detecting is activated or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoRecordingTrack:filename:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "File name",
	                                "The file name of the recorded audio (wav)",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * JammoRecordingTrack::pitch-detected:
	 * @track: the object which received the signal
	 *
	 * The ::pitch-detected signal is emitted when the recording track has calculated
	 * the frequency of the recording.
	 */
	signals[PITCH_DETECTED] = g_signal_new("pitch-detected", G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST, 0,
					       NULL, NULL,
					       g_cclosure_marshal_VOID__FLOAT,
					       G_TYPE_NONE, 1,
					       G_TYPE_FLOAT);

	g_type_class_add_private(gobject_class, sizeof(JammoRecordingTrackPrivate));
}

static void jammo_recording_track_init(JammoRecordingTrack* recording_track) {
	recording_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(recording_track, JAMMO_TYPE_RECORDING_TRACK, JammoRecordingTrackPrivate);
}
