/*
 * jammo-editing-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-editing-track.h"
#include "jammo-meam.h"
#include "jammo-meam-private.h"

G_DEFINE_TYPE(JammoEditingTrack, jammo_editing_track, JAMMO_TYPE_TRACK);

enum {
	PROP_0,
};

enum {
	SAMPLE_ADDED,
	SAMPLE_REMOVED,
	LAST_SIGNAL
};

struct _JammoEditingTrackPrivate {
	GstElement* bin;
	GstElement* composition;
	GList* samples;
};

static void link_compatible_pads(GstElement* source_element, GstPad* source_pad, GstElement* sink_element);
static void on_duration_notify(GObject* object, GParamSpec* param_spec, gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };

JammoEditingTrack* jammo_editing_track_new() {

	return JAMMO_EDITING_TRACK(g_object_new(JAMMO_TYPE_EDITING_TRACK, NULL));
}

void jammo_editing_track_add_sample(JammoEditingTrack* editing_track, JammoSample* sample, guint64 position) {
	JammoEditingTrack* previous_track;
	GList* list;
	GstElement* element;
	guint64 start;
	gboolean handled;
	
	g_object_ref_sink(sample);
	
	if ((previous_track = jammo_sample_get_editing_track(sample))) {
		jammo_editing_track_remove_sample(previous_track, sample);
	}
	
	for (list = editing_track->priv->samples; list; list = list->next) {
		element = _jammo_sample_get_element(JAMMO_SAMPLE(list->data));
		g_object_get(element,
		             "start", &start,
			     NULL);
		if (start >= position) {
			break;
		}
	}
	
	editing_track->priv->samples = g_list_insert_before(editing_track->priv->samples, list, sample);
	element = _jammo_sample_get_element(sample);
	g_object_set(element,
        	     "start", position,
		     NULL);
	gst_bin_add(GST_BIN(editing_track->priv->composition), element);
	_jammo_sample_set_editing_track(sample, editing_track);
	
	g_signal_emit(editing_track, signals[SAMPLE_ADDED], 0, sample, &handled);
	
	g_signal_connect(sample, "notify::duration", G_CALLBACK(on_duration_notify), editing_track);
	g_object_notify(G_OBJECT(editing_track), "duration");
}

void jammo_editing_track_remove_sample(JammoEditingTrack* editing_track, JammoSample* sample) {
	gboolean handled;
	
	editing_track->priv->samples = g_list_remove(editing_track->priv->samples, sample);
	gst_bin_remove(GST_BIN(editing_track->priv->composition), _jammo_sample_get_element(sample));
	_jammo_sample_set_editing_track(sample, NULL);

	g_signal_emit(editing_track, signals[SAMPLE_REMOVED], 0, sample, &handled);

	g_object_unref(sample);
}

JammoSample* jammo_editing_track_get_sample(JammoEditingTrack* editing_track, guint64 position) {
	JammoSample* sample = NULL;
	guint64 p;
	GList* list;
	JammoSample* s;
	guint64 duration;
	
	p = 0;
	for (list = editing_track->priv->samples; list; list = list->next) {
		s = JAMMO_SAMPLE(list->data);
		duration = jammo_sample_get_duration(s);
		if (p + duration >= position) {
			sample = s;
			break;
		}
		p += duration;
	}
	
	return sample;
}

guint64 jammo_track_get_sample_position(JammoEditingTrack* editing_track, JammoSample* sample) {
	guint64 position = (guint64)-1;
	guint64 p;
	GList* list;
	JammoSample* s;
	guint64 duration;
	
	p = 0;
	for (list = editing_track->priv->samples; list; list = list->next) {
		s = JAMMO_SAMPLE(list->data);
		duration = jammo_sample_get_duration(s);
		if (s == sample) {
			position = p;
			break;
		}
		p += duration;
	}
	
	return position;

}

static guint64 jammo_editing_track_get_duration(JammoTrack* track) {
	guint64 duration = 0;
	JammoEditingTrack* editing_track;
	GList* list;
	guint64 d;
	
	editing_track = JAMMO_EDITING_TRACK(track);
	for (list = editing_track->priv->samples; list; list = list->next) {
		if ((d = jammo_sample_get_duration(JAMMO_SAMPLE(list->data))) == JAMMO_DURATION_INVALID) {
			duration = JAMMO_DURATION_INVALID;
			break;
		}
		duration += d;
	}

	return duration;
}

static GstElement* jammo_editing_track_get_element(JammoTrack* track) {

	return JAMMO_EDITING_TRACK(track)->priv->bin;
}

static void jammo_editing_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoEditingTrack* track;
	
	track = JAMMO_EDITING_TRACK(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_editing_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoEditingTrack* editing_track;

	editing_track = JAMMO_EDITING_TRACK(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_editing_track_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_editing_track_parent_class)->finalize(object);
}

static void jammo_editing_track_dispose(GObject* object) {
	JammoEditingTrack* editing_track;

	editing_track = JAMMO_EDITING_TRACK(object);
	g_list_foreach(editing_track->priv->samples, (GFunc)_jammo_sample_set_editing_track, NULL);
	g_list_foreach(editing_track->priv->samples, (GFunc)g_object_unref, NULL);
	g_list_free(editing_track->priv->samples);
	editing_track->priv->samples = NULL;

	G_OBJECT_CLASS(jammo_editing_track_parent_class)->dispose(object);
}

static void jammo_editing_track_class_init(JammoEditingTrackClass* editing_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(editing_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(editing_track_class);

	track_class->get_duration = jammo_editing_track_get_duration;
	track_class->get_element = jammo_editing_track_get_element;
	gobject_class->finalize = jammo_editing_track_finalize;
	gobject_class->dispose = jammo_editing_track_dispose;
	gobject_class->set_property = jammo_editing_track_set_property;
	gobject_class->get_property = jammo_editing_track_get_property;

	/**
	 * JammoEditingTrack:sample-added:
	 * @editing_track: the object which received the signal
	 */
	signals[SAMPLE_ADDED] = g_signal_new("sample-added", G_TYPE_FROM_CLASS(gobject_class),
	                                     G_SIGNAL_RUN_LAST, 0,
					     NULL, NULL,
					     g_cclosure_marshal_VOID__OBJECT,
					     G_TYPE_NONE, 1,
					     JAMMO_TYPE_SAMPLE);
	/**
	 * JammoEditingTrack:sample-added:
	 * @editing_track: the object which received the signal
	 */
	signals[SAMPLE_REMOVED] = g_signal_new("sample-removed", G_TYPE_FROM_CLASS(gobject_class),
	                                       G_SIGNAL_RUN_LAST, 0,
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       JAMMO_TYPE_SAMPLE);

	g_type_class_add_private(gobject_class, sizeof(JammoEditingTrackPrivate));
}

static void jammo_editing_track_init(JammoEditingTrack* editing_track) {
	GstElement* source;
	GstElement* element;
	GstElement* convert;
	GstPad* pad;

	editing_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(editing_track, JAMMO_TYPE_EDITING_TRACK, JammoEditingTrackPrivate);
	
	/*
	   +--bin----------------------------------------------------------+
	   | +--gnlcomposition----------------+                            |
	   | | +--gnlsource-------------+     |                            |
	   | | | +--audiotestsrc--+     |     |     +--audioconvert--+     |
	   | | | | wave = 4       O =a> O =a> O =s> O                O =g> O
	   | | | +----------------+     |     |     +----------------+     |
	   | | +------------------------+     |                            |
	   | +--------------------------------+                            |
	   +---------------------------------------------------------------+
	   
	 =a> automatically: done by the gnl (behind the scenes)
	 =g> ghostpad: created a ghostpad (gst_ghost_pad_new)
	 =s> signalled: linked asyncronously ("pad-added" signal)
	 =m> manually: linked syncronously (gst_element_link)
	*/
	
	editing_track->priv->bin = gst_bin_new(NULL);
	editing_track->priv->composition = gst_element_factory_make("gnlcomposition", NULL);
	convert = gst_element_factory_make("audioconvert", NULL);
	gst_bin_add_many(GST_BIN(editing_track->priv->bin), editing_track->priv->composition, convert, NULL);
	g_signal_connect(editing_track->priv->composition, "pad-added", G_CALLBACK(link_compatible_pads), convert);

	/* TODO: There must be at least one glnsource with priority != -1 in order to get pad. */

	pad = gst_element_get_pad(convert, "src");
	gst_element_add_pad(editing_track->priv->bin, gst_ghost_pad_new("src", pad));
	gst_object_unref(pad);

	source = gst_element_factory_make("gnlsource", NULL);
	g_object_set(source,
	             "priority", -1,
		     "start", 0 * GST_SECOND,
		     "duration", 1000 * GST_SECOND,
		     "media-start", 0 * GST_SECOND,
		     "media-duration", 1000 * GST_SECOND,
		     NULL);	
	element = gst_element_factory_make("audiotestsrc", NULL);
	g_object_set(element,
	             "wave", 4, /* GST_AUDIO_TEST_SRC_WAVE_SILENCE */
		     NULL);
	gst_bin_add(GST_BIN(source), element);
	gst_bin_add(GST_BIN(editing_track->priv->composition), source);
}

static void link_compatible_pads(GstElement* source_element, GstPad* source_pad, GstElement* sink_element) {
	GstPad* sink_pad;
	
	if (!(sink_pad = gst_element_get_compatible_pad(sink_element, source_pad, gst_pad_get_caps(source_pad)))) {
		g_warning("No compatible pads found.\n");
	} else {
		gst_pad_link(source_pad, sink_pad);
		gst_object_unref(sink_pad);
	}
}

static void on_duration_notify(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	g_object_notify(G_OBJECT(user_data), "duration");
}
