/*
 * gems_structures.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <arpa/inet.h>
#include "gems_structures.h"
#include "groupmanager.h"
#include "gems_profile_manager.h"
#include "../cem/cem.h"

// host float to guint32 for sending over network
guint32 htonf(gfloat value)
{
	guint32 packed;
	guint32 sign;

	if (value < 0)
	{
		sign = 1;
		value = -value;
	}
	else
	{
		sign = 0;
	}
        
	packed = ((((guint32)value)&0x7fff)<<16) | (sign<<31); // whole part and sign
	packed |= (guint32)(((value - (gint)value) * 65536.0f))&0xffff; // fraction
	return packed;
}

// network guint32 to host float for receiving from network
gfloat ntohf(guint32 value)
{
	gfloat ret;
  ret = ((value>>16)&0x7fff); // whole part
  ret += (value&0xffff) / 65536.0f; // fraction

  if (((value>>31)&0x1) == 0x1) { ret = -ret; } // sign bit set
  
  return ret;
}

void gems_clear_gems_connection(gems_connection* element)
{
	gchar* logmsg = NULL;
	if(!element) return;
	
	if(element->connection != NULL)
	{
		if(element->profile != NULL)
		{
			/* TODO add to protocol: a notify from group member to owner if a disconnected member is detected
			   by some other than group owner -> group owner should check the connection */
			// Group is active and we are the owner
			if(gems_group_active() && (gems_group_get_owner(NULL) == gems_profile_manager_get_userid(NULL)))
			{
				// If disconnecting user is in our group
				if(gems_group_is_in_group(element->profile->id)) gems_group_disconnected_member_detected(element->profile->id);
			}
		}
		
		if(ph_c_connection_disconnect(element->connection) == TRUE)
		{
			logmsg = g_strdup_printf("Disconnected device %d", ph_c_connection_get_device_checksum(element->connection));
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
		}
		ph_c_connection_delete(element->connection);
	}
	
	if(element->game_connection != NULL)
	{
		if(ph_c_connection_disconnect(element->game_connection) == TRUE)
		{
			logmsg = g_strdup_printf("Disconnected game connection");
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
		}
		ph_c_connection_delete(element->game_connection);
	}
	
	memset(&element->nwbuffer,'\0',JAMMO_NETWORK_BUFFER_MAX_SIZE);
	
	gems_clear_peer_profile(element->profile);
	
	g_timer_destroy(element->profile_request_timer);
	g_timer_destroy(element->group_request_timer);
	
	if(element->last_action != NULL) g_timer_destroy(element->last_action);
	
	g_free(element);
	element = NULL;
}

gems_connection* gems_new_gems_connection(MAbstractConnection* newconn,
	unsigned short newport,
	int newconnectionid,
	jammo_state newstate,
	gint32 newjammoid)
{
	gems_connection* new_item = (gems_connection*)g_malloc(sizeof(gems_connection));
	
	new_item->connection = newconn;
	new_item->game_connection = NULL;
	new_item->port = newport;
	new_item->connectionid = newconnectionid;
	new_item->connection_state = newstate;
	new_item->jammoid = newjammoid;
	new_item->profile = NULL;
	new_item->profile_requests = 0;
	new_item->profile_request_timer = g_timer_new();
	new_item->group_request_timer = g_timer_new();
	new_item->last_action = NULL;
	
	gems_connection_clear_buffer(new_item);
	
	return new_item;
}

// Clear the networking buffer
void gems_connection_clear_buffer(gems_connection* element)
{
	memset(&element->nwbuffer,'\0',JAMMO_NETWORK_BUFFER_MAX_SIZE);
	element->add_position = 0;
}

void gems_connection_set_action(gems_connection* element)
{
	if(element->last_action == NULL) element->last_action = g_timer_new();
	else g_timer_start(element->last_action);
}

double gems_connection_get_last_action(gems_connection* element)
{
	// Wasn't set, set now
	if(element->last_action == NULL) gems_connection_set_action(element);
	return g_timer_elapsed(element->last_action,NULL);
}

// Add 16bit integer to network buffer
void gems_connection_add_16(gems_connection* element, guint16 value)
{
	// TODO report / handle if buffer exceeded
	if((element->add_position + sizeof(gint16)) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	*(guint16*)&element->nwbuffer[element->add_position] = value;
	element->add_position = element->add_position + sizeof(guint16);
}

// Add 32bit integer to network buffer
void gems_connection_add_32(gems_connection* element, guint32 value)
{
	// TODO report / handle if buffer exceeded
	if((element->add_position + sizeof(guint32)) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	*(guint32*)&element->nwbuffer[element->add_position] = value;
	element->add_position = element->add_position + sizeof(guint32);
}

// Add 64bit integer to network buffer
void gems_connection_add_64(gems_connection* element, guint64 value)
{
	// TODO report / handle if buffer exceeded
	if((element->add_position + sizeof(guint64)) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	*(guint64*)&element->nwbuffer[element->add_position] = value;
	element->add_position = element->add_position + sizeof(guint64);
}

// Add 32bit float to network buffer. This can be done because message has been received and
// it will not sent anymore
void gems_connection_add_float(gems_connection* element, gfloat value)
{
	// TODO report / handle if buffer exceeded
	if((element->add_position + sizeof(gfloat)) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	*(gfloat*)&element->nwbuffer[element->add_position] = value;
	element->add_position = element->add_position + sizeof(gfloat);
}

// Add characters to network buffer, define need for padding with last value (pad)
void gems_connection_add_char(gems_connection* element, gchar* value, gint length, gboolean pad)
{
	// TODO report / handle if buffer exceeded
	if((element->add_position + length) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	g_strlcat(&element->nwbuffer[element->add_position], value, element->add_position + length);
	element->add_position = element->add_position + length;
	
	// Adding partial read, erase the \0 character
	if(pad == FALSE) element->add_position = element->add_position - 1;
}

// Add padding - a NULL character
void gems_connection_add_padding(gems_connection* element)
{
	if((element->add_position + 1) >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	element->nwbuffer[element->add_position] = '\0';
	element->add_position = element->add_position + 1;
}

// Add data to the end of nwbuffer
void gems_connection_add_data(gems_connection* element, gchar* data, gint length)
{
	if(element->add_position + length >= JAMMO_NETWORK_BUFFER_MAX_SIZE) return;
	
	memcpy(&element->nwbuffer[element->add_position],data,length);
	element->add_position = element->add_position + length;
}

// Get 16 bit integer from position
guint16 gems_connection_get_16(gems_connection* element, gint from)
{
	if(from > element->add_position) return 0;
	
	return *(guint16*)&element->nwbuffer[from];
}

// Get 32 bit integer from position
guint32 gems_connection_get_32(gems_connection* element, gint from)
{
	if(from > element->add_position) return 0;
	
	return *(guint32*)&element->nwbuffer[from];
}

// Get 64 bit integer from position
guint64 gems_connection_get_64(gems_connection* element, gint from)
{
	if(from > element->add_position) return 0;
	
	return *(guint64*)&element->nwbuffer[from];
}

// Get 32 bit float from position
gfloat gems_connection_get_float_for_localization(gems_connection* element, gint from)
{
	gfloat ret;
	guint32 value;

	if(from > element->add_position) return 0;
	
	value = *(guint32*)&element->nwbuffer[from];
	 
  ret = ntohf(value);

  return ret;
}

// Get 32 bit float from position
gfloat gems_connection_get_float(gems_connection* element, gint from)
{
	gfloat value;

	if(from > element->add_position) return 0;
	
	value = *(gfloat*)&element->nwbuffer[from];

  return value;
}

// Get characters from position until next NULL character
gchar* gems_connection_get_char(gems_connection* element, gint from)
{
	// TODO do a real check - copy until \0 character is found !
	return (gchar*)&element->nwbuffer[from];
}

// Get data as chars from position until lenght is reached. Returns new gchar pointer, remember to free.
gchar* gems_connection_get_data(gems_connection* element, gint from, gint length)
{
	gchar* buffer = (gchar*)g_malloc(sizeof(gchar*)*length);
	memset(buffer,'\0',length);

	memcpy(buffer, &element->nwbuffer[from],length);
		
	return buffer;
}

// Copies the content in newcontent to nwbuffer of element
void gems_connection_set_content(gems_connection* element, gchar* newcontent, gint length)
{
	gems_connection_clear_buffer(element);
	memcpy(&element->nwbuffer,newcontent,length);
}

// Get the amount of partial data in the buffer
gint gems_connection_partial_data_amount(gems_connection* element)
{
	return element->add_position;
}

// Clear all disconnected gems_connection elements from list
GList* gems_cleanup_connections_from_list(GList* list)
{
	GList* iterator = g_list_first(list);
	GList* tempiter = NULL;
	
	while(iterator)
	{
		// Disconnected from our end ?
		if(ph_c_connection_is_connected((MAbstractConnection*)((gems_connection*)iterator->data)->connection) == FALSE)
		{
			tempiter = iterator;
			iterator = g_list_next(iterator);
			
			list = g_list_remove_link(list,tempiter);
			gems_clear_gems_connection((gems_connection*)tempiter->data);
			g_list_free_1(tempiter);
			gchar* logmsg = g_strdup_printf("gems_cleanup_connections_from_list: erased one");
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
			continue;
		}
	
		iterator = g_list_next(iterator);
	}
	return list;
}

gems_peer_profile* gems_new_peer_profile(gchar* username, guint32 id, guint16 age, guint16 avatarid)
{
	gems_peer_profile* profile = (gems_peer_profile*)g_malloc(sizeof(gems_peer_profile));
	
	profile->id = id;
	profile->username = g_strdup(username);
	profile->avatarid = avatarid;
	profile->age = age;
	
	return profile;
}

void gems_peer_profile_set_values(gems_peer_profile* profile, gchar* username, guint32 id, guint16 age, guint16 avatarid)
{
	profile->id = id;
	profile->username = g_strdup(username);
	profile->avatarid = avatarid;
	profile->age = age;
}

void gems_clear_peer_profile(gems_peer_profile* profile)
{
	if(profile == NULL) return;
	g_free(profile->username);
	profile->username = NULL;
	g_free(profile);
	profile = NULL;
}

void gems_connection_localize_data_in_buffer(gems_connection* element)
{
	gchar* logmsg = NULL;
	int temp_position = element->add_position;
	// first 8 bytes should be always the same - even with error messages
	element->add_position = 0;
	gems_connection_add_16(element,g_ntohs(gems_connection_get_16(element,element->add_position))); // pos should be 0
	gems_connection_add_32(element,g_ntohl(gems_connection_get_32(element,element->add_position))); // pos should be 2
	gems_connection_add_16(element,g_ntohs(gems_connection_get_16(element,element->add_position))); // pos should be 6
	
	switch(gems_connection_get_16(element,0))
	{
		case ERROR_MSG:
			// convert error type at 8
			gems_connection_add_16(element,g_ntohs(gems_connection_get_16(element,element->add_position))); // pos should be 8
			// TODO convert the error message (chars from 10->)
			break;
		case JAMMO_SERVICE_ID:
			logmsg = g_strdup_printf("gems_connection_localize_data_in_buffer: got JAMMO_SERVICE_ID - should not happen!!");
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
			break;
		case AA_SERVICE_ID:
			break;
		case PROFILE_SERVICE_ID:
			switch(gems_connection_get_16(element,6)) // the service command from position 6
			{
				case GET_PROFILE:
					// request type, add_position should be 8 and type should be in position 8 as 16bit int
					gems_connection_add_16(element,g_ntohs(gems_connection_get_16(element,element->add_position)));
					break;
				case PROFILE_PUB:
					// TODO convert username
					// Username -> add_position = add_position + Username length + tailing '\0'
					element->add_position = element->add_position + strlen(gems_connection_get_char(element,element->add_position)) + 1;
					// User id
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element,element->add_position)));
					// age
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element,element->add_position)));
					// avatar id
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element,element->add_position)));
					break;
				case PROFILE_FULL:
					break;
				default:
					break;
			}
		case GROUP_SERVICE_ID:
			switch(gems_connection_get_16(element,6)) // command
			{
				case REQUEST_MEMBERS:
					break;
				case REQUEST_GROUP_INFO:
					break;
				case OFFER_GROUP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					// TODO add group lock type handling
					break;
				case CURRENT_GROUP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					// TODO add group lock type handling
					break;
				case JOIN_GROUP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case MEMBER_LIST:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case NEW_MEMBER:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case LEAVE_GROUP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case REMOVED_FROM_GROUP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case MEMBER_DROPPED:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				default:
					break;
			}
			break;
		case COLLABORATION_SERVICE_ID:
			switch (gems_connection_get_16(element,6)) {
				case ACTION_LOOP:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					break;
				case ACTION_LOOP_SYNC:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					for (; element->add_position < gems_connection_get_32(element, 2);) {
						// loop id
						gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
						// slot
						gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					}
					break;
				case ACTION_MIDI:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					for (; element->add_position < gems_connection_get_32(element, 2);) {
						// note is a char. no need to do anything
						element->add_position += sizeof(unsigned char);
						// timestamp
						gems_connection_add_64(element, GUINT64_FROM_BE(gems_connection_get_64(element, element->add_position)));
						// type
						gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					}
					break;
				case ACTION_SLIDER:
					gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					gems_connection_add_16(element, g_ntohs(gems_connection_get_16(element, element->add_position)));
					for (; element->add_position < gems_connection_get_32(element, 2);) {
						// freq is float. it is received as an 32bit integer and localized to a float
						// see add float and get float functions for details
						gems_connection_add_float(element, gems_connection_get_float_for_localization(element, element->add_position));
						// timestamp
						gems_connection_add_64(element, GUINT64_FROM_BE(gems_connection_get_64(element, element->add_position)));
						// type
						gems_connection_add_32(element, g_ntohl(gems_connection_get_32(element, element->add_position)));
					}
					break;
				default:
					break;
			}
			break;
		case SONGBANK_SERVICE_ID:
			break;
		case DATATRANSFER_SERVICE_ID:
			break;
		case MENTOR_SERVICE_ID:
			break;
		case CONTROL_SERVICE_ID:
			break;
		case PROBE_SERVICE_ID:
			break;
		default:
			break;
	}
	
	element->add_position = temp_position;
}

GList* gems_connection_error_add(GList* list, guint device_checksum, guint errortype)
{
	gchar* logmsg = NULL;
	gems_connection_error* item = NULL;
	
	// Not found -> create new and add
	if((item = gems_connection_error_get(list,device_checksum)) == NULL)
	{
		item = (gems_connection_error*)g_malloc(sizeof(gems_connection_error));
		item->checksum = device_checksum;
		
		if(errortype == ERROR_CONNECTION) item->connection_errors = 1;
		else item->connection_errors = 0;
		
		if(errortype == ERROR_REJECTED) item->version_mismatch = TRUE;
		else item->version_mismatch = FALSE;
		
		if(errortype == ERROR_ALREADY_CONNECTED) item->already_connected = TRUE;
		else item->already_connected = FALSE;
		
		item->connection_timer = g_timer_new();
		g_timer_start(item->connection_timer);
		list = g_list_prepend(list,item);
		logmsg = g_strdup_printf("gems_connection_error_add: added new device %d, list = %d", item->checksum, g_list_length(list));
		cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
	else // found
	{
		switch(errortype)
		{
			case ERROR_CONNECTION:
				// Limit reached, do not add since the delay is 1min
				if(item->connection_errors < CONNECTION_ERRORLIMIT)	item->connection_errors += 1; // Add a error
				break;
			case ERROR_REJECTED:
				item->version_mismatch = TRUE;
				break;
			case ERROR_ALREADY_CONNECTED:
				item->already_connected = TRUE;
				break;
			default:
				break;
		}
		g_timer_start(item->connection_timer);
		logmsg = g_strdup_printf("gems_connection_error_add: added to existing device %d (errors : %d)", item->checksum, item->connection_errors);
		cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
	return list;
}

gems_connection_error* gems_connection_error_get(GList* list, guint device_checksum)
{
	GList* iterator = NULL;
	if(g_list_length(list) == 0) return NULL;
	
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection_error* element = (gems_connection_error*)iterator->data;
		
		if(element == NULL) continue;
		
		// Match - return element
		if(element->checksum == device_checksum) return element;
	}
	return NULL;	
}

guint gems_connection_error_get_connection_error_count(GList* list, guint device_checksum)
{
	gems_connection_error* item = gems_connection_error_get(list,device_checksum);
	if(item == NULL) return 0;
	else return item->connection_errors;
}

gboolean gems_connection_error_attempt_connection(GList* list, guint device_checksum)
{
	gems_connection_error* item = gems_connection_error_get(list,device_checksum);
	gdouble time = 0.0;
	
	if(item == NULL)
	{
		//logmsg = g_strdup_printf("gems_connection_error_attempt_connection: no device yet: %d", device_checksum);	
		return TRUE; // Not set
	}
	else
	{
		// reached errorlimit -> longer interval
		if(item->connection_errors > CONNECTION_ERRORLIMIT)	time = CONNECTION_INTERVAL * 10 + item->connection_errors;
		// Each error gives 1s more delay
		else time = item->connection_errors + CONNECTION_INTERVAL;
		
		//logmsg = g_strdup_printf("gems_connection_error_attempt_connection: checking time for %d, time = %f, errors = %d", item->checksum, time, item->connection_errors);
		
		if(item->version_mismatch) return FALSE;
		if(item->already_connected) return FALSE;
		
		// less than 'time' elapsed?
		if(g_timer_elapsed(item->connection_timer,NULL) < time)
		{
			//logmsg = g_strdup_printf("gems_connection_error_attempt_connection: %f passed ", g_timer_elapsed(item->connection_timer,NULL));
			return FALSE;
		}
		else
		{
			//logmsg = g_strdup_printf("gems_connection_error_attempt_connection: more than %f elapsed", time);
			g_timer_start(item->connection_timer);
			return TRUE;
		}
	}
	
	return TRUE;
}

GList* gems_connection_error_remove(GList* list, guint device_checksum)
{
	GList* iterator = NULL;
	if(list == NULL) return list;
	if(g_list_length(list) == 0) return list;
	
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection_error* element = (gems_connection_error*)iterator->data;
		
		if(element == NULL) continue;
		
		// Match
		if(element->checksum == device_checksum)
		{
			gchar* logmsg = g_strdup_printf("gems_connection_error_remove: removed %d",device_checksum);
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
			
			list = g_list_remove_link(list,iterator);
			gems_clear_connection_error((gems_connection_error*)iterator->data);
			g_list_free_1(iterator);
			return list;
		}
	}
	return list;
}

void gems_clear_connection_error(gems_connection_error* element)
{
	if(element == NULL) return;
	g_timer_destroy(element->connection_timer);
	g_free(element);
	element = NULL;
	return;
}

GList* gems_process_rejected(GList* list)
{
	GList* iterator = g_list_first(list);
	
	while(iterator)
	{
		gems_connection* element = (gems_connection*)iterator->data;
		
		if(element == NULL) continue;
		
		if(element->last_action != NULL)
		{
			// In list for over 5 minutes?
			if(gems_connection_get_last_action(element) > REJECTED_LIST_REMOVE_LIMIT)
			{
				GList* tempiter = iterator;
				iterator = g_list_next(iterator);
				
				list = g_list_remove_link(list,tempiter);
				gems_clear_gems_connection(element);
				g_list_free_1(tempiter);
				
				gchar* logmsg = g_strdup_printf("gems_process_rejected: Removed one from rejected list.");
				cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
				g_free(logmsg);
				
				continue;
			}
		}
		
		iterator = g_list_next(iterator);
	}
	return list;
}

