/*
License: GPLv2, read more from COPYING

This file contains the functionality of help menu.
 */
#include <tangle.h>
#include <string.h>
#include "communitymenu.h"
#include "startmenu.h"
#include "../jammo.h"
#include "help.h"
#include "community_utilities.h"
#include "profile_viewer.h"
#include "single_thread.h"
#include "src/configure.h"
#include <libintl.h>
#include "../../cem/cem.h"

static list_type questionslist = {0, 0};
//static int selectedQuestion = 0;
//static GList *helpdesk_messages = 0;
extern GList* paramsList;
static GList* questions = 0;
static TextField addquestion_titleTextField;
static TextField addquestion_questionTextField;

/**
 * Starts helpdesk
**/
void start_helpdesk(void)
{
	cem_add_to_log(_("Starting Helpdesk"), J_LOG_DEBUG);
	//helpdesk_messages = 0;
	helpdesk_show_helpdesk_view();
}

/**
 * Builds helpdesk view
**/
void helpdesk_show_helpdesk_view(void)
{
	//GList *questions = helpdesk_get_number_of_questions();
	GList *l = 0; //Iterator
	questions = 0;

	int number_of_questions = 0;

	questions = 0;
	
	ClutterActor* mainview = jammo_get_actor_by_id("main-views-widget");
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), CLUTTER_CALLBACK(tangle_actor_hide_animated), NULL);
	ClutterActor* help_view = jammo_get_actor_by_id("helpdesk-view");

	if(help_view)
	{

		if(!helpdesk_get_questions())
			printf("Failed to load helpdesk questions!\n");

		number_of_questions = g_list_length(questions);
		
		clutter_actor_show(CLUTTER_ACTOR(help_view));
		clutter_actor_show(jammo_get_actor_by_id("helpdesk-no-questions-textfield"));

		if(number_of_questions >= 5)
		{
			questionslist.list = CLUTTER_CONTAINER(jammo_get_actor_by_id("helpdesk-list-scrollable"));	
			ClutterAction *listAction = tangle_actor_get_action_by_type(CLUTTER_ACTOR(questionslist.list), TANGLE_TYPE_SCROLL_ACTION);
			g_signal_connect_swapped(listAction, "clamp-offset-y", G_CALLBACK(community_autoscroll_to_correct_pos), &questionslist);
		
		}else{	
			questionslist.list = CLUTTER_CONTAINER(jammo_get_actor_by_id("helpdesk-list-unscrollable"));
		}

		if(number_of_questions != 0)
		{	
			clutter_actor_hide(jammo_get_actor_by_id("helpdesk-no-questions-textfield"));
			
			for(l = questions; l; l = l->next){
				printf("5\n");
				helpdesk_add_question_to_list(l->data, 0);
			}
		}
		if(number_of_questions <= 5){
			clutter_actor_set_height(CLUTTER_ACTOR(questionslist.list), questionslist.listItemHeight * number_of_questions);	
		}else{	
			clutter_actor_set_height(CLUTTER_ACTOR(questionslist.list), questionslist.listItemHeight * 5);
		}	

	}else{ 	
		cem_add_to_log(_("Can't find 'helpdesk-view'"), J_LOG_DEBUG);
	}
}


/**
 * Builds a view needed to add new question
**/
gboolean helpdesk_show_addquestion_view(TangleActor* actor, gpointer data) {
	
	cem_add_to_log(_("Starting AddQuestion View"), J_LOG_DEBUG);

	clear_questions(questions);
	clear_container(questionslist.list);

	ClutterActor* mainview = jammo_get_actor_by_id("main-views-widget");
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), CLUTTER_CALLBACK(tangle_actor_hide_animated), NULL);
	
	ClutterActor* addquestion_view = jammo_get_actor_by_id("helpdesk-addquestion-view");	

	if(addquestion_view) 
	{
		clutter_actor_show(CLUTTER_ACTOR(addquestion_view));

		clutter_text_set_text(CLUTTER_TEXT(jammo_get_actor_by_id("helpdesk-addquestion-title")), _("Add Your Topic Here"));
		clutter_text_set_text(CLUTTER_TEXT(jammo_get_actor_by_id("helpdesk-addquestion-description")), _("Add Your Question Here"));

		addquestion_titleTextField.text = jammo_get_actor_by_id("helpdesk-addquestion-title");
		addquestion_titleTextField.lineCount = 3;
		addquestion_titleTextField.max_characters = 100;
		addquestion_titleTextField.handler_id = g_signal_connect_swapped(addquestion_titleTextField.text,
							"text-changed", (GCallback)limit_line_count, &addquestion_titleTextField);		

		addquestion_questionTextField.text = jammo_get_actor_by_id("helpdesk-addquestion-description");
		addquestion_questionTextField.lineCount = 10;
		addquestion_questionTextField.max_characters = 300;
		addquestion_questionTextField.handler_id = g_signal_connect_swapped(addquestion_questionTextField.text,
							"text-changed", (GCallback)limit_line_count, &addquestion_questionTextField);		
	}
	else 	
		cem_add_to_log("can't find 'helpdesk-addquestion-view' ", J_LOG_ERROR);

	return TRUE;
}


/***************************
Functions to fill lists
**************************/

/**
 * Adds a question to list
**/
gboolean helpdesk_add_question_to_list(gpointer data, guint user_id)
{	
	gchar* colorbar_path = g_strdup_printf("%s/communitymenu/colorbar13.png", DATA_DIR);
	gchar* avatarBackground_path = g_strdup_printf("%s/communitymenu/community_avatarbackground_small.png", DATA_DIR);

	ClutterContainer *listContainer = CLUTTER_CONTAINER(clutter_group_new());

	ClutterActor *colorbar = tangle_texture_new(colorbar_path);
	ClutterActor *avatarBackground = clutter_group_new();
	ClutterActor *composer = tangle_button_new_with_background_actor(avatarBackground);
	ClutterActor *commentButton = tangle_button_new_with_background_actor(tangle_texture_new(
					"communitymenu/community_comment_icon.png"));
	clutter_container_add_actor(CLUTTER_CONTAINER(avatarBackground), tangle_texture_new(avatarBackground_path));

	question_type * question = data;

	char topictitle[25];
	strncpy(topictitle, question->title, 25);

	ClutterActor *text = clutter_text_new_with_text(TEXT_CAPTION, topictitle);	
	ClutterActor *questionButton = tangle_button_new_with_background_actor(text);
	clutter_actor_set_name(questionButton, question->messages);

	profile_view_params *params = malloc(sizeof(profile_view_params));
	strcpy(params->parent_view, "helpdesk");
	params->user_id = question->user_id;

	paramsList = g_list_append(paramsList, params);

	//guint32 avatar_id = gems_profile_manager_get_avatar_id(gems_profile_manager_get_profile_of_user(user_id));
	guint32 avatar_id = question->user_id;

	if (avatar_id == 0)
		avatar_id = 260; //default avatar

	Avatar avatar = {0,0,0,0};

	community_get_avatar(&avatar, avatar_id);
	community_shrink_and_set_avatar(&avatar, CLUTTER_CONTAINER(avatarBackground));	

	clutter_container_add_actor(listContainer, composer);

	if(colorbar == NULL || commentButton == NULL)
	{	
		cem_add_to_log("NULL Pointer Detected!", J_LOG_DEBUG);
		return FALSE;	
	}

	if(questionslist.listItemHeight == 0) 
		questionslist.listItemHeight = clutter_actor_get_height(composer) + 1;

	clutter_actor_set_position(colorbar, 0, questionslist.listItemHeight - 1);
	clutter_container_add_actor(listContainer, colorbar);		//Add components inside the container

	clutter_actor_set_position(questionButton, 60, 23);			
	clutter_container_add_actor(listContainer, questionButton);

	clutter_actor_set_position(commentButton, 690, 15);
	clutter_container_add_actor(listContainer, commentButton);

	g_signal_connect_swapped(CLUTTER_ACTOR(questionButton), "clicked", G_CALLBACK(helpdesk_show_question_view), CLUTTER_ACTOR(questionButton));

	g_signal_connect_swapped(CLUTTER_ACTOR(commentButton), "clicked", G_CALLBACK(helpdesk_show_question_view), CLUTTER_ACTOR(questionButton));
	g_signal_connect_swapped(CLUTTER_ACTOR(composer), "clicked", G_CALLBACK(start_profile_view), (gpointer)params);

	clutter_container_add_actor(questionslist.list, CLUTTER_ACTOR(listContainer));	
	g_free(colorbar_path);
	g_free(avatarBackground_path);

	return TRUE;
}



/***************************
Clear functions
**************************/

void clear_questions(GList* questions_){

	questions_ = g_list_first(questions_);
	//question_type* q = 0;

	if(questions_ != NULL){
		
		//q = questions_->data;

		if(((question_type*)questions_->data)->id)
			free(((question_type*)questions_->data)->id);
			
		if(((question_type*)questions_->data)->messages)
			free(((question_type*)questions_->data)->messages);

		if(((question_type*)questions_->data))
			free((question_type*)questions_->data);
	
		while(questions_->next != 0)
		{
			questions_ = questions_->next;			
			//q = questions_->data;

			if(((question_type*)questions_->data)->id != NULL)
				free(((question_type*)questions_->data)->id);
			
			if(((question_type*)questions_->data)->messages != NULL)
				free(((question_type*)questions_->data)->messages);

			if(questions_->data != NULL)
				free((question_type*)questions_->data);	
			
		}
	
	} 
	g_list_free(questions_);
	questions_ = NULL;

}


/***************************
Functions for json
**************************/

gboolean helpdesk_goto_communitymenu(TangleButton *tanglebutton, gpointer none)
{
	end_helpdesk();
	start_communitymenu();
	return TRUE;
}

gboolean helpdesk_goto_startmenu(TangleButton *tanglebutton, gpointer none)
{
	end_helpdesk();
	startmenu_goto_startmenu(tanglebutton, none);
	return TRUE;
}

gboolean helpdesk_mentor_clicked (TangleButton *tanglebutton, gpointer data)   
{
	cem_add_to_log(_("Mentor Clicked"), J_LOG_DEBUG);
	return TRUE;
}

gboolean helpdesk_goto_helpdesk_clicked(TangleActor *actor, gpointer data){
	helpdesk_show_helpdesk_view();
	return TRUE;
}

gboolean helpdesk_add_question_clicked(TangleActor *actor, gpointer data){
	helpdesk_add_question();
	return TRUE;
}

/**
 * Checks that all the fields have enough text and calls helpview_send_question to send question to server
**/
void helpdesk_add_question(void)
{
	cem_add_to_log(_("Adding Question"), J_LOG_DEBUG);

	//int index = 0;	
	static gboolean is_title_ok = FALSE;
	static gboolean is_msg_ok = FALSE;
	
	ClutterActor *error_msg = jammo_get_actor_by_id("error_message");
	
	if(strlen(clutter_text_get_text(CLUTTER_TEXT(addquestion_titleTextField.text))) < 3){

		cem_add_to_log("Title must be at least 3 characters long!", J_LOG_DEBUG);
		clutter_text_set_text(CLUTTER_TEXT(error_msg), "Title must be at least 3 characters long!");

		is_title_ok = FALSE;
	}else{
		is_title_ok = TRUE;
		cem_add_to_log("Title ok.", J_LOG_DEBUG);
	}

	if(strlen(clutter_text_get_text(CLUTTER_TEXT(addquestion_questionTextField.text))) < 10){

		cem_add_to_log("Message must be longer than 10 characters!", J_LOG_DEBUG);
		clutter_text_set_text(CLUTTER_TEXT(error_msg), "Message must be longer than 10 characters!");

		is_msg_ok = FALSE;
	}else{
		is_msg_ok = TRUE;
		cem_add_to_log("Msg ok", J_LOG_DEBUG);	
	}

	if(is_title_ok == TRUE && is_msg_ok == TRUE){		
		
		time_type now;
		now = get_current_time();
		
		gint question_index = 0;
		guint32 user_id = /*gems_profile_manager_get_userid(NULL)*/TEST_USERID;
		gchar *sender = "Anonymous"; 
		gchar *msg = (gchar*)clutter_text_get_text(CLUTTER_TEXT(addquestion_questionTextField.text));
		gchar* date = now.time;
		gchar *title = (gchar*)clutter_text_get_text(CLUTTER_TEXT(addquestion_titleTextField.text));
		gchar *filepath = "";
		int type = TEXT_COMMENT;
		
		gchar* profile_path = g_strdup_printf("%s/user_%d.csv", configure_get_jammo_directory(), /*avatar_id*/TEST_USERID);
		char line[200];
	
		FILE *f;
		if ((f = fopen(profile_path, "r")) != NULL){
			while(fgets(line, sizeof(line), f)){
				strtok(line, ", ");  
    	  		sender = strtok(NULL, ", ");	
			}
		}
		g_free(profile_path);
		
		if(!helpview_send_question(question_index, user_id, sender, title, msg, filepath, date, type)){
			cem_add_to_log("Failed to add new question: error happened while trying to send file.", J_LOG_DEBUG);
			clutter_text_set_text(CLUTTER_TEXT(error_msg), "Failed to add new question. Please try again.");
		}else{
			cem_add_to_log("Added question.", J_LOG_DEBUG);
		}
		//clutter_text_set_text (CLUTTER_TEXT(jammo_get_actor_by_id("helpdesk-addquestion-title")),"");
		//clutter_text_set_text (CLUTTER_TEXT(jammo_get_actor_by_id("helpdesk-addquestion-descriptionn")),"");
		//clutter_text_set_text(CLUTTER_TEXT(error_msg), "");

		helpdesk_show_helpdesk_view();
	}else{
		cem_add_to_log("Cannot add question.", J_LOG_DEBUG);
	}
}


/**
 * Ends helpdesk and clears all containers
**/
void end_helpdesk(void)
{
	community_clear_paramslist(paramsList);	
	paramsList = 0;	

	clear_questions(questions);
		
	clear_container(questionslist.list);
}


/***************************
Callback functions
**************************/

/**
 * Shows the messages associated with specific question
**/
void helpdesk_show_question_view(ClutterActor *actor)
{
 	//helpdesk_find_selected_actor(actor);
 	const char* filename = clutter_actor_get_name(actor);

	clear_questions(questions);
	//questions = 0;
	clear_container(questionslist.list);

	ClutterActor* mainview = jammo_get_actor_by_id("main-views-widget");
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), CLUTTER_CALLBACK(tangle_actor_hide_animated), NULL);

	///TRANSLATORS: Leave %d as is because it is needed
	//cem_add_to_log(_("Showing Question: %d"), selectedQuestion);

	start_thread_view(/*helpdesk_messages,*/ "helpdesk", filename, 0);
}


/***************************
Server functions
**************************/

/**
 * Downloads all needed json files from server
**/
gboolean helpdesk_get_questions(){
	
	gboolean returnvalue = FALSE;
	cem_add_to_log(_("Fetching Questions From Community Server"), J_LOG_DEBUG);
	
	//TODO: get all_questions.json from server and check that the file was received
	if(!load_file_from_server("all_questions.json")){
		printf("Load threads from server error: Couldn't download all_questions.json from server!\n");
		return FALSE;
	}	
	GList* l = NULL; //Iterator
	int numberOfFound = 0;
	gchar * all_questions_filepath = g_strdup_printf("%s/all_questions.json", configure_get_temp_directory()); //Only for testing!
	
	if(!parse_questions_from_file(all_questions_filepath)){
		returnvalue = FALSE;		
	}else{
		numberOfFound = g_list_length(questions);
		returnvalue = TRUE;
	}
	g_free(all_questions_filepath);
	
	//Download user profiles
	if(returnvalue == TRUE){
		//TODO: download all user json files and check that files were received
		
		if(numberOfFound > 0){
	
			for (l = questions; l; l = l->next){
				question_type* temp = questions->data;
					
				//TODO: build user profile filename!
				gchar* user_profile = g_strdup_printf("user_%d.json", temp->user_id);
				
				if(!load_file_from_server(user_profile)){
					printf("Load questions from server error: Couldn't download user profiles from server!\n");
					returnvalue = FALSE;
				}
				g_free(user_profile);
				
			}//for
		}
	}
	return returnvalue;
}

/**
 * Sends a question to server
**/
gboolean sendJammoQuestion(gchar* question_path, gchar* messages_path)
{
	gboolean returnvalue = TRUE;
		
	//TODO: Send messages.json and all_questions.json to server 
	
	//TODO: Check that files where successfully sended
		
	return returnvalue;
}


/***************************
JSON Parsers & Builders
**************************/

/**
 * Loads all questions from json file
**/
gboolean parse_questions_from_file(gchar * filename) 
{	
	int j;
	//question_type *question = 0;
	JsonParser *parser;
	JsonNode *root;
	JsonObject *object;
	JsonNode *node;
	
	//Init JSON Parser
	parser = json_parser_new();
	g_assert (JSON_IS_PARSER(parser));
	GError *error = NULL;
	
	//Load file
	if (!json_parser_load_from_file (parser, filename, &error)){
		g_print ("Error: %s\n", error->message);
		g_error_free (error);
		g_object_unref (parser);
		return FALSE;
	}

	if(json_parser_get_root(parser) != NULL){
		root = json_parser_get_root (parser);
	}else{
		return FALSE;
	}
	if(JSON_NODE_TYPE(root) != JSON_NODE_OBJECT)
		return FALSE;

	object = json_node_get_object(root); 
	if(object == NULL)
		return FALSE;
	
	//Fill array
	JsonArray* question_array;
	guint length = 0;
	node = json_object_get_member (object, "questions");
	
	if (node != NULL && JSON_NODE_TYPE (node) == JSON_NODE_ARRAY){
		question_array =  json_node_get_array(node);
		length = json_array_get_length(question_array);
		printf("Found %d questions\n", length);
	}else {
		printf("No questions found\n"); 
		return TRUE;
	}
	
	//Check properties
	for (j = 0; j < length; j++) {

		JsonNode* question_node;
		question_node = json_array_get_element(question_array, j);

		if (question_node != NULL && (JSON_NODE_TYPE(question_node) == JSON_NODE_OBJECT)){
			
			JsonObject* sub_object = json_node_get_object(question_node);
			JsonNode *sub_node;
			
			gchar *id = "";	
			gint  question_index = 0;
			guint32 user_id = 0;
			gchar *sender = "";
			gchar *title = "";	
			gchar *q_messages = "";
			gchar *date = "";

			printf("\nFound question!\n");
			
			sub_node = json_object_get_member(sub_object, "id"); 
			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				id = (gchar*)json_node_get_string(sub_node);
				printf("id: '%s'\n", id);
			}	
			sub_node = json_object_get_member(sub_object, "userid"); 

			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				user_id = (guint32)atoi(json_node_get_string(sub_node));			
				printf("userid: '%d'\n", user_id);
			}
			sub_node = json_object_get_member(sub_object, "sender");
			
			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				sender = json_node_get_string(sub_node);
				printf("sender: '%s'\n", sender);
			}
			sub_node = json_object_get_member(sub_object, "index"); 

			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				question_index = (gint)atoi(json_node_get_string(sub_node));			
				printf("index: '%d'\n", question_index);
			}
			sub_node = json_object_get_member(sub_object, "title");
			
			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				title = json_node_get_string(sub_node);
				printf("title: '%s'\n", title);
			}
			sub_node = json_object_get_member(sub_object, "messages"); 
			
			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				q_messages = json_node_get_string(sub_node);			
				printf("messages: '%s'\n", q_messages);
			}		
			sub_node = json_object_get_member(sub_object, "date");
			
			if (sub_node != NULL && JSON_NODE_TYPE(sub_node) == JSON_NODE_VALUE){
				date = json_node_get_string(sub_node);
				printf("date: '%s'\n", date);
			}

			question_type *question = 0;

			//Save information into struct
			question = (question_type*)malloc(sizeof(question_type)); 
			if(question == NULL) 
				return FALSE;

			strcpy(question->title, title);
			question->user_id = user_id;
			question->index = question_index;	
			strcpy(question->date, date);
			
			question->messages = malloc(strlen(q_messages)+1*sizeof(gchar));
			strcpy(question->messages, q_messages);	
			question->id = malloc(strlen(id)+1*sizeof(gchar));

			strcpy(question->id, id);

			//Save struct into global array
		 	questions = g_list_append(questions, (gpointer)question);
			
		} //Question is ready

	} //Next question

	g_object_unref(parser);
	
	return TRUE;
}

/**
 * Builds json files needed for sending new question
**/
gboolean helpview_send_question(gint index_, guint32 userid, gchar* sender, gchar* title, gchar* msg, gchar* filepath, gchar* time_, int type)
{
	printf("helpview_send_question\n");

	//TODO: Download all_questions.json from server
	if(!load_file_from_server("all_questions.json")){
		printf("Couldn't download all_questions.json from server for writing!\n");
		return FALSE;
	}
	gchar *all_questions_path = g_strdup_printf("%s/all_questions.json", configure_get_temp_directory()); //Only for testing
	
	FILE *messageFile;
	gboolean returnvalue = FALSE;
	gchar *messages_file = g_strdup_printf("%s/messages_%s_%d.json", configure_get_temp_directory(), title, userid); 

	gchar* question_id = g_strdup_printf("question_%s_%d", title, userid);
	
	//write comments file
	if ((messageFile = fopen(messages_file, "w")) == NULL) {
			
		if ((messageFile = fopen(messages_file, "w")) == NULL) {
			cem_add_to_log("Error happened while trying to write messages file", J_LOG_ERROR);
			printf("Error happened while trying to write messages file\n");
			return FALSE;
		}
	}
	
	fprintf(messageFile, "{\n"); 
	fprintf(messageFile, "\"messages\": [\n"); 	
	fprintf(messageFile, "{\n"); 
	fprintf(messageFile, "\"index\": \"%d\",\n", index_);
	fprintf(messageFile, "\"userid\": \"%d\",\n", userid);
	fprintf(messageFile, "\"sender\": \"%s\",\n", sender);
	fprintf(messageFile, "\"title\": \"%s\",\n", title);
	fprintf(messageFile, "\"message\": \"%s\",\n", msg);
	fprintf(messageFile, "\"filepath\": \"%s\",\n", filepath);
	fprintf(messageFile, "\"type\": \"%d\",\n", type);
	fprintf(messageFile, "\"date\": \"%s\"\n", time_);
	fprintf(messageFile, "}\n"); 
	fprintf(messageFile, "]\n"); 	
	fprintf(messageFile, "}"); 
	
	fclose(messageFile);

	FILE * pFile;
	gboolean fileIsEmpty = FALSE;
	
	//check whether json file is empty
	if ((pFile = fopen(all_questions_path, "r")) == NULL)
	
		if((pFile = fopen(all_questions_path, "w")) == NULL){
			cem_add_to_log("Error happened while trying to read all_questions.json", J_LOG_ERROR);
			printf("Error happened while trying to read all_questions.json\n");
			return FALSE;
		}	
	
	if(fgetc(pFile) == EOF)
		fileIsEmpty = TRUE;		
	
	fclose(pFile);

	char * dest_filepath = g_strdup_printf("%s/temp.json", configure_get_temp_directory()); 

	if(fileIsEmpty){

		//write a new json file
		if ((pFile = fopen(all_questions_path, "w")) == NULL) {
			cem_add_to_log("Error happened while trying to write all_questions.json", J_LOG_ERROR);
			printf("Error happened while trying to write all_questions.json\n");
			return FALSE;
		}else{
			fprintf(pFile, "{\n"); 
			fprintf(pFile, "\"questions\": [\n"); 	
			fprintf(pFile, "{\n"); 
			fprintf(pFile, "\"id\": \"%s\",\n", question_id);
			fprintf(pFile, "\"userid\": \"%d\",\n", userid);
			fprintf(pFile, "\"sender\": \"%s\",\n", sender);
			fprintf(pFile, "\"index\": \"%d\",\n", 0);
			fprintf(pFile, "\"title\": \"%s\",\n", title);
			fprintf(pFile, "\"messages\": \"%s\",\n", messages_file);
			fprintf(pFile, "\"date\": \"%s\"\n", time_);
			fprintf(pFile, "}\n"); 
			fprintf(pFile, "]\n"); 	
			fprintf(pFile, "}"); 
		}
		fclose(pFile);	
	}else{
		
		//update json file
		FILE *destFile;
		int c, position_found = 0;
		
		//open source file
		if((pFile = fopen(all_questions_path, "r")) == NULL){
			printf("Error happened while trying to open source file\n");
			return FALSE;
		}
		//open destination file
		if((destFile = fopen(dest_filepath, "w")) == NULL){
			printf("Error happened while trying to open destination file\n");
			return FALSE;
		}	
		//copy the file
		fprintf(destFile, "{\n"); 
		fprintf(destFile, "\"questions\": [");
		
		do{
			c = fgetc(pFile);
			if(ferror(pFile)) {
      				printf("Error reading source file\n");
      				return FALSE;
    			}
			if(c == '[')
				position_found = 1;
			
			if(position_found == 1 && c != '[' && c != ']'){
				if(c != ']') fputc(c, destFile);
				if(ferror(destFile)) {
      					printf("Error writing destination file\n");
      					return FALSE;
				}
			}
		}while(c != ']');	
	
		fclose(pFile);
		
		fprintf(destFile, ",\n");
		fprintf(destFile, "{\n"); 
		fprintf(destFile, "\"id\": \"%s\",\n", question_id);
		fprintf(destFile, "\"userid\": \"%d\",\n", userid);
		fprintf(destFile, "\"sender\": \"%s\",\n", sender);
		fprintf(destFile, "\"index\": \"%d\",\n", 0);
		fprintf(destFile, "\"title\": \"%s\",\n", title);
		fprintf(destFile, "\"messages\": \"%s\",\n", messages_file);
		fprintf(destFile, "\"date\": \"%s\"\n", time_);
       		fprintf(destFile, "}\n");	      	 	
		fprintf(destFile, "]\n"); 	
		fprintf(destFile, "}"); 	 	
		fclose(destFile);
		
		if(remove(all_questions_path) != 0){ //We remove the original json
			printf("Error removing original thread json\n");
			return FALSE;
		}			
		rename(dest_filepath, all_questions_path); //And rename temp file with the name of the original
		
	}//Json file is ready

	gchar * log_message = g_strdup_printf("Sending new question '\"%s\"' to server...", title);
	cem_add_to_log(log_message, J_LOG_DEBUG);
	g_free(log_message);
	
	if(!sendJammoQuestion(dest_filepath, messages_file)){
		printf("Error happened when trying to send question to server\n");
		returnvalue = FALSE;
	}else{
		printf("Question was successfully sended\n");
		returnvalue = TRUE;
	}
	g_free(all_questions_path);
	g_free(dest_filepath);
	g_free(messages_file);
	g_free(question_id);
	
	return returnvalue;
}

