/*
* File community_utilities.h contains utility functions that might (not) be useful anywhere
*
*
*/

#ifndef COMMUNITY_UTILITIES_H
#define COMMUNITY_UTILITIES_H

#include <tangle.h>
#include "../jammo.h"
#include "../../configure.h"
#include <libintl.h>
#include "../../gems/gems_profile_manager.h"

#define DESTROY_ACTOR(x) {if(x==0); else{clutter_actor_destroy(x); x = 0;}}
#define TEXT_COMMENT 1
#define VOICE_COMMENT 2
#define MAX_FILES_IN_DIR 50
#define MAX_FILE_NAME_LEN 60

#define TEXT_CAPTION "Comic Sans 24"
#define TEXT_LARGE "Comic Sans 16"
#define TEXT_NORMAL "Comic Sans 14"
#define TEXT_SMALL "Comic Sans 12"

#define TEST_USERID 202179860 //ONLY FOR TESTING
#define TEST_USERID2 18039856 //ONLY FOR TESTING

/*
* Avatar structure contains parts needed to build avatar.
*/
typedef struct{
	ClutterActor *body;
	ClutterActor *hair;
	ClutterActor *head;
	ClutterActor *instrument;
}Avatar; 

/*
* message_type structure contains info about message, sender, time etc.
*/
typedef struct{
	gint  index;
	guint32 user_id;
	gchar sender[20];	
	gchar title[100];
	gchar time[17];
	gchar *msg;
	gchar *filepath;
	int type;
}message_type;


typedef struct{
	guint32 user_id;
	gchar songname[50];	
	gchar *songid;
	gchar *filename;
	//gchar *json;
	gchar *cover;
	gchar *comments_path;
	gchar date[17];
}song_type;

typedef struct{
	guint32 user_id;
	gchar loopname[50];	
	gchar *loopid;
	gchar *filename;
	//gchar *json;
	gchar date[17];
}loop_type;

/*
* Struct used to limit line count of text field.
*
* See function limit_line_count
*/
typedef struct {
	ClutterActor *text;     // actual textfield
	int lineCount;			// number of lines
	gulong handler_id;		// id of the handler function (this is what g_signal_connect returns)
	int max_characters;		// max number of characters in field
}TextField;

typedef struct{
	char time[19];
}time_type;

typedef struct{
	ClutterContainer *list;
	gint listItemHeight;
}list_type;


/*
* Struct needed to pass several params in a single callback function, used when showing record cover view 
*/
typedef struct{
	gchar* path;
	gchar parentview[15];
}show_label_params;
	
/**
 * This function is called when user clicks songname. Depending on whether click was long or short, one of above functions is called.
**/
gboolean song_name_press_action(ClutterActor* actor, ClutterButtonEvent *event);
/**
 * This function is called when user releases the clicked button. Depending on whether click was long or short, one of above functions is called.
**/
gboolean song_name_release_action(ClutterActor* actor, ClutterButtonEvent *event);

void song_on_held_function(TangleButton* button, gpointer user_data);
void song_on_clicked_function(TangleButton* button, gpointer user_data);

/**
 * Renames song/project and song's labelfile in list and in hard-drive
 * 
 * parameters:
 * -loopName is clutter actor holding the filename
 * -type: 0=finished song, 1=project
**/
void rename_loop(ClutterActor *loopName, int type); 
/**
 * Plays selected song. 
 * 
 * parameters:
 * -actor is clutter actor holding the filename
**/
void play_selected_loop(ClutterActor*);
/**
 * Reads label information from labelfile given as parameter, and shows the label viewer
**/
void show_song_label(show_label_params *params);
/**
 * Changes view from label viewer to parent view
**/
void hide_song_label();
/**
 * Function clears container that may contain other containers.
 *
 * parameters:
 * -listToClear is the container to be cleared
**/
void clear_container(ClutterContainer* listToClear);
/**
 * Function creates buttons from each .png type file in the directory.
 * 
 * parameters:
 * -path is the path of the directory
 * -list is a container where the buttons are added
 * -funtion is a pointer to a function which handles button press events
 * -width is the width of the button
 * -height is the height of the button
**/
gboolean add_images_from_directory(const char* path, ClutterContainer *list, GCallback function, int width, int height);
/**
 * Function creates and adds a single button into container.
 *
 * parameters:
 * -button is null pointer to a button to be created
 * -list is a container where button is added
 * -path is a path to image of the button
 * -width is the width of the button
 * -height is the height of the button
 * -funtion is a pointer to a function which handles button press events
**/
gboolean add_single_button(ClutterActor* button, ClutterContainer* list, char* path, int width,int height, GCallback function);
/*
 * Function for checking whether file exists or not.
 * 
 * parameters:
 * -filename is the fullpath of the file
 */
gboolean file_exist(gchar *filename);
/*
 * Function for getting current time.
 */
time_type get_current_time();
/**
 * A callback funtion that limits the line count of editable textfield.
 *
 * parameters:
 * -pointer to a textfield structure that contains info about how to limit line count etc.
**/
void limit_line_count(TextField *text);
/**
 * Frees dynamically allocated memory from GList that contains pointers to message structures.
**/
void clear_messagelist(GList* messages_);
/**
 * Frees dynamically allocated memory from GList that contains pointers to song_type structures.
**/
void clear_songlist(GList*);
/**
 * Computes the avatar from guint32 type number.
 *
 * Avatar is serialized in 32 bits in the following way:
 * 		-head id is the 4 least significant bits (bits 0-3)
 *		-hair color is bits 4-7
 *		-body is bits 8-15
 *		-hair type is bits 16-23 
 *		-instrument is in bits 24-31
 *
 *	Note that ids are actually index numbers of avatar parts. Parts are arranged in alphabetical order.
 *
 * parameters:
 * -avatar is pointer to a Avatar struct, where the avatar will be created.
 * -avatar_id tells the avatar type.
**/
gboolean community_get_avatar(Avatar* avatar, guint32 avatar_id);
/**
 * Computes and saves guint32 type number from user avatar.
 *
 * Avatar is serialized in 32 bits in the following way:
 * 		-head id is the 4 least significant bits (bits 0-3)
 *		-hair color is bits 4-7
 *		-body is bits 8-15
 *		-hair type is bits 16-23 
 *		-instrument is in bits 24-31
 *
 *	Note that ids are actually index numbers of avatar parts. Parts are arranged in alphabetical order.
 *
 * parameters:
 * -avatar is pointer to a Avatar struct, which contains user avatar
 * returns: avatar id in guint32
**/
guint32 community_save_avatar(Avatar* avatar);
/**
 * This function opens directory, and sorts files in alphabetical order.
 * Files must be sorted because file order from opendir() may be different in each device.
 *
 * parameters: 
 * -path is full path of directory to be opened
 * -suffix defines which type of files will be sorted, other files are ignored
 *   (if suffix is "DIR", only directories are listed)
 * -result is an array of strings that contains filenames in order when this funtion returns
 *
 * returns number of files found
**/
int community_open_and_arrange_dir(const char* path, char* suffix, char result[MAX_FILES_IN_DIR][MAX_FILE_NAME_LEN]);
/**
 * Transforms avatar into smaller scale, so it can be used in thread-views.
 *
 * parameters:
 * -avatar is the avatar to be scaled
 * -avatarBackground is container, where avatar will be set
**/
void community_shrink_and_set_avatar(Avatar* avatar, ClutterContainer *avatarBackground);
/**
 * Funtion frees GList that contains pointers to dynamically allocated profile_view_params structures
**/
void community_clear_paramslist(GList* paramsList);
/**
 * Funtion frees GList that contains pointers to dynamically allocated show_label_params structures
**/
void community_clear_labelparamslist(GList* paramslist);
/**
 * Scrolls the list to correct position automatically
**/
void community_autoscroll_to_correct_pos(list_type *list);
/**
 * Loads songs/loops from json file. Note that this function allocates memory which should be freed later. 
 *
 * parameters:
 * -filename is name of the json file
 * -list is a pointer pointing to GList where songs/loops will be saved to
 * -type: 0=loop, 1=song
**/
gboolean parse_songs_from_file(gchar* filename, GList** list, int type);
/**
 * Builds filename from fullpath. Useful for parsing songname from filepath. Returns filename. 
 * 
 * parameters:
 * -filename is a char pointer where the resulting filename will be saved to
 * -folder_path is folderpath where the file resides 
 * -filepath is the fullpath where the filename is parsed from
**/
gchar* build_filename(gchar* filepath, gchar* folder_path, gchar* filename, int n);
/**
 * Builds filepath for specific file. Returns full filepath. 
 * 
 * parameters:
 * -filename is the name of the file
 * -folder_path is folderpath where the file resides 
 * -filepath is a char pointer where the resulting fullpath will be saved to
**/
gchar* build_filepath(gchar* filename, gchar* folder_path, gchar* filepath);
/**
 * Deletes all files inside jammo temp directory. This should be called before jammo is terminated.  
**/
gboolean clear_community_temp_directory();
/**
 * Downloads a specific file from server
**/
gboolean load_file_from_server(gchar* filename);

#endif
