/*
 * jammo-mentor.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-mentor.h"
#include "../meam/jammo-sample.h"
#include <time.h>
#include <dbus/dbus-glib.h>
#include <string.h>

#include "../cem/cem.h"
G_DEFINE_TYPE(JammoMentor, jammo_mentor, TANGLE_TYPE_BUTTON)

#ifdef N900
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#else
#define MCE_TKLOCK_MODE_SIG "tklock_mode_ind"
#define MCE_SERVICE "com.nokia.mce"
#define MCE_SIGNAL_PATH "/com/nokia/mce/signal"
#define MCE_SIGNAL_IF "com.nokia.mce.signal"
#define MCE_DEVICE_LOCKED "locked"
#endif

enum {
	PROP_0,
	PROP_STANDBY_SCALE,
	PROP_ACTIVE_SCALE,
	PROP_IDLE_SPEECH,
	PROP_IDLE_TIMEOUT,
	PROP_DEVICE_LOCKED
};

struct _JammoMentorPrivate {
	gfloat standby_scale;
	gfloat active_scale;
	gchar* idle_speech;
	JammoMentorSpokenCallback idle_speech_spoken_callback;
	gpointer idle_speech_spoken_user_data;
	gulong idle_timeout;
	gulong default_idle_timeout;
	gchar* language;
	
	GList* spoken_speeches;
	JammoSample* active_sample;

	ClutterActor* stage;
	gulong stage_captured_event_handler_id;
	guint timeout_event_source_id;
	time_t last_activity_time;
	DBusGProxy* mce_proxy;
	
	guint idle_speech_spoken : 1;
	guint interrupted : 1;
	guint device_locked : 1;
	guint passive_mode : 1;
	guint holding_event_source_id;
};

static void speak(JammoMentor* mentor, const gchar* speech, gboolean highlight, gfloat offset_x, gfloat offset_y, JammoMentorSpokenCallback callback, gpointer user_data);
static void set_idle_timeout(JammoMentor* mentor, gboolean activate);
static void on_mce_tklock_mode_sig(DBusGProxy* proxy, const gchar* mode, gpointer user_data);

static GList* mentors;

ClutterActor* jammo_mentor_new(ClutterActor* actor, gfloat standby_scale, gfloat active_scale) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_MENTOR, "normal-background-actor", actor, "standby-scale", standby_scale, "active-scale", active_scale, "prefer-background-size", TRUE, NULL));
}

gfloat jammo_mentor_get_standby_scale(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), 0.0);

	return mentor->priv->standby_scale;
}

void jammo_mentor_set_standby_scale(JammoMentor* mentor, gfloat scale) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor->priv->standby_scale != scale) {
		mentor->priv->standby_scale = scale;
		if (!mentor->priv->active_sample) {
			tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1500, "scale-x", mentor->priv->standby_scale, "scale-y", mentor->priv->standby_scale, NULL);
		}
		g_object_notify(G_OBJECT(mentor), "standby-scale");
	}
}

gfloat jammo_mentor_get_active_scale(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), 0.0);

	return mentor->priv->active_scale;
}

void jammo_mentor_set_active_scale(JammoMentor* mentor, gfloat scale) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor->priv->active_scale != scale) {
		mentor->priv->active_scale = scale;
		if (mentor->priv->active_sample) {
			tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1500, "scale-x", mentor->priv->active_scale, "scale-y", mentor->priv->active_scale, NULL);
		}
		g_object_notify(G_OBJECT(mentor), "active-scale");
	}
}

void jammo_mentor_speak(JammoMentor* mentor, const gchar* speech) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	speak(mentor, speech, FALSE, 0.0, 0.0, NULL, NULL);
}

void jammo_mentor_speak_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	speak(mentor, speech, FALSE, 0.0, 0.0, callback, user_data);
}

void jammo_mentor_speak_and_highlight(JammoMentor* mentor, const gchar* speech, gfloat offset_x, gfloat offset_y) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	speak(mentor, speech, TRUE, offset_x, offset_y, NULL, NULL);
}

void jammo_mentor_speak_and_highlight_with_callback(JammoMentor* mentor, const gchar* speech, gfloat offset_x, gfloat offset_y, JammoMentorSpokenCallback callback, gpointer user_data) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	speak(mentor, speech, TRUE, offset_x, offset_y, callback, user_data);
}

void jammo_mentor_speak_once(JammoMentor* mentor, const gchar* speech) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, FALSE, 0.0, 0.0, NULL, NULL);
	}
}

void jammo_mentor_speak_once_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, FALSE, 0.0, 0.0, callback, user_data);
	}
}

void jammo_mentor_speak_and_highlight_once(JammoMentor* mentor, const gchar* speech, gfloat offset_x, gfloat offset_y) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, TRUE, offset_x, offset_y, NULL, NULL);
	}
}

void jammo_mentor_speak_and_highlight_once_with_callback(JammoMentor* mentor, const gchar* speech, gfloat offset_x, gfloat offset_y, JammoMentorSpokenCallback callback, gpointer user_data) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (!g_list_find_custom(mentor->priv->spoken_speeches, speech, (GCompareFunc)g_strcmp0)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, TRUE, offset_x, offset_y, callback, user_data);
	}
}

void jammo_mentor_shut_up(JammoMentor* mentor) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor->priv->active_sample) {
		mentor->priv->interrupted = TRUE;
		jammo_sample_stop(mentor->priv->active_sample);
	}
}

const gchar* jammo_mentor_get_idle_speech(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), NULL);

	return mentor->priv->idle_speech;
}

void jammo_mentor_set_idle_speech(JammoMentor* mentor, const gchar* speech) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	jammo_mentor_set_idle_speech_with_callback(mentor, speech, NULL, NULL);
}

void jammo_mentor_set_idle_speech_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (g_strcmp0(mentor->priv->idle_speech, speech)) {
		g_free(mentor->priv->idle_speech);
		mentor->priv->idle_speech = g_strdup(speech);
		mentor->priv->idle_speech_spoken = FALSE;
		set_idle_timeout(mentor, TRUE);
		g_object_notify(G_OBJECT(mentor), "idle-speech");
	}
	mentor->priv->idle_speech_spoken_callback = callback;
	mentor->priv->idle_speech_spoken_user_data = user_data;
}

gulong jammo_mentor_get_idle_timeout(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), 0);

	return mentor->priv->idle_timeout;
}

void jammo_mentor_set_idle_timeout(JammoMentor* mentor, gulong milliseconds) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor->priv->idle_timeout != milliseconds) {
		mentor->priv->idle_timeout = milliseconds;
		set_idle_timeout(mentor, TRUE);
		g_object_notify(G_OBJECT(mentor), "idle-timeout");
	}
}

const gchar* jammo_mentor_get_language(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), NULL);

	return mentor->priv->language;
}

void jammo_mentor_set_language(JammoMentor* mentor, const gchar* language) {
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor->priv->language)
		g_free(mentor->priv->language);
	mentor->priv->language = g_strdup(language);
}

void jammo_mentor_set_passive(JammoMentor* mentor, gboolean passive){
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	mentor->priv->passive_mode=passive;
	if (passive) {
		cem_add_to_log("Mentor goes to passive mode",J_LOG_DEBUG);
		mentor->priv->passive_mode=1;
		clutter_actor_set_opacity (CLUTTER_ACTOR(mentor), 60);
		//store current idle_timeout
		mentor->priv->default_idle_timeout=jammo_mentor_get_idle_timeout(mentor);
		g_object_set (mentor,"idle-timeout", G_MAXULONG,NULL);
	}
	else {
		cem_add_to_log("Mentor goes to active mode",J_LOG_DEBUG);
		mentor->priv->passive_mode=0;
		clutter_actor_set_opacity (CLUTTER_ACTOR(mentor), 255);
		//return stored idle_timeout
		jammo_mentor_set_idle_timeout(mentor,mentor->priv->default_idle_timeout);
	}
}


gboolean jammo_mentor_get_device_locked(JammoMentor* mentor) {
	g_return_val_if_fail(JAMMO_IS_MENTOR(mentor), FALSE);

	return mentor->priv->device_locked;
}

JammoMentor* jammo_mentor_get_default(void) {

	return (mentors ? JAMMO_MENTOR(mentors->data) : NULL);
}

static gboolean jammo_mentor_clicked(TangleButton* button) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(button);
	
	if (mentor->priv->active_sample) {
		mentor->priv->interrupted = TRUE;
		jammo_sample_stop(mentor->priv->active_sample);
	} else if (mentor->priv->idle_speech) {
		speak(mentor, mentor->priv->idle_speech, FALSE, 0.0, 0.0, mentor->priv->idle_speech_spoken_callback, mentor->priv->idle_speech_spoken_user_data);
	}

	return FALSE;
}

static void jammo_mentor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(object);

	switch (prop_id) {
		case PROP_STANDBY_SCALE:
			jammo_mentor_set_standby_scale(mentor, g_value_get_float(value));
			break;
		case PROP_ACTIVE_SCALE:
			jammo_mentor_set_active_scale(mentor, g_value_get_float(value));
			break;
		case PROP_IDLE_SPEECH:
			jammo_mentor_set_idle_speech(mentor, g_value_get_string(value));
			break;
		case PROP_IDLE_TIMEOUT:
			jammo_mentor_set_idle_timeout(mentor, g_value_get_ulong(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_mentor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoMentor* mentor;

	mentor = JAMMO_MENTOR(object);

        switch (prop_id) {
		case PROP_STANDBY_SCALE:
			g_value_set_float(value, mentor->priv->standby_scale);
			break;
		case PROP_ACTIVE_SCALE:
			g_value_set_float(value, mentor->priv->active_scale);
			break;
		case PROP_IDLE_SPEECH:
			g_value_set_string(value, mentor->priv->idle_speech);
			break;
		case PROP_IDLE_TIMEOUT:
			g_value_set_ulong(value, mentor->priv->idle_timeout);
			break;
		case PROP_DEVICE_LOCKED:
			g_value_set_boolean(value, mentor->priv->device_locked);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_mentor_show(ClutterActor* actor) {
	set_idle_timeout(JAMMO_MENTOR(actor), TRUE);
	
	CLUTTER_ACTOR_CLASS(jammo_mentor_parent_class)->show(actor);
}

static void jammo_mentor_hide(ClutterActor* actor) {
	set_idle_timeout(JAMMO_MENTOR(actor), FALSE);
	
	CLUTTER_ACTOR_CLASS(jammo_mentor_parent_class)->hide(actor);
}

static void jammo_mentor_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_parent_class)->finalize(object);
}

static void jammo_mentor_dispose(GObject* object) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(object);
	
	set_idle_timeout(mentor, FALSE);
	
	if (mentor->priv->mce_proxy) {
		g_object_unref(mentor->priv->mce_proxy);
		mentor->priv->mce_proxy = NULL;
	}

	G_OBJECT_CLASS(jammo_mentor_parent_class)->dispose(object);
}


static gboolean tap_and_hold_function(gpointer user_data) {
	JammoMentor* mentor;
	mentor = JAMMO_MENTOR(user_data);

	mentor->priv->holding_event_source_id = 0;

	cem_add_to_log("Mentor tap-and-hold",J_LOG_DEBUG);
	jammo_mentor_shut_up(mentor);

	jammo_mentor_set_passive(mentor,!mentor->priv->passive_mode);

	return FALSE;

}

static gboolean jammo_mentor_pressed (ClutterActor *actor, ClutterButtonEvent *event) {
	JAMMO_MENTOR(actor)->priv->holding_event_source_id = g_timeout_add(3500, tap_and_hold_function, actor);
	return TRUE;
}


static gboolean jammo_mentor_released (ClutterActor *actor, ClutterButtonEvent *event) {
	if (JAMMO_MENTOR(actor)->priv->holding_event_source_id) {
		g_source_remove(JAMMO_MENTOR(actor)->priv->holding_event_source_id);
		JAMMO_MENTOR(actor)->priv->holding_event_source_id = 0;

		//If tap_and_hold_function is triggered, we do not want call this:
		jammo_mentor_clicked(TANGLE_BUTTON(actor));
	}

	return TRUE;
}

static void jammo_mentor_class_init(JammoMentorClass* mentor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(mentor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(mentor_class);
	//TangleButtonClass* button_class = TANGLE_BUTTON_CLASS(mentor_class);
	//button_class->clicked = jammo_mentor_clicked;

	gobject_class->finalize = jammo_mentor_finalize;
	gobject_class->dispose = jammo_mentor_dispose;
	gobject_class->set_property = jammo_mentor_set_property;
	gobject_class->get_property = jammo_mentor_get_property;

	clutter_actor_class->show = jammo_mentor_show;
	clutter_actor_class->hide = jammo_mentor_hide;
	clutter_actor_class->button_release_event = jammo_mentor_released;
	clutter_actor_class->button_press_event = jammo_mentor_pressed;

	/**
	 * JammoMentor:standby-scale:
	 *
	 * The scaling that is efective when the mentor is in standby.
	 */
	g_object_class_install_property(gobject_class, PROP_STANDBY_SCALE,
	                                g_param_spec_float("standby-scale",
	                                                   "Standby scale",
	                                                   "The scaling that is efective when the mentor is in standby",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:active-scale:
	 *
	 * The scaling that is efective when the mentor is active.
	 */
	g_object_class_install_property(gobject_class, PROP_ACTIVE_SCALE,
	                                g_param_spec_float("active-scale",
	                                                   "Active scale",
	                                                   "The scaling that is efective when the mentor is active",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:idle-speech:
	 *
	 * The speech that the mentor will speak after a specified time (see :idle-time).
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_SPEECH,
	                                g_param_spec_string("idle-speech",
	                                                    "Idle speech",
	                                                    "The speech that the mentor will speak after a specified time",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:idle-time:
	 *
	 * The time in seconds after the mentor will speak (see :idle-speech) if an user has been idle.
	 * Value 0 means disabled.
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_TIMEOUT,
	                                g_param_spec_ulong("idle-timeout",
	                                                   "Idle timeout",
	                                                   "The time in seconds after the mentor will speak if an user has been idle.",
	                                                   0, G_MAXULONG, 0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_CONSTRUCT));
	/**
	 * JammoMentor:device-locked:
	 *
	 * Whether the device is locked.
	 *
	 * Despite being conceptually unrelated to a mentor, this property keeps track of the device locking state.
	 * The mentor itself will only speak if an user has been idle and the device is unlocked. Thus, it monitors
	 * the device locking state and exposes the result for other uses also using this property.
	 */
	g_object_class_install_property(gobject_class, PROP_DEVICE_LOCKED,
	                                g_param_spec_boolean("device-locked",
	                                                     "Device locked",
	                                                     "Whether the device is locked.",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (JammoMentorPrivate));
}

static void jammo_mentor_init(JammoMentor* mentor) {
	DBusGConnection* connection;

	mentor->priv = G_TYPE_INSTANCE_GET_PRIVATE(mentor, JAMMO_TYPE_MENTOR, JammoMentorPrivate);
	mentor->priv->passive_mode=0;
	mentors = g_list_append(mentors, mentor);
	
	if ((connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, NULL))) {
		mentor->priv->mce_proxy = dbus_g_proxy_new_for_name(connection, MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF);
		dbus_g_proxy_add_signal(mentor->priv->mce_proxy, MCE_TKLOCK_MODE_SIG, G_TYPE_STRING, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal(mentor->priv->mce_proxy, MCE_TKLOCK_MODE_SIG, G_CALLBACK(on_mce_tklock_mode_sig), mentor, NULL);
	}
}

static const gchar* find_characters(const gchar* string, const gchar* characters_parameter) {
	const gchar* found = NULL;
	const gchar* c;
	
	while (!found && *string) {
		for (c = characters_parameter; *c; c++) {
			if (*string == *c) {
				found = string;
				break;
			}
		}
		string++;
	}
	
	return found;
}

static const gchar* find_characters_inverse(const gchar* string, const gchar* characters_parameter) {
	const gchar* found = NULL;
	const gchar* c;
	
	while (*string) {
		for (c = characters_parameter; *c; c++) {
			if (*string == *c) {
				break;
			}
		}
		if (!*c) {
			found = string;
			break;
		}
		string++;
	}
	
	return found;
}

static void real_speak(JammoMentor* mentor, const gchar* next_speech, JammoMentorSpokenCallback callback, const gchar* speech, gpointer user_data);

static void on_sample_stopped(JammoSample* sample, gpointer user_data) {
	TangleVault* vault;
	JammoMentor* mentor;
	const gchar* next_speech;
	JammoMentorSpokenCallback callback;
	const gchar* speech;
	gpointer data;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 5, JAMMO_TYPE_MENTOR, &mentor, G_TYPE_STRING, &next_speech, G_TYPE_POINTER, &callback, G_TYPE_STRING, &speech, G_TYPE_POINTER, &data);
	
	g_object_unref(mentor->priv->active_sample);
	mentor->priv->active_sample = NULL;

	if (next_speech && !mentor->priv->interrupted) {
		real_speak(mentor, next_speech, callback, speech, data);
	} else {

		tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1000, "squint-x", 0.0, "squint-y", 0.0, "scale-x", mentor->priv->standby_scale, "scale-y", mentor->priv->standby_scale, NULL);

		set_idle_timeout(mentor, TRUE);

		if (callback) {
			callback(mentor, speech, mentor->priv->interrupted, data);
		}

		mentor->priv->interrupted = FALSE;

		/* TODO: g_free(speech); ? */
	}
}

#define WHITE_SPACES " \t\n\r"

static void real_speak(JammoMentor* mentor, const gchar* next_speech, JammoMentorSpokenCallback callback, const gchar* speech, gpointer user_data) {
	const gchar* space;
	gchar* current_speech;
	gchar* localized_speech;
	gchar* filename;
	TangleVault* vault;
	
	if ((space = find_characters(next_speech, WHITE_SPACES))) {
		current_speech = g_strndup(next_speech, space - next_speech);
		next_speech = find_characters_inverse(space, WHITE_SPACES);
	} else {
		current_speech = g_strdup(next_speech);
		next_speech = NULL;
	}
	
	if (current_speech[0] == '/' || !mentor->priv->language) {
		localized_speech = g_strdup(current_speech);
	} else {
		localized_speech = g_strconcat(mentor->priv->language, "/", current_speech, NULL);
	}
	if ((filename = tangle_lookup_filename(localized_speech))) {
		mentor->priv->active_sample = jammo_sample_new_from_file(filename);
		g_free(filename);
	} else {
		mentor->priv->active_sample = jammo_sample_new_from_file(localized_speech);
	}

	cem_add_to_log(localized_speech,J_LOG_MENTOR_SPEECH);
	//printf("%s * %s * %s * %s\n", current_speech, localized_speech, speech, next_speech);

	g_free(localized_speech);
	g_free(current_speech);
	
	vault = tangle_vault_new(5, JAMMO_TYPE_MENTOR, mentor, G_TYPE_STRING, next_speech, G_TYPE_POINTER, callback, G_TYPE_STRING, speech, G_TYPE_POINTER, user_data);
	tangle_signal_connect_vault(mentor->priv->active_sample, "stopped", G_CALLBACK(on_sample_stopped), vault);

	jammo_sample_play(mentor->priv->active_sample);
}

static void speak(JammoMentor* mentor, const gchar* speech, gboolean highlight, gfloat offset_x, gfloat offset_y, JammoMentorSpokenCallback callback, gpointer user_data) {	

	set_idle_timeout(mentor, FALSE);
	tangle_actor_show(TANGLE_ACTOR(mentor));

	if (mentor->priv->passive_mode) {
		cem_add_to_log(" Mentor in passive mode, skip",J_LOG_DEBUG);
		if (callback) {
			callback(mentor, speech, TRUE, user_data);
			}
	return;
	}


	if (mentor->priv->active_sample) {
		jammo_sample_stop(mentor->priv->active_sample);
	}

	real_speak(mentor, speech, callback, speech, user_data);
	
	tangle_actor_animate(TANGLE_ACTOR(mentor), CLUTTER_EASE_IN_QUAD, 1000, "squint-x", offset_x, "squint-y", offset_y, "scale-x", mentor->priv->active_scale, "scale-y", mentor->priv->active_scale, NULL);
}

static gboolean on_stage_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(user_data);
	
	mentor->priv->last_activity_time = time(NULL);
	
	return FALSE;
}

static gboolean on_timeout(gpointer user_data) {
	JammoMentor* mentor;
	time_t elapsed_time;
	
	mentor = JAMMO_MENTOR(user_data);
	
	elapsed_time = time(NULL) - mentor->priv->last_activity_time;
	if (elapsed_time >= mentor->priv->idle_timeout) {
		mentor->priv->timeout_event_source_id = 0;
		mentor->priv->idle_speech_spoken = TRUE;
		speak(mentor, mentor->priv->idle_speech, FALSE, 0.0, 0.0, mentor->priv->idle_speech_spoken_callback, mentor->priv->idle_speech_spoken_user_data);
	} else {
		mentor->priv->timeout_event_source_id = g_timeout_add((mentor->priv->idle_timeout - elapsed_time) * 1000, on_timeout, mentor);
	}
	
	return FALSE;
}

static void set_idle_timeout(JammoMentor* mentor, gboolean activate) {
	if (mentor->priv->timeout_event_source_id) {
		g_source_remove(mentor->priv->timeout_event_source_id);
		mentor->priv->timeout_event_source_id = 0;
	}
	
	if (activate && mentor->priv->idle_timeout && !mentor->priv->device_locked && mentor->priv->idle_speech && !mentor->priv->idle_speech_spoken && !mentor->priv->active_sample && CLUTTER_ACTOR_IS_VISIBLE(mentor)) {
		if (!mentor->priv->stage && (mentor->priv->stage = clutter_actor_get_stage(CLUTTER_ACTOR(mentor)))) {
			mentor->priv->stage_captured_event_handler_id = g_signal_connect(mentor->priv->stage, "captured-event", G_CALLBACK(on_stage_captured_event), mentor);
		}

		mentor->priv->last_activity_time = time(NULL);

		mentor->priv->timeout_event_source_id = g_timeout_add(mentor->priv->idle_timeout * 1000, on_timeout, mentor);
	} else if (!activate && mentor->priv->stage) {
		g_signal_handler_disconnect(mentor->priv->stage, mentor->priv->stage_captured_event_handler_id);
		mentor->priv->stage = NULL;
	}
}

static void on_mce_tklock_mode_sig(DBusGProxy* proxy, const gchar* mode, gpointer user_data) {
	JammoMentor* mentor;
	gboolean old_device_locked;
	
	mentor = JAMMO_MENTOR(user_data);

	old_device_locked = mentor->priv->device_locked;
	if (!strcmp(mode, MCE_DEVICE_LOCKED)) {
		mentor->priv->device_locked = TRUE;
		jammo_mentor_shut_up(mentor);
	} else {
		mentor->priv->device_locked = FALSE;
		mentor->priv->idle_speech_spoken = FALSE;
	}
	if (old_device_locked != mentor->priv->device_locked) {
		set_idle_timeout(mentor, !mentor->priv->device_locked);
		g_object_notify(G_OBJECT(mentor), "device-locked");
	}
}
