/*
 * gems_teacher_server_utils.h
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#ifndef __GEMS_TEACHER_SERVER_UTILS_H_
#define __GEMS_TEACHER_SERVER_UTILS_H_

#define WRITE_FAIL_CONTENT_NOT_SET 1
#define WRITE_FAIL_CANNOT_REMOVE_FILE 2 
#define WRITE_FAIL_CANNOT_SET_FILE_CONTENT 3
#define WRITE_OK 0
#define READ_FAIL_CANNOT_READ 1
#define READ_FAIL_FILE_NOT_EXIST 2
#define READ_OK 0

#define TSRV_DATABASE "teacherprofiles.db"

#include <glib.h>
#include "gems_definitions.h"
#include "gems_utils.h"

typedef struct {
	guint32 userid;
	gchar* username;
	gchar* password;
	guchar* salt;
	gchar* firstname;
	gchar* lastname;
	gchar* authlevel;
	guint16 age;
	guint32 points;
	guint32 avatarid;
	gchar* encrypted; // Contains the whole encrypted profile (salt+pwhash+phash+profile)
	guint encrypted_length; // Length of encrypted -field
} jammo_profile_container;

typedef struct {
	GList* profiles; // contains jammo_profile_container*
	guchar* salt;
	guchar* pwhash;
	guchar* datahash;
	guchar* data; // encrypted data - all profiles
	guint datalength;
} teacher_server_profile_data;

/* Allocate new jammo_profile_container
 *
 * Returns:
 *  pointer to jammo_profile_container struct
 */
jammo_profile_container* gems_teacher_server_new_profile_container();

/* Allocate new teacher_server_profile_data container.
 * 
 * Returns:
 *  pointer to teacher_server_profile_data
 */
teacher_server_profile_data* gems_teacher_server_new_profile_data_container();

/* Free the given jammo_profile_container struct
 *
 * Params:
 *  container - pointer to container struct
 */
void gems_teacher_server_clear_profile_container(jammo_profile_container* container);

/* Free the given teacher_server_profile_data struct. Calls gems_teacher_server_clear_profile_list()
 * to free the list of profiles and each saved profile.
 *
 * Params:
 *  container - pointer to teacher_server_profile_data
 */
void gems_teacher_server_clear_profile_data_container(teacher_server_profile_data* data);

/* Create a new profile with given information, profile will be also encrypted - calls
 * gems_teacher_profile_encrypt_profile().
 *
 * Params (duplicates are created from gchars):
 *  _userid - user id to use, should be something else than 0
 *  _username - username
 *  _password - password that user gives
 *  _authlevel - level of authentication (atm. "none" should be ok..)
 *  _firstname - firstname of the user
 *  _lastname - lastname of the user
 *  _age - age of the user
 *  _avatarid - id of the avatar of the user
 * Returns:
 *  pointer to jammo_profile_container struct when all could be done
 *  NULL if some parameter was not set or the profile couldn't be encrypted
 */
jammo_profile_container* gems_teacher_server_create_profile(guint32 _userid, gchar* _username, gchar* _password,
	gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age, guint32 _avatarid);
	
/* Setup a profile with given information, profile will be also encrypted - calls
 * gems_teacher_profile_encrypt_profile().
 *
 * Params (duplicates are created from gchars):
 *  _userid - user id to use, should be something else than 0
 *  _username - username
 *  _password - password that user gives
 *  _salt - password salt
 *  _authlevel - level of authentication (atm. "none" should be ok..)
 *  _firstname - firstname of the user
 *  _lastname - lastname of the user
 *  _age - age of the user
 *  _points - amount of points
 *  _avatarid - id of the avatar of the user
 * Returns:
 *  pointer to jammo_profile_container struct when all could be done
 *  NULL if some parameter was not set or the profile couldn't be encrypted
 */
jammo_profile_container* gems_teacher_server_setup_profile(guint32 _userid, gchar* _username, gchar* _password,
	guchar* _salt, gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age, guint32 _points, guint32 _avatarid);

/* Get a profile from database
 * 
 * Params:
 *  data - the teacher profile db
 *  username - username to use for searching
 * Returns:
 *  pointer to jammo_profile_container if found
 *  NULL if not found
 */
jammo_profile_container* gems_teacher_server_get_profile_for_user(teacher_server_profile_data* data, gchar* username);

/* Get a profile from database with user id, useful for checking for duplicate id's
 * 
 * Params:
 *  data - the teacher profile db
 *  userid - user id to use for searching
 * Returns:
 *  pointer to jammo_profile_container if found
 *  NULL if not found
 */
jammo_profile_container* gems_teacher_server_get_profile_for_user_id(teacher_server_profile_data* data, guint32 userid);

/* Encrypt the profile content, uses data in container and sets the encrypted content + length
 * Normally used through other functions (create profile & change password) - no checks for container data !
 *
 * Params:
 *  container - pointer to container struct, should have all profile details set!
 * Returns:
 *  TRUE when profile could be encrypted
 *  FALSE if something was missing
 */
gboolean gems_teacher_server_encrypt_profile(jammo_profile_container* container);

/* Change password for given profile, profile data will be re-encrypted with new password
 *
 * Params:
 *  container - profile to use, pointer to jammo_profile_container struct
 *  password - new password
 * Returns:
 *  TRUE when password could be changed
 *  FALSE when there are problems setting password (cannot initialize security or encryption fails)
 */ 
gboolean gems_teacher_server_change_password(jammo_profile_container* container, gchar* password);

/* Write encrypted profile to "username.jpf" into configure_get_jammo_directory()/JAMMOPROFILE
 *
 * Params:
 *   container - pointer to jammo_profile_container struct containing encrypted data
 * Returns:
 *  WRITE_FAIL_CONTENT_NOT_SET for invalid parameters
 *  WRITE_FAIL_CANNOT_REMOVE_FILE if previous profile cannot be removed
 *  WRITE_FAIL_CANNOT_SET_FILE_CONTENT if cannot set the encrypted data to file
 *  WRITE_OK when profile was written
 */
gint gems_teacher_server_write_profile(jammo_profile_container* container);

/* Save a single profile to disk with given encrypted data. Creates a jammo_profile_container with
 * encrypted data and calls gems_teacher_write_profile() to write this single profile onto disk. 
 * The profile is saved to configure_get_jammo_directory()/JAMMOPROFILE. Useful for storing received
 * encrypted profile to disk.
 *
 * Params:
 *  username - username to use (for saving the profile to username.jpf)
 *  profiledata - encrypted data to be saved to disk - the content of the profile file
 *  datalength - length of the encrypted data.
 * Returns:
 *  TRUE when profile was written
 *  FALSE if profile couldn't be written to disk.
 */
gboolean gems_teacher_server_save_encrypted_profile(gchar* username, gchar* profiledata, guint datalength);

/* Serialize ALL profiles into one big gchar*. Integers are stored as they are based on their size and
 * charstrings will have a 8 bit integer in front informing the length of the charstring (NOTE: limitation
 * for charstring length is atm 255!). This will be called when encrypting database.
 * 
 * Params:
 *  list - list of jammo_profile_container*
 *  length - variable to store the length of the returned gchar*
 * Returns:
 *  gchar* containing all profiles or NULL if there is not enough memory to allocate for the gchar
 */
gchar* gems_teacher_server_serialize_all_profiles(GList* list, guint* length);

/* Deserialize profiles and add profiles as jammo_profile_container* to list. This will be called when
 * database is decrypted.
 *
 * Params:
 *  list - list where deserialized profiles are added
 *  serialized_data - profiles in serialized format
 *  length - length of serialized_data
 * Returns:
 *  GList containing pointers to jammo_profile_containers
*/
GList* gems_teacher_server_deserialize_profiles(GList* list, gchar* serialized_data, guint length);

/* Clear the GList containing pointers to jammo_profile_container structs. Will be called when clearing
 * the teacher_server_profile_data with gems_teacher_server_clear_profile_data_container().
 *
 * Param:
 *  list - GList containing pointers to jammo_profile_container structs
 */
void gems_teacher_server_clear_profile_list(GList* list);

/* Debugging: print jammo_profile_containers in given list,
 * 
 * Param:
 *  list - GList containing pointers to jammo_profile_container structs
 */
void gems_teacher_server_print_profile_list(GList* list);

/* Debugging: print a profile to stdout.
 *
 * Param:
 *  profile - profile to print
 */
void gems_teacher_server_print_profile(jammo_profile_container* profile);

/* Serialize all data in database and encrypt the database. Every profile in the database will be put into huge
 * gchar* ( calls gems_teacher_server_serialize_all_profiles()) and this data is then encrypted. TODO: change 
 * this to use blocks of data! To write the database on to disk call gems_teacher_server_write_profile_database().
 *
 * Params:
 *  data - pointer to database
 *  passwd - password to use for encryption
 * Returns:
 *  TRUE when encryption and serialization are success.
 *  FALSE when something fails
 */
gboolean gems_teacher_server_encrypt_profiles(teacher_server_profile_data* data, gchar* passwd);

/* Decrypt data-field content and add pass decrypted to deserialization. Use this after
 * gems_teacher_server_read_profile_database(). Decrypted profiles will be deserialized into profile list 
 * (calls gems_teacher_server_deserialize_profiles()) of the teacher_server_profile_data struct.
 *
 * Params:
 *  data - pointer to database
 *  passwd - password to use for decryption
 * Returns:
 *  TRUE when decryption and deserialization are success.
 *  FALSE when something fails
 */
gboolean gems_teacher_server_decrypt_profiles(teacher_server_profile_data* data, gchar* passwd);

/* Write encrypted data to disk from database, will return WRITE_FAIL_CONTENT_NOT_SET if data is not encrypted. 
 * Encrypt database before with gems_teacher_server_encrypt_profiles(). Database will be written to 
 * configure_get_jammo_directory()/TSRV_DATABASE. 
 * 
 * Param:
 *  data - pointer to database struct, salt, password hash, data hash and data including length of data must be set.
 * Returns:
 *  WRITE_FAIL_CONTENT_NOT_SET if some of the parameters are missing (the database might not be encrypted yet)
 *  WRITE_FAIL_CANNOT_REMOVE_FILE if the previous database cannot be removed
 *  WRITE_FAIL_CANNOT_SET_FILE_CONTENT if the database cannot be put to file
 *  WRITE_OK when all is written
 */
gint gems_teacher_server_write_profile_database(teacher_server_profile_data* data);

/* Read the encrypted database from disk from configure_get_jammo_directory()/TSRV_DATABASE. When READ_OK is returned
 * decrypt the database with gems_teacher_server_decrypt_profiles().
 * 
 * Param:
 *  data - pointer to newly allocated (gems_teacher_server_new_profile_data_container()) and EMPTY teacher_server_profile_data* struct
 * Returns:
 *  READ_FAIL_FILE_NOT_EXIST if there is no profile file, proceed with empty database
 *  READ_FAIL_CANNOT_READ if the database is unreadable
 *  READ_OK when encrypted database is successfully read
 */
gint gems_teacher_server_read_profile_database(teacher_server_profile_data** data);

#endif
