/*
 *  Copyright (C) 2011 Javier S. Pedro
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <gconf/gconf-client.h>
#include <hildon-extras/he-color-button.h>

#include "settings.h"
#include "widget.h"

#define GCONF_PATH "/apps/maemo/ironcope"

void ironcope_settings_load(IroncopeWidget *self)
{
	GConfClient *gconf = gconf_client_get_default();

	self->peak = gconf_client_get_bool(gconf, GCONF_PATH "/peak", NULL);
	self->red = gconf_client_get_float(gconf, GCONF_PATH "/red", NULL);
	self->green = gconf_client_get_float(gconf, GCONF_PATH "/green", NULL);
	self->blue = gconf_client_get_float(gconf, GCONF_PATH "/blue", NULL);

	g_object_unref(gconf);
}

void ironcope_settings_save(IroncopeWidget *self)
{
	GConfClient *gconf = gconf_client_get_default();

	gconf_client_set_bool(gconf, GCONF_PATH "/peak", self->peak, NULL);
	gconf_client_set_float(gconf, GCONF_PATH "/red", self->red, NULL);
	gconf_client_set_float(gconf, GCONF_PATH "/green", self->green, NULL);
	gconf_client_set_float(gconf, GCONF_PATH "/blue", self->blue, NULL);

	g_object_unref(gconf);
}

static void on_dialog_response(GtkDialog *dialog, gint response_id, gpointer user_data)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(user_data);
	g_return_if_fail(self->settings_window);

	if (response_id == GTK_RESPONSE_OK) {
		int sel = hildon_picker_button_get_active(
			HILDON_PICKER_BUTTON(self->settings_picker));
		self->peak = sel == 0;

		GdkColor color;
		he_color_button_get_color(HE_COLOR_BUTTON(self->settings_color), &color);
		self->red = color.red / 65535.;
		self->green = color.green / 65535.;
		self->blue = color.blue / 65535.;

		ironcope_settings_save(self);
		ironcope_widget_reset(self);
	}

	ironcope_settings_dispose(self);
}

static void build_ui(IroncopeWidget *self)
{
	self->settings_window = gtk_dialog_new_with_buttons(
		_("Ironscope Settings"),
		NULL, 0,
		GTK_STOCK_SAVE, GTK_RESPONSE_OK,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		NULL);
	g_signal_connect(G_OBJECT(self->settings_window), "response",
	                 G_CALLBACK(on_dialog_response), self);

	GtkWidget *vbox = GTK_DIALOG(self->settings_window)->vbox;
	gtk_container_set_border_width(GTK_CONTAINER(vbox), HILDON_MARGIN_DEFAULT);

	GtkWidget *selector = hildon_touch_selector_new_text();
	hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(selector),
		_("Peak amplitude"));
	hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(selector),
		_("Average amplitude"));
	self->settings_picker = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
		HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title(HILDON_BUTTON(self->settings_picker), _("Type"));
	hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(self->settings_picker),
		HILDON_TOUCH_SELECTOR(selector));
	gtk_box_pack_start(GTK_BOX(vbox), self->settings_picker, FALSE, FALSE, 0);

	self->settings_color = he_color_button_new();
	gtk_widget_set_size_request(self->settings_color,
		HILDON_ICON_PIXEL_SIZE_THUMB, HILDON_ICON_PIXEL_SIZE_THUMB);
	gtk_box_pack_start(GTK_BOX(vbox), self->settings_color, FALSE, FALSE, 0);
}

void ironcope_settings_show(IroncopeWidget *self)
{
	if (!self->settings_window) {
		build_ui(self);
	}

	hildon_picker_button_set_active(HILDON_PICKER_BUTTON(self->settings_picker),
		self->peak ? 0 : 1);

	GdkColor color = { 0,
		self->red * 65535., self->green * 65535., self->blue * 65535. };
	he_color_button_set_color(HE_COLOR_BUTTON(self->settings_color), &color);

	gtk_widget_show_all(self->settings_window);
}

void ironcope_settings_dispose(IroncopeWidget *self)
{
	gtk_widget_destroy(self->settings_window);
	self->settings_window = NULL;
	self->settings_picker = NULL;
	self->settings_color = NULL;
}

