/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "oauthutil.h"

#include <QCryptographicHash>
#include <QUrl>


OAuthUtil::OAuthUtil(QObject *parent) :
    QObject(parent) {
}

bool parameterLessThan(const OAuthUtil::Parameter &p1, const OAuthUtil::Parameter &p2) {

    int result = QString::compare(OAuthUtil::urlencodeRFC3986(p1.key),
                                  OAuthUtil::urlencodeRFC3986(p2.key));
    if (result == 0) {
        result = QString::compare(OAuthUtil::urlencodeRFC3986(p1.value),
                                  OAuthUtil::urlencodeRFC3986(p2.value));
    }
    return result < 0;
}

QString OAuthUtil::normalizeParameters(OAuthUtil::ParameterList parameterList) {

    qSort(parameterList.begin(), parameterList.end(), parameterLessThan);

    QStringList normalizedParameters;
    foreach (Parameter parameter, parameterList) {
        normalizedParameters.append(urlencodeRFC3986(parameter.key) + "=" + urlencodeRFC3986(parameter.value));
    }
    return normalizedParameters.join("&");
}

QString OAuthUtil::normalizeParameters(const QVariantMap &parameterMap) {

    return normalizeParameters(ParameterList(parameterMap));
}

QString OAuthUtil::urlencodeRFC3986(const QString &string) {

    return QUrl::toPercentEncoding(string, "~");
}

QString OAuthUtil::urldecodeRFC3986(const QString &string) {

    return QUrl::fromPercentEncoding(string.toUtf8()); // no exta stuff needed for ~, goes correctly automatically
}

QString OAuthUtil::generateBaseString(const QString &httpMethod, const QString &uri, OAuthUtil::ParameterMap parameterMap) {

    if (parameterMap.contains("oauth_signature")) {
        parameterMap.remove("oauth_signature"); // this key is explicitly ignored
    }

    QString parameterString = normalizeParameters(parameterMap);

    return urlencodeRFC3986(httpMethod) + "&" +
           urlencodeRFC3986(uri) + "&" +
           urlencodeRFC3986(parameterString);
}

QString OAuthUtil::calculateHMACSHA1Signature(const QString &baseString, const QString &consumerSecret, const QString &tokenSecret) {

    QByteArray key(urlencodeRFC3986(consumerSecret).toUtf8() + "&" + urlencodeRFC3986(tokenSecret).toUtf8());
    if (key.size() > 64) {
        key = QCryptographicHash::hash(key, QCryptographicHash::Sha1);
    }

    QByteArray okey = key.leftJustified(64, 0);
    QByteArray ikey = key.leftJustified(64, 0);

    for (int i = 0; i < 64; i++) {
        okey[i] = okey[i] ^ 0x5c;
        ikey[i] = ikey[i] ^ 0x36;
    }

    okey.append(QCryptographicHash::hash(ikey.append(baseString), QCryptographicHash::Sha1));
    return QCryptographicHash::hash(okey, QCryptographicHash::Sha1).toBase64();
}

QString OAuthUtil::calculateOAuthSignature(const QString &httpMethod, const QString &uri,
                                           const OAuthUtil::ParameterMap &parameterMap,
                                           const QString &consumerSecret, const QString &oauthTokenSecret) {

    QString baseString = generateBaseString(httpMethod, uri, parameterMap);
    return calculateHMACSHA1Signature(baseString, consumerSecret, oauthTokenSecret);
}

QString OAuthUtil::calculateOAuthSignature(const QString &httpMethod, const QString &uri,
                                           const QVariantMap &parameterMap,
                                           const QString &consumerSecret, const QString &oauthTokenSecret) {

    return calculateOAuthSignature(httpMethod, uri, ParameterMap(parameterMap), consumerSecret, oauthTokenSecret);
}
