#include "events.h"
#include <QDateTime>
#include <QCoreApplication>
#include <QDir>

Events::Events(QObject *parent) :
    QObject(parent),
    m_downloader(new ImageDownloader(this))
{
    m_timer.setInterval(5000);
    m_timer.setSingleShot(true);

    connect(&m_timer, SIGNAL(timeout()), this, SLOT(notify()));
    connect(m_downloader, SIGNAL(imagesDownloaded(Event)), this, SLOT(onImagesDownload(Event)));

    QDir dir("/home/user/Event Feed/.images/");
    dir.mkpath(dir.path());

    clearUnusedImages();
}

qlonglong Events::addEvent(const QString &sourceName, const QString &sourceDisplayName, const QString &icon, const QString &title, const QString &body, const QStringList &images, const QString &footer, QString timestamp, const QString &action) {
    m_timer.stop();

    qsrand(QDateTime::currentMSecsSinceEpoch());
    int eventId = qrand();

    qint64 ts = timestamp.toLongLong();

    if (ts <= 0) {
        QDateTime dt = QDateTime::fromString(timestamp, Qt::ISODate);

        if (!dt.isValid()) {
            dt = QDateTime::fromString(timestamp, Qt::TextDate);

            if (!dt.isValid()) {
                dt = QDateTime::fromString(timestamp.section(' ', 0, -2).remove(','), "ddd d MMM yyyy hh:mm:ss");

                if (!dt.isValid()) {
                    dt = QDateTime::currentDateTime();
                }
            }
        }

        ts = dt.toLocalTime().toMSecsSinceEpoch();
    }

    if ((icon.startsWith("http")) || ((!images.isEmpty()) && (images.first().startsWith("http")))) {
        Event event;
        event.setEventId(eventId);
        event.setSourceName(sourceName);
        event.setSourceDisplayName(sourceDisplayName);
        event.setIconPath(icon);
        event.setTitle(title);
        event.setBody(body.isEmpty() ? tr("No description") : body);
        event.setFooter(footer);
        event.setTimestamp(ts);
        event.setAction(action);

        if (!images.isEmpty()) {
            event.setImagePathOne(images.first());

            if (images.size() > 1) {
                event.setImagePathTwo(images.at(1));
            }
        }

        m_eventQueue.append(event);

        if (m_eventQueue.size() == 1) {
            getNextEvent();
        }

        return eventId;
    }

    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    if (m_database.addEvent(eventId, sourceName, sourceDisplayName, icon, title, body.isEmpty() ? tr("No description") : body, images, footer, ts, action)) {
        m_newEvents++;
        return eventId;
    }

    return -1;
}

void Events::onImagesDownload(const Event &event) {
    if (m_database.addEvent(event.eventId(),
                        event.sourceName(),
                        event.sourceDisplayName(),
                        event.iconPath(),
                        event.title(),
                        event.body(),
                        QStringList() << event.imagePathOne() << event.imagePathTwo(),
                        event.footer(),
                        event.timestamp(),
                        event.action())) {

        m_newEvents++;
    }

    if (!m_eventQueue.isEmpty()) {
        m_eventQueue.removeFirst();
        getNextEvent();
    }
}

void Events::getNextEvent() {
    if (!m_eventQueue.isEmpty()) {
        m_downloader->downloadImages(m_eventQueue.first());
    }
    else {
        m_timer.start();
    }
}

bool Events::removeEvent(qlonglong eventId) {
    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    return m_database.removeEvent(eventId);
}

bool Events::removeEventsBySourceName(const QString &sourceName) {
    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    return m_database.removeEventsBySourceName(sourceName);
}

bool Events::addRefreshAction(const QString &sourceName, const QString &action) {
    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    if (m_database.addRefreshAction(sourceName, action)) {
        emit refreshActionsUpdated();
        return true;
    }

    return false;
}

bool Events::removeRefreshAction(const QString &sourceName) {
    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    if (m_database.removeRefreshAction(sourceName)) {
        emit refreshActionsUpdated();
        return true;
    }

    return false;
}

QString Events::getRefreshAction(const QString &sourceName) {
    if (m_eventQueue.isEmpty()) {
        m_timer.start();
    }

    return m_database.getRefreshAction(sourceName);
}

void Events::notify() {
    if (m_newEvents > 0) {
        emit eventsUpdated();
    }

    QCoreApplication::instance()->quit();
}

void Events::clearUnusedImages() {
    QStringList ids = m_database.getEventIdStrings();
    QDir dir("/home/user/Event Feed/.images/");
    QStringList images = dir.entryList();

    if (ids.isEmpty()) {
        while (!images.isEmpty()) {
            dir.remove(images.takeFirst());
        }
    }
    else {
        while (!images.isEmpty()) {
            QString image = images.takeFirst();

            bool found = false;
            int i = 0;

            while ((!found) && (i < ids.size())) {
                found = image.startsWith(ids.at(i));
                i++;
            }

            if (!found) {
                dir.remove(image);
            }
        }
    }
}
