#include "database.h"
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QVariant>
#include <QDir>
#include <QDebug>

Database::Database(QObject *parent) :
    QObject(parent)
{
    QDir dir;
    dir.mkpath("/home/user/Event Feed/");
    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName("/home/user/Event Feed/events.sqlite");
    initialize();
}

void Database::initialize() {
    if ((m_database.isOpen()) || (m_database.open())) {
        m_database.exec("CREATE TABLE IF NOT EXISTS events (eventId TEXT UNIQUE, sourceName TEXT, sourceDisplayName TEXT, icon TEXT, title TEXT, body TEXT, images TEXT, footer TEXT, timestamp TEXT, action TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS refreshActions (sourceName TEXT UNIQUE, refreshAction TEXT)");
        m_database.close();
    }
    else {
        qDebug() << m_database.lastError().text();
    }
}

Database::~Database() {
    m_database.close();
}

bool Database::addEvent(qint64 eventId, const QString &sourceName, const QString &sourceDisplayName, const QString &icon, const QString &title, const QString &body, const QStringList &images, const QString &footer, qint64 timestamp, const QString &action) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO events VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        query.addBindValue(eventId);
        query.addBindValue(sourceName);
        query.addBindValue(sourceDisplayName);
        query.addBindValue(icon);
        query.addBindValue(title);
        query.addBindValue(body);
        query.addBindValue(images.join(", "));
        query.addBindValue(footer);
        query.addBindValue(timestamp);
        query.addBindValue(action);

        return query.exec();
    }

    return false;
}

bool Database::removeEvent(qint64 eventId) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM events WHERE eventId = ?");
        query.addBindValue(eventId);

        return query.exec();
    }

    return false;
}

bool Database::removeEventsBySourceName(const QString &sourceName) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM events WHERE sourceName = ?");
        query.addBindValue(sourceName);

        return query.exec();
    }

    return false;
}

bool Database::addRefreshAction(const QString &sourceName, const QString &refreshAction) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO refreshActions VALUES (?, ?)");
        query.addBindValue(sourceName);
        query.addBindValue(refreshAction);

        return query.exec();
    }

    return false;
}

bool Database::removeRefreshAction(const QString &sourceName) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM refreshActions WHERE sourceName = ?");
        query.addBindValue(sourceName);

        return query.exec();
    }

    return false;
}

QString Database::getRefreshAction(const QString &sourceName) {
    QString action;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.prepare("SELECT refreshAction FROM refreshActions WHERE sourceName = ?");
        query.addBindValue(sourceName);
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                action = query.value(0).toString();
            }
        }
    }

    return action;
}

QStringList Database::getRefreshActions() {
    QStringList actions;

    if ((m_database.isOpen()) || (m_database.open())) {

        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT refreshAction FROM refreshActions");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                actions.append(query.value(0).toString());
            }
        }
    }

    return actions;
}

QList<Event> Database::getEvents() {
    QList<Event> events;

    if ((m_database.isOpen()) || (m_database.open())) {

        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM events ORDER BY timestamp DESC");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                Event event;
                event.setEventId(query.value(0).toLongLong());
                event.setSourceName(query.value(1).toString());
                event.setSourceDisplayName(query.value(2).toString());
                event.setIconPath(query.value(3).toString());
                event.setTitle(query.value(4).toString());
                event.setBody(query.value(5).toString());
                event.setImagePaths(query.value(6).toString().split(", ", QString::SkipEmptyParts));
                event.setFooter(query.value(7).toString());
                event.setTimestamp(query.value(8).toLongLong());
                event.setAction(query.value(9).toString());
                events.append(event);
            }
        }
    }

    return events;
}
