import re
import httplib
import gnome.gconf
from oauth import oauth
from xml.dom.minidom import parseString
from org.maemo.hermes.engine.phonenumber import PhoneNumber
from org.maemo.hermes.engine.friend import Friend

class LinkedInApi():
    """LinkedIn API for Hermes.
                
       Copyright (c) Fredrik Wendt <fredrik@wendt.se> 2010.
       Copyright (c) Andrew Flegg <andrew@bleb.org> 2010.
       Released under the Artistic Licence."""
    
    LI_API_URL = "https://api.linkedin.com"
    LI_CONN_API_URL = LI_API_URL + "/v1/people/~/connections:(id,first-name,last-name,picture-url,site-standard-profile-request:(url),date-of-birth,main-address,location:(country:(code)),phone-numbers,member-url-resources)"
    LI_PROFILE_API_URL = LI_API_URL + "/v1/people/~"
    
    
    # -----------------------------------------------------------------------
    def __init__(self, consumer, api_request):
        """Instantiate with an OAuth consumer"""
        self.consumer = consumer
        self._make_api_request = api_request
    

    # -----------------------------------------------------------------------
    def verify_verifier(self, access_token):
        try:
            xml = self._make_api_request(self.LI_PROFILE_API_URL)
            dom = parseString(xml)
            friends = self._parse_dom(dom)
            return friends[0].get_name()
        except Exception, e:
            import traceback
            traceback.print_exc()
            raise Exception("LinkedIn authorization failed, try again")

    
    
    # -----------------------------------------------------------------------
    def get_friends(self):
        """ Returns a Friend object for each LinkedIn connection."""
        
        xml = self._make_api_request(self.LI_CONN_API_URL)
        dom = parseString(xml)
        friends = self._parse_dom(dom)

        return friends
        

    # -----------------------------------------------------------------------
    def get_friend_details(self, url, header_value):
        oauth_request = oauth.OAuthRequest.from_consumer_and_token(self.consumer, token=self.access_token, http_url=url)
        oauth_request.sign_request(self.sig_method, self.consumer, self.access_token)

        headers = oauth_request.to_header()
        headers[u'x-li-auth-token'] = header_value
        connection = httplib.HTTPConnection("api.linkedin.com")
        connection.request(oauth_request.http_method, url, headers=headers)
        data = connection.getresponse().read()
        return data


    # -----------------------------------------------------------------------
    def _parse_dom(self, dom):
        def get_first_tag(node, tagName):
            if node:
                tags = node.getElementsByTagName(tagName)
                if tags and len(tags) > 0:
                    return tags[0]
        
        def extract(node, tagName):
            tag = get_first_tag(node, tagName)
            if tag:
                return tag.firstChild.nodeValue
            
        def extract_public_url(node):
            tag = get_first_tag(node, 'site-standard-profile-request')
            if tag:
                url = extract(tag, 'url').replace("&amp;", "&")
                return re.sub('[?&](auth|trk)\w*=[^&]*', '', url)
            
        def extract_phone_numbers(node):
            country = extract(get_first_tag(node, 'country'), 'code')
            tag = get_first_tag(node, 'phone-numbers')
            numbers = []
            if tag:
                for element in tag.childNodes:
                    if element.nodeName != 'phone-number':
                        continue
                     
                    phone_type = extract(element, 'phone-type')
                    device = phone_type == 'mobile' and phone_type or None
                    type = phone_type in set(['home', 'work']) and phone_type or None
                    
                    number = PhoneNumber(extract(element, 'phone-number'), device = device, type = type, country = country)
                    numbers.append(number)
            return numbers
        
        def extract_urls(node):
            tag = get_first_tag(node, 'member-url-resources')
            urls = []
            if tag:
                for element in tag.getElementsByTagName('url'):
                    urls.append(element.firstChild.nodeValue.replace("&amp;", "&"))
            return urls
        
        def extract_birthday(node):
            tag = get_first_tag(node, 'date-of-birth')
            bday = None
            if tag:
                month = extract(tag, 'month')
                day   = extract(tag, 'day')
                year  = extract(tag, 'year')
                if month and day:
                    bday = '%s/%s' % (extract(tag, 'month'), extract(tag, 'day'))
                    if year:
                        bday = '%s/%s' % (bday, year)
                    
            return bday
        
        # look for errors
        errors = dom.getElementsByTagName('error')
        if (errors and len(errors) > 0):
            details = ""
            try:
                details = " (" + extract(errors[0], "message") + ")"
            except:
                pass
            raise Exception("LinkedIn communication errors detected" + details)
        
        friends = []
        people = dom.getElementsByTagName('person')
        for p in people:
            fn = extract(p, 'first-name')
            ln = extract(p, 'last-name')
            photo_url = extract(p, 'picture-url')
            id = extract(p, 'id')
            public_url = extract_public_url(p)
            bday = extract_birthday(p)

            name = fn + " " + ln
            friend = Friend(name)
            friend.add_url(public_url)
            if photo_url:
                friend.set_photo_url(photo_url)
                
            if bday:
                friend.set_birthday_date(bday)
                
            for number in extract_phone_numbers(p):
                friend.add_phone(number)
                
            for url in extract_urls(p):
                friend.add_url(url)
            
            friends.append(friend)

        return friends
