/**
 * contact-update                          (c) Andrew Flegg 2009
 * ~~~~~~~~~~~~~~                          Released under the Artistic Licence.
 *
 * A very quick and dirty program to modify an EDS contact's photo and
 * birthday. This is necessary because evolution-python only exposes the two
 * structs GBoxed, which cannot be created or manipulated in Python (AFAICT).
 *
 * Error handling: limited
 * Syntax: contact-update <name> --photo <mime-type> <file>
 *         contact-update <name> --birthday <year> <month> <day>
 */
#include <libebook/e-book.h>
#include <libebook/e-contact.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>

// gcc -o contact-update -std=c99 `pkg-config --cflags --libs libebook-1.2 glib-2.0` contact-update.c

GError *error = NULL;


/**
 * Simple error handling.
 *
 * @param result Result of an expression. If false, <var>message</var> will be shown
 *               and the process terminated.
 * @param message Message to display if <var>result</var> is false.
 */
void* try(gboolean result, char* message) {
        if (!result) {
                fprintf(stderr, "Error %s: %s\n", message, error ? error->message : "");
                exit(EXIT_FAILURE);
        }
}


/**
 * Load a photo from disk into an EContactPhoto record.
 *
 * @param mime_type MIME type to store in the photo record.
 * @param filename Filename on disk to load.
 */
EContactPhoto load_photo(char* mime_type, char* filename) {
        struct stat stbuf;
        stat(filename, &stbuf);
        char* photo_data = malloc(stbuf.st_size);
        try(photo_data != NULL, "allocating memory for photo");
        FILE *file = fopen(filename, "r");
        try(file != NULL, "opening file");
        int read;
        char* current = photo_data;
        while ((read = fread(current, 1, stbuf.st_size, file)) > 0) {
          current += read;
        }
        fclose(file);
        
        EContactPhoto photo;
        photo.type = E_CONTACT_PHOTO_TYPE_INLINED;
        photo.data.inlined.mime_type = mime_type;
        photo.data.inlined.length = stbuf.st_size;
        photo.data.inlined.data = photo_data;
        
        return photo;
}


/**
 * Create a date record corresponding to the given strings.
 *
 * @param year Character representation of the year.
 * @param month Character representation of the month.
 * @param day Character representation of the day.
 */
EContactDate read_date(char* year, char* month, char* day) {
        EContactDate date;
        date.year  = atoi(year);
        date.month = atoi(month);
        date.day   = atoi(day);
        return date;
}


/**
 * Main entry point: do the work of updating matched records.
 */
int main(int argc, char* argv[]) {
        g_type_init();
        
        try(argc > 2, "\nsyntax: <name> --photo <mime-type> <file>\n        <name> --birthday <year> <month> <day>");

        // -- Data structures we'll use later...
        //
        gboolean is_photo;
        gboolean is_birthday;
        EContactPhoto photo;
        EContactDate  birthday;
        if (strcmp(argv[2], "--photo") == 0) {
          try(argc == 5, "syntax: <name> --photo <mime-type> <file>");
          photo = load_photo(argv[3], argv[4]);
          is_photo = 1;
          
        } else if (strcmp(argv[2], "--birthday") == 0) {
          try(argc == 6, "syntax: <name> --birthday <year> <month> <day>");
          birthday = read_date(argv[3], argv[4], argv[5]);
          is_birthday = 1;
          
        } else {
          try(1 == 0, "\nsyntax: <name> --photo <mime-type> <file>\n        <name> --birthday <year> <month> <day>");
        }
        
        // -- Open the address book...
        //
        EBook *book = e_book_new_system_addressbook(&error);
        try(book != NULL, "finding address book");
        try(e_book_open(book, FALSE, &error), "opening address book");
        try(e_book_is_writable(book), "book is writable");

        // -- Find the list of contacts...
        //
        GList *contacts = NULL;
        EBookQuery *query = e_book_query_field_test(E_CONTACT_FULL_NAME, E_BOOK_QUERY_IS, argv[1]);
        try(e_book_get_contacts(book, query, &contacts, &error), "listing contacts");
        e_book_query_unref(query);
        
        // -- Update the contacts...
        //
        int i = 0;
        for (GList* node = g_list_first(contacts); node != NULL; node = g_list_next(node)) {
          EContact *contact = E_CONTACT(node->data);
          if (is_photo)    e_contact_set(contact, E_CONTACT_PHOTO, &photo);
          if (is_birthday) e_contact_set(contact, E_CONTACT_BIRTH_DATE, &birthday);
            
          if (is_photo || is_birthday) {
            try(e_book_commit_contact(book, contact, &error), "committing contact");
            i++;
          }
        }
        printf("Modified %d records\n", i);

        return EXIT_SUCCESS;
}

