/*
 * headset-control: Allows for the control of the Media Player with a press of the headset button
 *
 * Copyright (C) 2010 Faheem Pervez <trippin1@gmail.com>. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *      
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *      
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

/*
 * ALSA code stolen from http://alsa.opensrc.org/index.php/HowTo_access_a_mixer_control
 */
 
/* headset-control-obj.c */

#include <stdlib.h>
#include <string.h>
#include <alsa/asoundlib.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include "headset-control-obj.h"
#include "hal-marshal.h"
#include "headset-control-service.h"
#include <libplayback/playback.h>
#include <dbus/dbus-glib-bindings.h>

#define HEADSET_CONTROL_TYPE_2_STRINGS G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID
#define HAL_CONDITION "Condition"
#define BUTTON_PRESSED_CONDITION "ButtonPressed"
#define CONDITION_DETAIL_PHONE "phone"
#define HAL_SERVICE "org.freedesktop.Hal"
#define HEADSET_HAL_OBJ_PATH "/org/freedesktop/Hal/devices/computer_logicaldev_input_1"
#define HAL_DEVICE_INTERFACE "org.freedesktop.Hal.Device"

#define HEADSET_CONTROL_NEXT_MODE_KEY HEADSET_CONTROL_GCONF_FOLDER NEXT_MODE_GCONF_KEY
#define WANTED_RENDERER "Mafw-Gst-Renderer"

#define IS_IN_CALL g_str_equal (call_state, MCE_CALL_STATE_RINGING) || g_str_equal (call_state, MCE_CALL_STATE_ACTIVE)

#define HEADSET_CONTROL_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
						HEADSET_TYPE_CONTROL, HeadsetControlPrivate))

struct _HeadsetControlPrivate
{
	DBusGProxy *hal_proxy;
	MafwRenderer *gst_renderer;
	DBusGConnection *sess_dbus_conn;
	pb_playback_t *playback;
	DBusGProxy *mce_signal_proxy;
	
	MafwPlayState state;
	enum pb_state_e new_pb_state;

	gboolean next_mode;
	gboolean suspended;
	gboolean in_call;

	guint gconfnotify_id;
	guint suspend_timer_id;
};

enum
{
	PROP_0,
	PROP_NEXT_MODE,
	PROP_SUSPENDED,
	PROP_IN_CALL,
	PROPCOUNT
};

#include "headset-control-gtype_shit.inc"

/* -- */

static void headset_control_constructed (GObject *);
static void headset_control_call_state_changed (DBusGProxy *, const gchar *, const gchar *, HeadsetControl *);
static void headset_control_button_pressed (DBusGProxy *, const gchar *, const gchar *, HeadsetControl *);
static void headset_control_renderer_removed (MafwRegistry *, MafwRenderer *, HeadsetControl *);
static void headset_control_renderer_added_cb (MafwRegistry *, GObject *, HeadsetControl *);
static gboolean headset_control_suspend_off (HeadsetControl *);
static void headset_control_stop_suspend_off_timer (HeadsetControl *);

/* -- */

static void headset_control_get_property (GObject *object, guint property_id, GValue *value, GParamSpec * pspec)
{
	HeadsetControl *self = HEADSET_CONTROL (object);
 	  	 
	switch (property_id)
	{
		case PROP_NEXT_MODE:
			g_value_set_boolean (value, self->priv->next_mode);
			break;

		case PROP_SUSPENDED: 	  	 
			g_value_set_boolean (value, self->priv->suspended);
			break;

		case PROP_IN_CALL:
			g_value_set_boolean (value, self->priv->in_call);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

static void headset_control_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec * pspec)
{
	HeadsetControl *self = HEADSET_CONTROL (object);
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);
 	  	 
	switch (property_id)
	{
		case PROP_NEXT_MODE:
			self->priv->next_mode = g_value_get_boolean (value);
			g_object_notify (object, "next-mode");
			break;

		case PROP_SUSPENDED:
			self->priv->suspended = g_value_get_boolean (value);
			g_object_notify (object, "suspended");
			break;

		case PROP_IN_CALL:	  	 
			self->priv->in_call = g_value_get_boolean (value);
			g_object_notify (object, "in-call");
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
			break;
	}
}

/* -- */

static void headset_control_finalize (GObject *object)
{
	HeadsetControl *self = HEADSET_CONTROL (object);
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);

	DBusGProxy *bus_proxy = dbus_g_proxy_new_for_name (klass->sys_dbus_conn, DBUS_SERVICE_DBUS, DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);
	if (bus_proxy)
	{
		org_freedesktop_DBus_release_name (bus_proxy, HEADSET_CONTROL_SERVICE_NAME, NULL, NULL);
		g_object_unref (G_OBJECT (bus_proxy));
	}

	headset_control_stop_suspend_off_timer (self);

	self->priv->state = _LastMafwPlayState;

	if (self->priv->mce_signal_proxy)
	{
		dbus_g_proxy_disconnect_signal (self->priv->mce_signal_proxy, MCE_CALL_STATE_SIG, G_CALLBACK (headset_control_call_state_changed), self);
		g_object_unref (G_OBJECT (self->priv->mce_signal_proxy));
		self->priv->mce_signal_proxy = NULL;
	}

	if (self->priv->hal_proxy)
	{
		dbus_g_proxy_disconnect_signal (self->priv->hal_proxy, HAL_CONDITION, G_CALLBACK (headset_control_button_pressed), self);
		g_object_unref (G_OBJECT (self->priv->hal_proxy));
		self->priv->hal_proxy = NULL;
	}

	g_signal_handlers_disconnect_by_func (klass->mafw_registry, G_CALLBACK (headset_control_renderer_added_cb), self); /* Don't really desire much for renderers to be added... */
	headset_control_renderer_removed (NULL, self->priv->gst_renderer, self);

	if (self->priv->playback)
	{
		pb_playback_destroy (self->priv->playback);
		self->priv->playback = NULL;
	}

	if (self->priv->sess_dbus_conn)
	{
		dbus_g_connection_flush (self->priv->sess_dbus_conn);
		dbus_g_connection_unref (self->priv->sess_dbus_conn);
		self->priv->sess_dbus_conn = NULL;
	}

	if (self->priv->gconfnotify_id != 0)
	{
		gconf_client_notify_remove (klass->gconf_client, self->priv->gconfnotify_id);
		self->priv->gconfnotify_id = 0;
		gconf_client_remove_dir (klass->gconf_client, HEADSET_CONTROL_GCONF_FOLDER, NULL);
	}

	if (klass->gconf_client)
	{
		gconf_client_clear_cache (klass->gconf_client);
		g_object_unref (klass->gconf_client);
		klass->gconf_client = NULL;
	}

	mafw_shared_deinit ();
	klass->mafw_registry = NULL;

	if (klass->sys_dbus_conn)
	{
		dbus_g_connection_flush (klass->sys_dbus_conn);
		dbus_g_connection_unref (klass->sys_dbus_conn);
		klass->sys_dbus_conn = NULL;
	}
	
	self->priv->state = _LastMafwPlayState;

	if (G_OBJECT_CLASS (headset_control_parent_class)->finalize)
		G_OBJECT_CLASS (headset_control_parent_class)->finalize (object);
}

/* -- */

static void headset_control_class_init (HeadsetControlClass *klass)
{
	GError *error = NULL;

	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->set_property = headset_control_set_property;
	object_class->get_property = headset_control_get_property;
	object_class->finalize = (GObjectFinalizeFunc) headset_control_finalize;
	object_class->constructed = headset_control_constructed;

	klass->sys_dbus_conn = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (!klass->sys_dbus_conn)
	{
		g_error ("Failed to open connection to system bus: %s\n", error->message);
		g_clear_error (&error); /* Unlikely that we shall get called, but, anyway... */
	}

	klass->mafw_registry = MAFW_REGISTRY (mafw_registry_get_instance ());
	if (!klass->mafw_registry)
		g_error ("No MAFW Registry object.");

	if (!mafw_shared_init (klass->mafw_registry, &error))
	{
		g_error ("Failed to init MAFW: %s\n", error->message);
		g_clear_error (&error); /* Unlikely that we shall get called, but, anyway... */
	}

	klass->gconf_client = gconf_client_get_default ();

	g_object_class_install_property (object_class, PROP_NEXT_MODE, g_param_spec_boolean ("next-mode", "Next Mode", "If TRUE, pressing the headset button will skip to the next track, instead.", FALSE, G_PARAM_READWRITE));
	g_object_class_install_property (object_class, PROP_SUSPENDED, g_param_spec_boolean ("suspended", "60 sec. pause active", "When TRUE, Headset Control will not do anything for 60 seconds when the button is pressed.", FALSE, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_IN_CALL, g_param_spec_boolean ("in-call", "Self-explanatory, I'd've thought...", "When TRUE, Headset Control will not do anything because the phone is in a call.", FALSE, G_PARAM_READABLE));

	dbus_g_object_type_install_info (HEADSET_TYPE_CONTROL, &dbus_glib_headset_control_object_info);
	g_type_class_add_private (klass, sizeof (HeadsetControlPrivate));
}

/* -- */

static void headset_control_stop_suspend_off_timer (HeadsetControl *self)
{
	if (self->priv->suspend_timer_id != 0)
		g_source_remove (self->priv->suspend_timer_id);
	self->priv->suspend_timer_id = 0;
}

static void headset_control_start_suspend_off_timer (HeadsetControl *self)
{
	headset_control_stop_suspend_off_timer (self);
	self->priv->suspend_timer_id = g_timeout_add_seconds (60, (GSourceFunc) headset_control_suspend_off, self);
}

static gboolean headset_control_enable_headset_button (gconstpointer user_data G_GNUC_UNUSED) /* Fuck you, Nokia */
{
	gchar *type_contents = NULL;
	gint type_int = 0;
	const gchar *card = "hw:0";
	const gchar *bias_switch = "Jack Bias Switch";
	gint err = 0;
	snd_hctl_t *hctl = NULL;
	snd_ctl_elem_id_t *id;
	snd_hctl_elem_t *elem;
	snd_ctl_elem_value_t *control;

	/* Check to see if we have a Nokia headset with a button */
	if (!g_file_get_contents ("/sys/devices/platform/nokia-av/type", &type_contents, NULL, NULL))
		goto ret_false;	/* Don't wish to risk anything */

	type_int = atoi (type_contents);
	g_free (type_contents);
	
	if (type_int != 4)
		goto ret_false;

	err = snd_hctl_open (&hctl, card, 0);
	if (err < 0)
		goto ret_false;

	err = snd_hctl_load (hctl);
	if (err < 0)
		goto close_hctl;

	snd_ctl_elem_id_alloca (&id);
	snd_ctl_elem_id_set_interface (id, SND_CTL_ELEM_IFACE_MIXER);
	snd_ctl_elem_id_set_name (id, bias_switch);

	elem = snd_hctl_find_elem (hctl, id);
	if (!elem)
		goto close_hctl;

	snd_ctl_elem_value_alloca (&control);
	snd_ctl_elem_value_set_id (control, id);
	snd_ctl_elem_value_set_boolean (control, 0, TRUE);
	snd_hctl_elem_write (elem, control);

close_hctl:
	if (hctl)
		snd_hctl_close (hctl);

ret_false:
	return FALSE;
}

/* -- */

static gboolean headset_control_suspend_off (HeadsetControl *self)
{
	self->priv->suspended = FALSE;
	self->priv->suspend_timer_id = 0;
	return FALSE;
}

static void headset_control_libplayback_state_reply (pb_playback_t *pb, enum pb_state_e granted_state G_GNUC_UNUSED, const char *reason G_GNUC_UNUSED, pb_req_t *req, void *data G_GNUC_UNUSED)
{
	pb_playback_req_completed (pb, req);
}

static void headset_control_rend_command_sent (MafwRenderer *renderer G_GNUC_UNUSED, HeadsetControl *self, const GError *error)
{
	if (error || !self->priv->playback)
		return;

	pb_playback_req_state (self->priv->playback, self->priv->new_pb_state, (PBStateReply) headset_control_libplayback_state_reply, NULL);
}

static void headset_control_button_pressed (DBusGProxy *hal_proxy G_GNUC_UNUSED, const gchar *condition_name, const gchar *condition_detail, HeadsetControl *self)
{
	if (!g_str_equal (condition_name, BUTTON_PRESSED_CONDITION) || !g_str_equal (condition_detail, CONDITION_DETAIL_PHONE))
		return;

	if (self->priv->in_call)
		return;

	if (self->priv->suspended)
	{
		g_warning ("Sixty Second Pause mode is active!");
		return;
	}

	if (self->priv->next_mode)
	{
		mafw_renderer_next (self->priv->gst_renderer, NULL, NULL);
		return;
	}

	switch (self->priv->state)
	{
		case Playing:
			self->priv->new_pb_state = PB_STATE_STOP;
			mafw_renderer_pause (self->priv->gst_renderer, (MafwRendererPlaybackCB) headset_control_rend_command_sent, self);
			break;

		case Paused:
			self->priv->new_pb_state = PB_STATE_PLAY;
			mafw_renderer_resume (self->priv->gst_renderer, (MafwRendererPlaybackCB) headset_control_rend_command_sent, self);
			break;

		case Stopped:
			self->priv->new_pb_state = PB_STATE_PLAY;
			mafw_renderer_play (self->priv->gst_renderer, (MafwRendererPlaybackCB) headset_control_rend_command_sent, self);
			break;

		default: /* What do we do when it is Transitioning? */
			g_return_if_reached ();
	}
}

static void headset_control_state_changed (MafwRenderer *renderer G_GNUC_UNUSED, MafwPlayState state, HeadsetControl *self)
{
        self->priv->state = state;
}

static void headset_control_status_recieved (MafwRenderer *renderer, MafwPlaylist *playlist G_GNUC_UNUSED, guint index G_GNUC_UNUSED, MafwPlayState state, const gchar *object_id G_GNUC_UNUSED, HeadsetControl *self, const GError *error)
{
	if (error)
		return;
	
	headset_control_state_changed (renderer, state, self);
}

static void headset_control_renderer_added_cb (MafwRegistry *registry G_GNUC_UNUSED, GObject *renderer, HeadsetControl *self)
{
	if (MAFW_IS_RENDERER (renderer))
	{
		const gchar *name = mafw_extension_get_name (MAFW_EXTENSION (renderer));

		if (g_str_equal (name, WANTED_RENDERER))
		{
			self->priv->gst_renderer = g_object_ref (renderer); /* Throw some D's^H^H^Hrefs on it */
			g_signal_connect (self->priv->gst_renderer, "state-changed", G_CALLBACK (headset_control_state_changed), self);
			mafw_renderer_get_status (self->priv->gst_renderer, (MafwRendererStatusCB) headset_control_status_recieved, self);
		}
	}
}

static void headset_control_renderer_removed (MafwRegistry *registry G_GNUC_UNUSED, MafwRenderer *renderer, HeadsetControl *self)
{
	if (MAFW_IS_RENDERER (renderer))
	{
		if (self->priv->gst_renderer != NULL && renderer == self->priv->gst_renderer)
		{
			g_signal_handlers_disconnect_by_func (self->priv->gst_renderer, G_CALLBACK (headset_control_state_changed), self);
			g_object_unref (self->priv->gst_renderer);
			self->priv->gst_renderer = NULL;
		}
	}
}

static void headset_control_libplayback_state_request_handler (pb_playback_t *pb, enum pb_state_e req_state G_GNUC_UNUSED, pb_req_t *req, void *data G_GNUC_UNUSED)
{
    pb_playback_req_completed (pb, req); /* Couldn't care less */
}

static void headset_control_next_mode_key_changed (GConfClient *gconf_client G_GNUC_UNUSED, guint cnxn_id G_GNUC_UNUSED, GConfEntry *entry, HeadsetControl *self)
{
	g_object_set (G_OBJECT (self), "next-mode", gconf_value_get_bool (entry->value), NULL);
}

static void headset_control_call_state_changed (DBusGProxy *mce_proxy G_GNUC_UNUSED, const gchar *call_state, const gchar *emergency_call_state G_GNUC_UNUSED, HeadsetControl *self)
{
	self->priv->in_call = IS_IN_CALL;

	if (!self->priv->in_call)
		g_timeout_add_seconds (5, (GSourceFunc) headset_control_enable_headset_button, NULL); /* Stupidest fucking hack ever */
}

/* -- */

static void headset_control_constructed (GObject *object)
{
	HeadsetControl *self = HEADSET_CONTROL (object);

	if (G_OBJECT_CLASS (headset_control_parent_class)->constructed)
		G_OBJECT_CLASS (headset_control_parent_class)->constructed (object);

	dbus_g_proxy_connect_signal (self->priv->hal_proxy, HAL_CONDITION, G_CALLBACK (headset_control_button_pressed), self, NULL);
	(void) headset_control_enable_headset_button (NULL);
}

static inline void headset_control_mce_phone_watcher_init (HeadsetControl *self)
{
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);
	DBusGProxy *mce_request_proxy;

	g_return_if_fail (klass->sys_dbus_conn);
	
	mce_request_proxy = dbus_g_proxy_new_for_name (klass->sys_dbus_conn, MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF);
	if (mce_request_proxy)
	{
		gchar *call_state = NULL;
		gchar *emergency_call_state = NULL;
		
		if (dbus_g_proxy_call (mce_request_proxy, MCE_CALL_STATE_GET, NULL, G_TYPE_INVALID, G_TYPE_STRING, &call_state, G_TYPE_STRING, &emergency_call_state, G_TYPE_INVALID))
		{
			self->priv->in_call = IS_IN_CALL;

			g_free (call_state);
			g_free (emergency_call_state);

			g_object_unref (G_OBJECT (mce_request_proxy));
		}
	}

	self->priv->mce_signal_proxy = dbus_g_proxy_new_for_name (klass->sys_dbus_conn, MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF);
	if (!self->priv->mce_signal_proxy)
		return;

	dbus_g_proxy_add_signal (self->priv->mce_signal_proxy, MCE_CALL_STATE_SIG, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (self->priv->mce_signal_proxy, MCE_CALL_STATE_SIG, G_CALLBACK (headset_control_call_state_changed), self, NULL);
}

static inline void headset_control_hal_proxy_init (HeadsetControl *self)
{
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);

	g_assert (klass->sys_dbus_conn);

	self->priv->hal_proxy = dbus_g_proxy_new_for_name (klass->sys_dbus_conn, HAL_SERVICE, HEADSET_HAL_OBJ_PATH, HAL_DEVICE_INTERFACE);
	if (!self->priv->hal_proxy)
	{
		g_error ("My God! We haz no HAL proxy!");
		g_object_unref (self);
	}

	dbus_g_object_register_marshaller (hal_marshal_VOID__STRING_STRING, G_TYPE_NONE, HEADSET_CONTROL_TYPE_2_STRINGS);
	dbus_g_proxy_add_signal (self->priv->hal_proxy, HAL_CONDITION, HEADSET_CONTROL_TYPE_2_STRINGS);
}

static inline void headset_control_mafw_renderer_init (HeadsetControl *self)
{
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);
	GList *renderers = NULL;

	g_assert (klass->mafw_registry);

	g_signal_connect (klass->mafw_registry, "renderer-added", G_CALLBACK (headset_control_renderer_added_cb), self);
	g_signal_connect (klass->mafw_registry, "renderer-removed", G_CALLBACK (headset_control_renderer_removed), self);

	renderers = mafw_registry_get_renderers (klass->mafw_registry);
	while (renderers)
	{
		headset_control_renderer_added_cb (klass->mafw_registry, G_OBJECT (renderers->data), self);
		renderers = g_list_next (renderers);
	}
}

static inline void headset_control_playback_init (HeadsetControl *self)
{
	enum pb_state_e pb_state = self->priv->state == Playing ? PB_STATE_PLAY : PB_STATE_STOP;
	self->priv->sess_dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
	
	if (!self->priv->sess_dbus_conn)
		return;

	self->priv->playback = pb_playback_new_2 (dbus_g_connection_get_connection (self->priv->sess_dbus_conn), PB_CLASS_MEDIA, PB_FLAG_AUDIO, pb_state, (PBStateRequest) headset_control_libplayback_state_request_handler, NULL);
}

static inline void headset_control_gconf_init (HeadsetControl *self)
{
	GError *error = NULL;
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);

	g_return_if_fail (klass->gconf_client);

	gboolean next_mode = gconf_client_get_bool (klass->gconf_client, HEADSET_CONTROL_NEXT_MODE_KEY, NULL);	
	g_object_set (G_OBJECT (self), "next-mode", next_mode, NULL);

	gconf_client_add_dir (klass->gconf_client, HEADSET_CONTROL_GCONF_FOLDER, GCONF_CLIENT_PRELOAD_NONE, &error);
	if (error)
	{
		g_warning ("Failed to add directory for watching: %s\n", error->message);
		g_clear_error (&error);
		return;
	}

	self->priv->gconfnotify_id = gconf_client_notify_add (klass->gconf_client, HEADSET_CONTROL_NEXT_MODE_KEY, (GConfClientNotifyFunc) headset_control_next_mode_key_changed, self, NULL, &error);
	if (error)
	{
		g_warning ("Failed to add next-mode key for watching: %s\n", error->message);
		g_clear_error (&error);
		gconf_client_remove_dir (klass->gconf_client, HEADSET_CONTROL_GCONF_FOLDER, NULL);
	}
}

static inline void headset_control_dbus_server_init (HeadsetControl *self)
{
	HeadsetControlClass *klass = HEADSET_CONTROL_GET_CLASS (self);

	guint request_name_ret;
	GError *error = NULL;
	DBusGProxy *bus_proxy;

	bus_proxy = dbus_g_proxy_new_for_name (klass->sys_dbus_conn, DBUS_SERVICE_DBUS, DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);
	org_freedesktop_DBus_request_name (bus_proxy, HEADSET_CONTROL_SERVICE_NAME, DBUS_NAME_FLAG_DO_NOT_QUEUE, &request_name_ret, &error);
	if (error)
	{
		g_warning ("Failed to request name: %s\n", error->message);
		g_clear_error (&error);
		g_object_unref (G_OBJECT (bus_proxy));
		return;
	}

	switch (request_name_ret)
	{
		case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
			dbus_g_connection_register_g_object (klass->sys_dbus_conn, HEADSET_CONTROL_OBJECT_PATH, G_OBJECT (self));
			g_object_unref (G_OBJECT (bus_proxy));
			break;

		case DBUS_REQUEST_NAME_REPLY_EXISTS:
			g_warning ("Already running\n");
			g_object_unref (G_OBJECT (bus_proxy));
			g_object_unref (G_OBJECT (self));

		default:
			g_warning ("Some other motherfucker took my name!");
			g_object_unref (G_OBJECT (bus_proxy));
	}
}

static void headset_control_init (HeadsetControl *self)
{
	self->priv = HEADSET_CONTROL_GET_PRIVATE (self);
	g_return_if_fail (G_LIKELY (self->priv != NULL));

	memset (self->priv, 0, sizeof (HeadsetControlPrivate));
	self->priv->state = _LastMafwPlayState;
	self->priv->suspended = FALSE;

	headset_control_gconf_init (self); /* A shame gconf-bridge sucks cock */
	headset_control_hal_proxy_init (self);
	headset_control_mafw_renderer_init (self);
	headset_control_playback_init (self);
	headset_control_mce_phone_watcher_init (self);
	headset_control_dbus_server_init (self);
}

/* -- */

/* Our (one) D-Bus method! */
gboolean headset_control_sixty_second_pause (HeadsetControl *self, GError **error G_GNUC_UNUSED)
{
	self->priv->suspended = TRUE;
	headset_control_start_suspend_off_timer (self);

	return TRUE;
}

/* -- */

HeadsetControl* headset_control_new (void)
{
	return HEADSET_CONTROL (g_object_new (HEADSET_TYPE_CONTROL, NULL));
}
