#include <gst/gst.h>

#if !GST_CHECK_VERSION(0,10,33)
#include <gst/base/gstbytereader.h>
#include <string.h>

#include <gst/tag/tag.h>

static void
gst_vorbis_tag_add_coverart (GstTagList * tags, gchar * img_data_base64,
    gint base64_len)
{
  GstBuffer *img;
  gsize img_len;

  if (base64_len < 2)
    goto not_enough_data;

  /* img_data_base64 points to a temporary copy of the base64 encoded data, so
   * it's safe to do inpace decoding here
   */
  g_base64_decode_inplace (img_data_base64, &img_len);
  if (img_len == 0)
    goto decode_failed;

  img =
      gst_tag_image_data_to_image_buffer ((const guint8 *) img_data_base64,
      img_len, GST_TAG_IMAGE_TYPE_NONE);

  if (img == NULL)
    goto convert_failed;

  gst_tag_list_add (tags, GST_TAG_MERGE_APPEND,
      GST_TAG_PREVIEW_IMAGE, img, NULL);

  gst_buffer_unref (img);
  return;

/* ERRORS */
not_enough_data:
  {
    GST_WARNING ("COVERART tag with too little base64-encoded data");
    return;
  }
decode_failed:
  {
    GST_WARNING ("Couldn't decode base64 image data from COVERART tag");
    return;
  }
convert_failed:
  {
    GST_WARNING ("Couldn't extract image or image type from COVERART tag");
    return;
  }
}

/* Standardized way of adding pictures to vorbiscomments:
 * http://wiki.xiph.org/VorbisComment#METADATA_BLOCK_PICTURE
 */
static void
gst_vorbis_tag_add_metadata_block_picture (GstTagList * tags,
    gchar * value, gint value_len)
{
  GstByteReader reader;
  guint32 img_len = 0, img_type = 0;
  guint32 img_mimetype_len = 0, img_description_len = 0;
  gsize decoded_len;
  const guint8 *data = NULL;

  /* img_data_base64 points to a temporary copy of the base64 encoded data, so
   * it's safe to do inpace decoding here
   */
  g_base64_decode_inplace (value, &decoded_len);
  if (decoded_len == 0)
    goto decode_failed;

  gst_byte_reader_init (&reader, (guint8 *) value, decoded_len);

  if (!gst_byte_reader_get_uint32_be (&reader, &img_type))
    goto error;

  if (!gst_byte_reader_get_uint32_be (&reader, &img_mimetype_len))
    goto error;
  if (!gst_byte_reader_skip (&reader, img_mimetype_len))
    goto error;

  if (!gst_byte_reader_get_uint32_be (&reader, &img_description_len))
    goto error;
  if (!gst_byte_reader_skip (&reader, img_description_len))
    goto error;

  /* Skip width, height, color depth and number of colors for
   * indexed formats */
  if (!gst_byte_reader_skip (&reader, 4 * 4))
    goto error;

  if (!gst_byte_reader_get_uint32_be (&reader, &img_len))
    goto error;

  if (!gst_byte_reader_get_data (&reader, img_len, &data))
    goto error;

  gst_tag_list_add_id3_image (tags, data, img_len, img_type);

  return;

error:
  GST_WARNING
      ("Couldn't extract image or image type from METADATA_BLOCK_PICTURE tag");
  return;
decode_failed:
  GST_WARNING ("Failed to decode Base64 data from METADATA_BLOCK_PICTURE tag");
  return;
}

/**
 * gst_tag_list_from_vorbiscomment_buffer:
 * @buffer: buffer to convert
 * @id_data: identification data at start of stream
 * @id_data_length: length of identification data
 * @vendor_string: pointer to a string that should take the vendor string
 *                 of this vorbis comment or NULL if you don't need it.
 *
 * Creates a new tag list that contains the information parsed out of a
 * vorbiscomment packet.
 *
 * Returns: A new #GstTagList with all tags that could be extracted from the
 *          given vorbiscomment buffer or NULL on error.
 */
GstTagList *
gst_hack_tag_list_from_vorbiscomment_buffer (const GstBuffer * buffer,
    const guint8 * id_data, const guint id_data_length, gchar ** vendor_string)
{
#define ADVANCE(x) G_STMT_START{                                                \
  data += x;                                                                    \
  size -= x;                                                                    \
  if (size < 4) goto error;                                                     \
  cur_size = GST_READ_UINT32_LE (data);                                         \
  data += 4;                                                                    \
  size -= 4;                                                                    \
  if (cur_size > size) goto error;                                              \
  cur = (gchar*)data;                                                                   \
}G_STMT_END
  gchar *cur, *value;
  guint cur_size;
  guint iterations;
  guint8 *data;
  guint size, value_len;
  GstTagList *list;

  g_return_val_if_fail (GST_IS_BUFFER (buffer), NULL);
  g_return_val_if_fail (id_data != NULL || id_data_length == 0, NULL);

  data = GST_BUFFER_DATA (buffer);
  size = GST_BUFFER_SIZE (buffer);
  list = gst_tag_list_new ();

  if (size < 11 || size <= id_data_length + 4)
    goto error;

  if (id_data_length > 0 && memcmp (data, id_data, id_data_length) != 0)
    goto error;

  ADVANCE (id_data_length);

  if (vendor_string)
    *vendor_string = g_strndup (cur, cur_size);

  ADVANCE (cur_size);
  iterations = cur_size;
  cur_size = 0;

  while (iterations) {
    ADVANCE (cur_size);
    iterations--;
    cur = g_strndup (cur, cur_size);
    value = strchr (cur, '=');
    if (value == NULL) {
      g_free (cur);
      continue;
    }
    *value = '\0';
    value++;
    value_len = strlen (value);
    if (value_len == 0 || !g_utf8_validate (value, value_len, NULL)) {
      g_free (cur);
      continue;
    }
    /* we'll just ignore COVERARTMIME and typefind the image data */
    if (g_ascii_strcasecmp (cur, "COVERARTMIME") == 0) {
      continue;
    } else if (g_ascii_strcasecmp (cur, "COVERART") == 0) {
      gst_vorbis_tag_add_coverart (list, value, value_len);
    } else if (g_ascii_strcasecmp (cur, "METADATA_BLOCK_PICTURE") == 0) {
      gst_vorbis_tag_add_metadata_block_picture (list, value, value_len);
    } else {
      gst_vorbis_tag_add (list, cur, value);
    }
    g_free (cur);
  }

  return list;

error:
  gst_tag_list_free (list);
  return NULL;
#undef ADVANCE
}
#endif
