#include <QMenuBar>
#include <QDebug>
#include <QPainter>
#include "entrieswindow.h"
#include "contentwindow.h"

EntriesWindow::EntriesWindow(QWidget *parent, Feed *f) : QMainWindow(parent) {
	setAttribute(Qt::WA_Maemo5StackedWindow);

	feed = f;

	menuBar()->addAction(tr("Fetch more"), this, SLOT(sync()));
	menuBar()->addAction(tr("Mark all as read"), this, SLOT(markRead()));

	QActionGroup *filter_group = new QActionGroup(this);
	show_all = new QAction(tr("Show All"), filter_group);
	show_all->setCheckable(true);
	show_all->setChecked(true);
	show_updated = new QAction(tr("Show Updated"), filter_group);
	show_updated->setCheckable(true);
	menuBar()->addActions(filter_group->actions());

	setWindowTitle(f->title);

	list = new QListView();
	setCentralWidget(list);
	list->setItemDelegate(new EntryListDelegate(this));

	connect(list, SIGNAL(activated(const QModelIndex &)),
		SLOT(entrySelected(const QModelIndex &)));

	connect(feed, SIGNAL(updateFeedComplete()),
		SLOT(entriesUpdated()));

	if(feed->getEntriesSize() == 0 || feed->lastUpdated < feed->reader->lastUpdated)
		sync();
	else
		entriesUpdated();
}

EntriesWindow::~EntriesWindow() {
}

void EntriesWindow::sync() {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
	feed->fetch(feed->lastUpdated > feed->reader->lastUpdated);
}

void EntriesWindow::markRead() {
	feed->markRead();
}

void EntriesWindow::entriesUpdated() {
	QList<Entry *>entries = feed->getEntries();
	QAbstractItemModel *old_model = list->model();
	EntryListModel *new_model = new EntryListModel(this, entries, show_updated->isChecked());
	list->setModel(new_model);
	connect(show_updated, SIGNAL(toggled(bool)), new_model, SLOT(showUpdated(bool)));
	delete(old_model);
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void EntriesWindow::entrySelected(const QModelIndex &index) {
	Entry *e = qVariantValue<Entry *>(index.data());

	ContentWindow *w = new ContentWindow(this, e);
	w->show();
}

int EntryListModel::rowCount(const QModelIndex &) const {
	int size = 0;
	
	if(show_updated) {
		for(int i = 0; i < entry_list.size(); i++) {
			if((entry_list.at(i)->flags & ENTRY_FLAG_READ) == 0)
				size++;
		}
	}
	else
		size = entry_list.size();

	return size;;
}

QVariant EntryListModel::data(const QModelIndex &index, int role) const {
	if(!index.isValid())
		return QVariant();

	if(index.row() >= rowCount() || index.row() < 0) {
		return QVariant();
	}

	if(role == Qt::DisplayRole) {
		if(show_updated) {
			int i, j;
			for(i = 0, j = 0; i < entry_list.size(); i++) {
				if((entry_list.at(i)->flags & ENTRY_FLAG_READ) == 0) j++;
				if(j > index.row())
					return qVariantFromValue(entry_list.at(i));
			}
		}
		else
			return qVariantFromValue(entry_list.at(index.row()));
	}

	return QVariant();
}

void EntryListModel::showUpdated(bool updated) {
	beginResetModel();
	show_updated = updated;
	endResetModel();
}

void EntryListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option,
	const QModelIndex &index) const {
    
	QStyledItemDelegate::paint(painter, option, index);

	Entry *e = qVariantValue<Entry *>(index.data());

	QFont font = option.font;
	QRect rect = option.rect;
	rect.adjust(20, 8, -20, -8);
	QPoint topleft = rect.topLeft();
	QPoint bottomleft = rect.bottomLeft();
	rect.adjust(36, 0, 0, 0);

	painter->save();

	if(((e->flags & ENTRY_FLAG_READ) == 0) &&
	   !(option.state & QStyle::State_Selected)) {
		painter->setPen(option.palette.highlight().color());
	}

	painter->drawText(rect, Qt::AlignTop | Qt::AlignLeft, e->title);

	painter->setPen(option.palette.mid().color());
	font.setPointSizeF(font.pointSizeF() * 0.70);
	painter->setFont(font);

	painter->drawText(rect, Qt::AlignBottom | Qt::AlignLeft, e->author);

	QString date;
	if(e->published.date() == QDateTime::currentDateTime().date())
		date = e->published.time().toString(Qt::DefaultLocaleShortDate);
	else
		date = e->published.date().toString();

	painter->drawText(rect, Qt::AlignBottom | Qt::AlignRight, date);

	if(e->flags & ENTRY_FLAG_STARRED) {
		QImage img = QImage(QLatin1String(":/images/star-1"));
		painter->drawImage(topleft, img);
	}

	if(e->flags & ENTRY_FLAG_SHARED) {
		QImage img = QImage(QLatin1String(":/images/shared-1"));
		bottomleft.ry() -= img.height();
		painter->drawImage(bottomleft, img);
	}

	painter->restore();
}

