#include <hildon/hildon.h>

#include "../common.h"

// Screen (window) size
static int screen_w = 800;
static int screen_h = 480;

// Video mode
static int vmode_w = 240;
static int vmode_h = 160;

// Video filtering
static int bounce_buf[400 * 272 * 2 / 4];
static enum software_filter {
  SWFILTER_NONE = 0,
  SWFILTER_SCALE2X,
  SWFILTER_SCALE3X,
  SWFILTER_EAGLE2X,
} sw_filter;

// Key bindings file
static char *keys_path = NULL;

// Virtual keypad
static GtkWidget *keypad_widget;
static HildonAnimationActor *keypad_actor;

// Video output
static GdkImage *vout_image = NULL;
static GtkWidget *vout_widget;
static HildonAnimationActor *vout_actor;
static GtkWidget *window;

// Pause pop-up
static GtkDialog *pause_dialog = NULL;
static int autopause = FALSE;
static int pause_timeout = 0;
static int pause_key_down = FALSE;
static int pause_gesture_start_x;
static int pause_gesture_start_y;
static void activate_pause(int autoresume);

// Input states for keyboard, accelerometer and touch screen
static int keybd_keystate = 0;
static int combo_keystate = 0;
static int accel_keystate = 0;
static int touch_keystate = 0;
static int touch_idle = TRUE;

// Map hardware key codes to local key numbers
static unsigned char keymap[65536];
static int use_zoom_keys = FALSE;

// Accelerometer parameters
static int accel_enable = FALSE;
static int accel_deadzone = 150;
static int accel_center_y = 150;
static int accel_center_x = 0;

// Local key masks
enum gpsp_key {
  // Basic keys
  GKEY_UP          = 1 << 0,
  GKEY_DOWN        = 1 << 1,
  GKEY_LEFT        = 1 << 2,
  GKEY_RIGHT       = 1 << 3,
  GKEY_START       = 1 << 4,
  GKEY_SELECT      = 1 << 5,
  GKEY_A           = 1 << 6,
  GKEY_B           = 1 << 7,
  GKEY_L           = 1 << 8,
  GKEY_R           = 1 << 9,
  // Diagonals
  GKEY_LEFT_UP     = 1 << 10,
  GKEY_RIGHT_UP    = 1 << 11,
  GKEY_LEFT_DOWN   = 1 << 12,
  GKEY_RIGHT_DOWN  = 1 << 13,
  // Special keys
  GKEY_LOADSTATE   = 1 << 14,
  GKEY_SAVESTATE   = 1 << 15,
  GKEY_FASTFORWARD = 1 << 16,
  GKEY_FPS         = 1 << 17,
  GKEY_QUIT        = 1 << 18,
  GKEY_PAUSE       = 1 << 19
};

// Local key order
u32 button_plat_mask_to_config[PLAT_BUTTON_COUNT] =
{
  GKEY_UP,
  GKEY_DOWN,
  GKEY_LEFT,
  GKEY_RIGHT,
  GKEY_START,
  GKEY_SELECT,
  GKEY_A,
  GKEY_B,
  GKEY_L,
  GKEY_R,
  GKEY_LOADSTATE,
  GKEY_SAVESTATE,
  GKEY_FASTFORWARD,
  GKEY_FPS
};

// External key order
u32 gamepad_config_map[PLAT_BUTTON_COUNT] =
{
  BUTTON_ID_UP,
  BUTTON_ID_DOWN,
  BUTTON_ID_LEFT,
  BUTTON_ID_RIGHT,
  BUTTON_ID_START,
  BUTTON_ID_SELECT,
  BUTTON_ID_A,
  BUTTON_ID_B,
  BUTTON_ID_L,
  BUTTON_ID_R,
  BUTTON_ID_LOADSTATE,
  BUTTON_ID_SAVESTATE,
  BUTTON_ID_FASTFORWARD,
  BUTTON_ID_FPS
};

// Touch region identifiers
enum region_id {
  REG_LEFT_UP,
  REG_UP,
  REG_RIGHT_UP,
  REG_LEFT,
  REG_RIGHT,
  REG_LEFT_DOWN,
  REG_DOWN,
  REG_RIGHT_DOWN,
  REG_SELECT,
  REG_START,
  REG_L,
  REG_R,
  REG_A,
  REG_B,
  REG_CENTER,
  REGION_COUNT
};

// Region mapping
static GdkRegion **regions;
static GdkRegion *regions_hor[REGION_COUNT];
static GdkRegion *regions_ver[REGION_COUNT];

static int
get_keys_for_coordinates(gdouble x, gdouble y)
{
  // Directional top row
  if (gdk_region_point_in(regions[REG_LEFT_UP], x, y))
    return GKEY_LEFT | GKEY_UP;
  else if (gdk_region_point_in(regions[REG_UP], x, y))
    return GKEY_UP;
  else if (gdk_region_point_in(regions[REG_RIGHT_UP], x, y))
    return GKEY_RIGHT | GKEY_UP;

  // Directional middle row
  else if (gdk_region_point_in(regions[REG_LEFT], x, y))
    return GKEY_LEFT;
  else if (gdk_region_point_in(regions[REG_RIGHT], x, y))
    return GKEY_RIGHT;

  // Directional bottom row
  else if (gdk_region_point_in(regions[REG_LEFT_DOWN], x, y))
    return GKEY_LEFT | GKEY_DOWN;
  else if (gdk_region_point_in(regions[REG_DOWN], x, y))
    return GKEY_DOWN;
  else if (gdk_region_point_in(regions[REG_RIGHT_DOWN], x, y))
    return GKEY_RIGHT | GKEY_DOWN;

  // Select, Start
  else if (gdk_region_point_in(regions[REG_SELECT], x, y))
    return GKEY_SELECT;
  else if (gdk_region_point_in(regions[REG_START], x, y))
    return GKEY_START;

  // L, R
  else if (gdk_region_point_in(regions[REG_L], x, y))
    return GKEY_L;
  else if (gdk_region_point_in(regions[REG_R], x, y))
    return GKEY_R;

  // A, B
  else if (gdk_region_point_in(regions[REG_A], x, y))
    return GKEY_A;
  else if (gdk_region_point_in(regions[REG_B], x, y))
    return GKEY_B;

  // No match
  else
    return 0;
}

static int
delayed_focus_handler(gpointer user_data)
{
  // At this point it should be certain enough if the game should be paused
  if (!gtk_window_is_active(GTK_WINDOW(window)))
    activate_pause(TRUE);

  // Stop the timer
  pause_timeout = 0;
  return FALSE;
}

static void
focus_handler(GtkWindow *window)
{
  if (gtk_window_is_active(window)) {
    // Focus received, resume the game
    gtk_dialog_response(pause_dialog, 0);
  } else {
    // Focus lost, but there could be different reasons to it:
    // + screen is locked (pause the game)
    // + screen is blanked (pause the game)
    // + a notification popped up (do nothing)
    // + tasknav activated (pause the game, but leave it to topmost_handler)

    // Try to determine the screen state
    FILE* lockfile = fopen("/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_ts", "r");
    int lockstate = 0;
    if (lockfile) {
      fscanf(lockfile, "%d", &lockstate);
      fclose(lockfile);
    }

    if (lockstate) {
      // Screen is locked, pause the game
      activate_pause(TRUE);
    } else {
      // It is not certain what should be done, so wait a moment (3000 ms, which
      // is for how long the screen is dimmed before blanking) and then decide
      g_source_remove(pause_timeout);
      pause_timeout = gtk_timeout_add(3000, delayed_focus_handler, NULL);
    }
  }
}

static void
topmost_handler(GtkWindow *window)
{
  if (!hildon_window_get_is_topmost(HILDON_WINDOW(window)))
    activate_pause(TRUE);
}

static void
pause_keybd_handler(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
  // Prevent immediate resuming due to the initial key press-release sequence
  if (pause_key_down)
    pause_key_down = FALSE;
  // Ignore modifiers so that tasknav can be activated without resuming the game
  else if (!event->state)
    gtk_dialog_response(pause_dialog, 0);
}

static void activate_pause(int automatic)
{
  if (pause_dialog) return;

  if (automatic) {
    pause_dialog = GTK_DIALOG (hildon_note_new_information (GTK_WINDOW (window), "Paused"));
    g_signal_connect (G_OBJECT (pause_dialog), "notify::is-active", G_CALLBACK (focus_handler), NULL);
    gtk_dialog_run (GTK_DIALOG (pause_dialog));
  } else {
    pause_dialog = GTK_DIALOG (gtk_dialog_new_with_buttons ("Pause", GTK_WINDOW (window), 0,
                                                            "Quit", GTK_RESPONSE_ACCEPT,
                                                            "Resume", GTK_RESPONSE_REJECT,
                                                            NULL));

    // Create mute button
    GtkWidget *mute_button = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label (GTK_BUTTON (mute_button), "Mute");
    hildon_check_button_set_active (HILDON_CHECK_BUTTON (mute_button), !global_enable_audio);

    // Add mute button
    GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (pause_dialog));
    gtk_container_add (GTK_CONTAINER (content_area), mute_button);
    gtk_widget_show (mute_button);

    g_signal_connect (G_OBJECT (pause_dialog), "key-release-event", G_CALLBACK (pause_keybd_handler), NULL);

    if (gtk_dialog_run (GTK_DIALOG (pause_dialog)) == GTK_RESPONSE_ACCEPT)
      quit();

    global_enable_audio = !hildon_check_button_get_active (HILDON_CHECK_BUTTON (mute_button));
  }

  gtk_widget_destroy (GTK_WIDGET(pause_dialog));
  pause_dialog = NULL;
}

static void
keybd_handler(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
  int *target_keystate = &keybd_keystate;
  u32 key = keymap[event->hardware_keycode];

  if (key == 0xff) return;

  key = 1 << key;

  switch (key) {
    case GKEY_LEFT_UP:
    case GKEY_RIGHT_UP:
    case GKEY_LEFT_DOWN:
    case GKEY_RIGHT_DOWN:
      target_keystate = &combo_keystate;
      break;
    case GKEY_QUIT:
      quit();
      return;
    case GKEY_PAUSE:
      pause_key_down = TRUE;
      activate_pause(FALSE);
      return;
  }

  if (event->type == GDK_KEY_PRESS)
    *target_keystate |= key;
  else if (event->type == GDK_KEY_RELEASE)
    *target_keystate &= ~key;
}

static void
touch_handler(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
  if (event->type == GDK_BUTTON_PRESS) {
    touch_keystate = get_keys_for_coordinates(event->x, event->y);
    touch_idle = FALSE;
  } else if (event->type == GDK_BUTTON_RELEASE) {
    touch_keystate = 0;
    touch_idle = TRUE;

    if (pause_gesture_start_y < screen_h && event->y < screen_h
    &&  event->x - pause_gesture_start_x > screen_w - 50)
      activate_pause(FALSE);
  }

  pause_gesture_start_x = event->x;
  pause_gesture_start_y = event->y;
}

static void
touch_motion_handler(GtkWidget *widget, GdkEventMotion *event, gpointer user_data)
{
  if (touch_idle) return;

  if (!touch_keystate)
    touch_keystate = get_keys_for_coordinates(event->x, event->y);
  else if (gdk_region_point_in(regions[REG_CENTER], event->x, event->y))
    touch_keystate = 0;
  else if (touch_keystate & (GKEY_LEFT | GKEY_RIGHT | GKEY_UP | GKEY_DOWN)) {
    int check_keystate = get_keys_for_coordinates(event->x, event->y);
    if (check_keystate & (GKEY_LEFT | GKEY_RIGHT | GKEY_UP | GKEY_DOWN))
      touch_keystate = check_keystate;
  }
}

static void
configure_handler(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data)
{
  screen_w = event->width;

  if (event->width > event->height) {
    // Landscape
    screen_h = event->height;
    regions = regions_hor;
    hildon_animation_actor_set_show (keypad_actor, FALSE);
  } else {
    // Portrait
    screen_h = event->height - 480; // make some room for the virtual keyapad
    regions = regions_ver;
    hildon_animation_actor_set_show (keypad_actor, TRUE);
  }

  fb_set_mode(0, 0, 0, screen_scale, 0, screen_filter2);
}

static void
delete_handler()
{
  quit();
}

void print_help()
{
  printf("gpSP - Game Boy Advance emulator"                                         "\n"
         "Usage: gpspm FILE [options]"                                              "\n"
         "       gpspm [options]"                                                   "\n"
         "  --rom FILE          - the ROM file to open"                             "\n"
         "  --keys FILE         - read key bindings from nonstandard location"      "\n"
         "  --accelerometer     - enable accelerometer input"                       "\n"
         "  --autopause         - pause emulation when losing focus"                "\n"
         "  --scale N           - none (0), double (1), triple (2), fullscreen (3)" "\n"
         "  --filter N          - none (0), scale2x (1), scale3x (2), eagle2x (3)"  "\n"
         "  --frameskip-type N  - disabled (0), auto (1), manual (2)"               "\n"
         "  --frameskip-value N - how many frames should be skipped"                "\n"
         "  --random-frameskip  - some randomness in choosing which frames to skip" "\n"
         "  --no-sound          - disable sound"                                    "\n"
         "  --sound-buffer N    - size of the audio buffer, 2^(4+N)"                "\n"
         "  --breakpoint N      - set the breakpoint to N"                          "\n"
         "  --help              - display this message"                             "\n");
  exit(1);
}

void read_keys(FILE *keys_file)
{
  // Bind keys like described in the provided configuration file
  int i, key_code;
  for (i = 0; i < 20; i++) {
    fscanf(keys_file, "%i", &key_code);
    keymap[key_code] = i;

    // Check whether zoom keys should be captured
    if (key_code == 73 || key_code == 74)
        use_zoom_keys = TRUE;
  }
}

#define setaopt(optname) if (++i < *argc) optname = argv[i]; else print_help()
#define setiopt(optname) if (++i < *argc) optname = atoi(argv[i]); else print_help()

int gpsp_plat_args(int *argc, char *argv[])
{
  if (*argc < 2)
    print_help();

  // Some default settings
  screen_scale             = 2; // Triple scale
  current_frameskip_type   = 1; // Auto frameskip
  frameskip_value          = 1; // Skip up to 1 frame
  random_skip              = 0; // Uniform skipping
  audio_buffer_size_number = 8; // Sound buffer of 4096

  char *rom_path = NULL;
  char *breakpoint = NULL;

  int i;
  for (i = 1; i < *argc; i++) {
         if (!strcmp(argv[i], "--rom"))              setaopt(rom_path);
    else if (!strcmp(argv[i], "--keys"))             setaopt(keys_path);
    else if (!strcmp(argv[i], "--accelerometer"))    accel_enable = TRUE;
    else if (!strcmp(argv[i], "--autopause"))        autopause = TRUE;
    else if (!strcmp(argv[i], "--scale"))            setiopt(screen_scale);
    else if (!strcmp(argv[i], "--filter"))           setiopt(screen_filter2);
    else if (!strcmp(argv[i], "--frameskip-type"))   setiopt(current_frameskip_type);
    else if (!strcmp(argv[i], "--frameskip-value"))  setiopt(frameskip_value);
    else if (!strcmp(argv[i], "--random-frameskip")) random_skip = TRUE;
    else if (!strcmp(argv[i], "--no-sound"))         global_enable_audio = FALSE;
    else if (!strcmp(argv[i], "--sound-buffer"))     setiopt(audio_buffer_size_number);
    else if (!strcmp(argv[i], "--help"))             print_help();
    else if (!strcmp(argv[i], "--breakpoint"))       setaopt(breakpoint);
    else if (rom_path) print_help(); else rom_path = argv[i];
  }

  // Initialize GTK+
  gtk_init (argc, &argv);

  // Prepare command line for the emulator
  if (rom_path) {
    argv[1] = rom_path;
    *argc = 2;
    if (breakpoint) {
      argv[2] = breakpoint;
      *argc = 3;
    }
  }

  return 0;
}

void gpsp_plat_init(void)
{
  // Horizontal layout definition
  GdkRectangle rect_hor_left_up    = {20,  60,  120, 120};
  GdkRectangle rect_hor_up         = {140, 60,  120, 120};
  GdkRectangle rect_hor_right_up   = {260, 60,  120, 120};
  GdkRectangle rect_hor_left       = {20,  180, 120, 120};
  GdkRectangle rect_hor_center     = {140, 180, 120, 120};
  GdkRectangle rect_hor_right      = {260, 180, 120, 120};
  GdkRectangle rect_hor_left_down  = {20,  300, 120, 120};
  GdkRectangle rect_hor_down       = {140, 300, 120, 120};
  GdkRectangle rect_hor_right_down = {260, 300, 120, 120};
  GdkRectangle rect_hor_select     = {0,   420, 400, 60};
  GdkRectangle rect_hor_start      = {400, 420, 400, 60};
  GdkRectangle rect_hor_l          = {0,   0,   400, 60};
  GdkRectangle rect_hor_r          = {400, 0,   400, 60};
  GdkRectangle rect_hor_a          = {600, 60,  200, 360};
  GdkRectangle rect_hor_b          = {400, 60,  200, 360};

  // Vertical layout definition
  GdkRectangle rect_ver_left_up    = {0,   380, 120, 120};
  GdkRectangle rect_ver_up         = {120, 380, 120, 120};
  GdkRectangle rect_ver_right_up   = {240, 380, 120, 120};
  GdkRectangle rect_ver_left       = {0,   500, 120, 120};
  GdkRectangle rect_ver_center     = {120, 500, 120, 120};
  GdkRectangle rect_ver_right      = {240, 500, 120, 120};
  GdkRectangle rect_ver_left_down  = {0,   620, 120, 120};
  GdkRectangle rect_ver_down       = {120, 620, 120, 120};
  GdkRectangle rect_ver_right_down = {240, 620, 120, 120};
  GdkRectangle rect_ver_select     = {0,   740, 240, 60};
  GdkRectangle rect_ver_start      = {240, 740, 240, 60};
  GdkRectangle rect_ver_l          = {0,   320, 240, 60};
  GdkRectangle rect_ver_r          = {240, 320, 240, 60};
  GdkRectangle rect_ver_a          = {360, 380, 120, 180};
  GdkRectangle rect_ver_b          = {360, 560, 120, 180};

  // Create the horizontal layout
  regions_hor[REG_LEFT_UP]    = gdk_region_rectangle(&rect_hor_left_up);
  regions_hor[REG_UP]         = gdk_region_rectangle(&rect_hor_up);
  regions_hor[REG_RIGHT_UP]   = gdk_region_rectangle(&rect_hor_right_up);
  regions_hor[REG_LEFT]       = gdk_region_rectangle(&rect_hor_left);
  regions_hor[REG_CENTER]     = gdk_region_rectangle(&rect_hor_center);
  regions_hor[REG_RIGHT]      = gdk_region_rectangle(&rect_hor_right);
  regions_hor[REG_LEFT_DOWN]  = gdk_region_rectangle(&rect_hor_left_down);
  regions_hor[REG_DOWN]       = gdk_region_rectangle(&rect_hor_down);
  regions_hor[REG_RIGHT_DOWN] = gdk_region_rectangle(&rect_hor_right_down);
  regions_hor[REG_SELECT]     = gdk_region_rectangle(&rect_hor_select);
  regions_hor[REG_START]      = gdk_region_rectangle(&rect_hor_start);
  regions_hor[REG_L]          = gdk_region_rectangle(&rect_hor_l);
  regions_hor[REG_R]          = gdk_region_rectangle(&rect_hor_r);
  regions_hor[REG_A]          = gdk_region_rectangle(&rect_hor_a);
  regions_hor[REG_B]          = gdk_region_rectangle(&rect_hor_b);

  // Create the vertical layout
  regions_ver[REG_LEFT_UP]    = gdk_region_rectangle(&rect_ver_left_up);
  regions_ver[REG_UP]         = gdk_region_rectangle(&rect_ver_up);
  regions_ver[REG_RIGHT_UP]   = gdk_region_rectangle(&rect_ver_right_up);
  regions_ver[REG_LEFT]       = gdk_region_rectangle(&rect_ver_left);
  regions_ver[REG_CENTER]     = gdk_region_rectangle(&rect_ver_center);
  regions_ver[REG_RIGHT]      = gdk_region_rectangle(&rect_ver_right);
  regions_ver[REG_LEFT_DOWN]  = gdk_region_rectangle(&rect_ver_left_down);
  regions_ver[REG_DOWN]       = gdk_region_rectangle(&rect_ver_down);
  regions_ver[REG_RIGHT_DOWN] = gdk_region_rectangle(&rect_ver_right_down);
  regions_ver[REG_SELECT]     = gdk_region_rectangle(&rect_ver_select);
  regions_ver[REG_START]      = gdk_region_rectangle(&rect_ver_start);
  regions_ver[REG_L]          = gdk_region_rectangle(&rect_ver_l);
  regions_ver[REG_R]          = gdk_region_rectangle(&rect_ver_r);
  regions_ver[REG_A]          = gdk_region_rectangle(&rect_ver_a);
  regions_ver[REG_B]          = gdk_region_rectangle(&rect_ver_b);

  // Build the configuration directory path and export as the main path
  char *config_home = getenv("XDG_CONFIG_HOME");
  if (config_home == NULL || config_home[0] == 0)
    sprintf(main_path, "%s" PATH_SEPARATOR ".config" PATH_SEPARATOR "gpspm", getenv("HOME"));
  else
    sprintf(main_path, "%s" PATH_SEPARATOR "gpspm", config_home);

  // Assign the default (do-nothing) value to all keys
  int i;
  for (i = 0; i < 65536; i++)
      keymap[i] = 0xff;

  char filename[512];
  FILE *keys_file = NULL;

  if (keys_path) {
    // Open specified configuration file
    keys_file = fopen(keys_path, "r");

    if (keys_file) {
      // Read specified file
      read_keys(keys_file);
    }
  } else {
    // Open user's keys configuration file
    sprintf(filename, "%s" PATH_SEPARATOR "%s", main_path, "keys");
    keys_file = fopen(filename, "r");

    if (keys_file) {
      // Read user's file
      read_keys(keys_file);
    } else {
      // Don't give up, open global keys configuration file
      sprintf(filename, "%s" PATH_SEPARATOR "%s", INSTALL_PATH, "keys");
      keys_file = fopen(filename, "r");

      if (keys_file) {
        // Read global file
        read_keys(keys_file);
      }
    }
  }

  if (keys_file)
    fclose(keys_file);
  else
    fprintf(stderr, "ERROR: could not open keys configuration file\n");

  // Create the main window, declare support for portrait mode and zoom keys
  window = hildon_window_new ();
  hildon_gtk_window_set_portrait_flags (GTK_WINDOW(window), HILDON_PORTRAIT_MODE_SUPPORT);
  hildon_gtk_window_enable_zoom_keys (GTK_WINDOW(window), use_zoom_keys);

  // Make the window black
  GdkColor window_bg = {0, 0, 0, 0};
  gtk_widget_modify_bg(window, GTK_STATE_NORMAL, &window_bg);

  // Prepare the window for display
  gtk_widget_realize (window);
  gtk_window_fullscreen (GTK_WINDOW(window));

  // Connect signals
  g_signal_connect (G_OBJECT (window), "key-press-event", G_CALLBACK (keybd_handler), NULL);
  g_signal_connect (G_OBJECT (window), "key-release-event", G_CALLBACK (keybd_handler), NULL);
  g_signal_connect (G_OBJECT (window), "button-press-event", G_CALLBACK (touch_handler), NULL);
  g_signal_connect (G_OBJECT (window), "button-release-event", G_CALLBACK (touch_handler), NULL);
  g_signal_connect (G_OBJECT (window), "motion-notify-event", G_CALLBACK (touch_motion_handler), NULL);
  g_signal_connect (G_OBJECT (window), "configure-event", G_CALLBACK (configure_handler), NULL);
  g_signal_connect (G_OBJECT (window), "delete-event", G_CALLBACK (delete_handler), NULL);
  if (autopause) {
    g_signal_connect (G_OBJECT (window), "notify::is-active", G_CALLBACK (focus_handler), NULL);
    g_signal_connect (G_OBJECT (window), "notify::is-topmost", G_CALLBACK (topmost_handler), NULL);
  }

  // Make sure that the required signals can reach us
  gtk_widget_add_events (window,GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK);

  // Create the video output actor
  vout_actor = HILDON_ANIMATION_ACTOR (hildon_animation_actor_new());
  hildon_animation_actor_set_parent (vout_actor, GTK_WINDOW (window));

  // Create the video output widget and attach it to the actor
  vout_widget = gtk_image_new ();
  gtk_container_add (GTK_CONTAINER (vout_actor), vout_widget);

  // Create the keypad actor
  keypad_actor = HILDON_ANIMATION_ACTOR (hildon_animation_actor_new());
  hildon_animation_actor_set_parent (keypad_actor, GTK_WINDOW (window));

  // Crate the keypad widget and attach it to the actor
  keypad_widget = gtk_image_new ();
  sprintf(filename, "%s" PATH_SEPARATOR "%s", INSTALL_PATH, "keypad.png");
  gtk_image_set_from_file(GTK_IMAGE (keypad_widget), filename);
  gtk_container_add (GTK_CONTAINER (keypad_actor), keypad_widget);

  // Position the keypad
  gtk_window_resize (GTK_WINDOW (keypad_actor), 480, 481); // 480 is not enough
  hildon_animation_actor_set_position (keypad_actor, 0, 320);
  hildon_animation_actor_set_scale (keypad_actor, 1.0, 1.0);

  // Show everything
  gtk_widget_show_all (GTK_WIDGET (vout_actor));
  gtk_widget_show_all (GTK_WIDGET (keypad_actor));
  gtk_widget_show_all (GTK_WIDGET (window));
}

void gpsp_plat_quit(void)
{
  gtk_main_quit();
}

inline u32 gpsp_plat_joystick_read(void)
{
  int keystate = keybd_keystate | accel_keystate | touch_keystate;

  // Expand keys that translate to multiple basic keys
  if (combo_keystate & GKEY_LEFT_UP)
    keystate |= GKEY_LEFT | GKEY_UP;
  if (combo_keystate & GKEY_RIGHT_UP)
    keystate |= GKEY_RIGHT | GKEY_UP;
  if (combo_keystate & GKEY_LEFT_DOWN)
    keystate |= GKEY_LEFT | GKEY_DOWN;
  if (combo_keystate & GKEY_RIGHT_DOWN)
    keystate |= GKEY_RIGHT | GKEY_DOWN;

  return keystate;
}

inline u32 gpsp_plat_buttons_to_cursor(u32 buttons)
{
  // Refresh input state
  fb_flip_screen();

  // This is just to press "any key" for some error messages
  return buttons ?  CURSOR_EXIT : CURSOR_NONE;
}

void *fb_flip_screen(void)
{
  if (accel_enable) {
    accel_keystate = 0;

    // Read the accelerometer
    int x, y, z;
    FILE* accel = fopen("/sys/class/i2c-adapter/i2c-3/3-001d/coord", "r");
    if (accel) {
      fscanf(accel, "%d %d %d", &x, &y, &z);
      fclose(accel);
    } else {
      fprintf (stderr, "CRITICAL: cannot access the accelerometer\n");
      exit(1);
    }

    // Apply calibration
    x += accel_center_x;
    y += accel_center_y;

    // X axis
    if (x > accel_deadzone)
      accel_keystate = GKEY_LEFT;
    else if (x < -accel_deadzone)
      accel_keystate = GKEY_RIGHT;

    // Y axis
    if (y > accel_deadzone)
      accel_keystate |= GKEY_UP;
    else if (y < -accel_deadzone)
      accel_keystate |= GKEY_DOWN;
  }

  switch (sw_filter) {
    case SWFILTER_SCALE2X:
      neon_scale2x_16_16(bounce_buf, vout_image->mem, vmode_w, vmode_w*2, vmode_w*2*2, vmode_h);
      break;
    case SWFILTER_SCALE3X:
      neon_scale3x_16_16(bounce_buf, vout_image->mem, vmode_w, vmode_w*2, vmode_w*3*2, vmode_h);
      break;
    case SWFILTER_EAGLE2X:
      neon_eagle2x_16_16(bounce_buf, vout_image->mem, vmode_w, vmode_w*2, vmode_w*2*2, vmode_h);
      break;
    case SWFILTER_NONE:
    default:
      break;
  }

  // Redraw
  gtk_widget_queue_draw (vout_widget);

  // Process GTK+ events
  while (gtk_events_pending())
    gtk_main_iteration();

  return (void *) (sw_filter ? bounce_buf : vout_image->mem);
}

void fb_wait_vsync(void)
{
}

void fb_set_mode(int w, int h, int buffers, int scale, int filter, int filter2)
{
  // Use old values if new are not present
  if (w) vmode_w = w;
  if (h) vmode_h = h;

  // Calculate the actual size on the screen
  int target_w;
  int target_h;
  switch (scale) {
    case 0: // Pixel to pixel
      target_w = vmode_w;
      target_h = vmode_h;
      break;
    case 1: // Double scale
      target_w = vmode_w * 2;
      target_h = vmode_h * 2;
      break;
    case 2: // Triple scale
      target_w = vmode_w * 3;
      target_h = vmode_h * 3;
      break;
    case 3: // Fullscreen
    case 15:
      target_w = screen_w;
      target_h = screen_h;
      break;
    default:
      fprintf(stderr, "CRITICAL: unknown scale: %d\n", scale);
      exit(1);
  }

  // Make sure that the image is not too wide
  if (target_w > screen_w) {
    target_h = target_h * screen_w / target_w;
    target_w = screen_w;
  }

  // Make sure that the image is not too high
  if (target_h > screen_h) {
    target_w = target_w * screen_h / target_h;
    target_h = screen_h;
  }

  // Decide which filter to use
  if (filter2 == SWFILTER_SCALE3X && vmode_h*3 > screen_h)
    filter2 = SWFILTER_SCALE2X;

  if ((filter2 == SWFILTER_SCALE2X || filter2 == SWFILTER_EAGLE2X) && vmode_h*2 > screen_h)
    filter2 = SWFILTER_NONE;

  sw_filter = filter2;

  // Resolution multiplier
  int k = filter2 == SWFILTER_SCALE2X ? 2 :
          filter2 == SWFILTER_SCALE3X ? 3 :
          filter2 == SWFILTER_EAGLE2X ? 2 : 1;

  // Calculate the output position on the screen
  int target_x = (screen_w - target_w) / 2;
  int target_y = (screen_h - target_h) / 2;

  // Free the previous output image
  if (vout_image)
    gdk_image_destroy(vout_image);

  // Create a new output image
  vout_image = gdk_image_new (GDK_IMAGE_FASTEST, gdk_visual_get_system(), vmode_w*k, vmode_h*k);

  // Connect the output image to the output widget
  gtk_image_set_from_image (GTK_IMAGE(vout_widget), vout_image, NULL);

  // Apply scaling and positioning
  gtk_window_resize (GTK_WINDOW (vout_actor), vmode_w*k, vmode_h*k);
  hildon_animation_actor_set_position (vout_actor, target_x, target_y);
  hildon_animation_actor_set_scale (vout_actor, (gdouble)target_w/vmode_w/k, (gdouble)target_h/vmode_h/k);
}
