/*
 *  GPSData for Maemo.
 *  Copyright (C) 2011 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <QtGui>

#include "satellitesignalstrength.h"

void SatelliteSignalStrength::updateWidget(const QList<QGeoSatelliteInfo> &satellites, bool inUseList)
{
    if (inUseList)
        m_satellitesInUse = satellites;
    else
        m_satellitesInView = satellites;

    update();
}

void SatelliteSignalStrength::showTitle(bool show)
{
    m_showTitle = show;
}

void SatelliteSignalStrength::paintEvent(QPaintEvent * /* event */)
{
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);

    int title_height = 0;
    if (m_showTitle) {
        int title_x = rect().x() + m_margin;
        int title_y = rect().y() + m_margin;
        int title_width = rect().width() - (2 * m_margin);
        title_height = painter.fontMetrics().height();
        QRectF titleArea(title_x, title_y, title_width, title_height);

        /* Draw title */
        paintTitle(painter, titleArea);
     }

    /* Signal strength graph */
    int graph_x = rect().x();
    int graph_y = rect().y() + title_height + m_spacing;
    int graph_width = rect().width();
    int graph_height = rect().height() - title_height - m_spacing;
    QRectF graphArea(graph_x, graph_y, graph_width, graph_height);

    /* Signal strength bar graphs */
    for (int index = 1; index <= m_numOfSatellites; ++index) {
        paintBarGraph(painter, graphArea, index);
    }
}

void SatelliteSignalStrength::paintTitle(QPainter &painter, const QRectF &area)
{
    painter.setPen(QApplication::palette().color(QPalette::Text));
    painter.drawText(area, Qt::AlignLeft, tr("Signal strength:"));
}

void SatelliteSignalStrength::paintBarGraph(QPainter &painter, const QRectF &area, int index)
{
    /* Bar graph backround */
    double barGraphBackground_x;
    double barGraphBackground_y;
    double barGraphBackground_width = (double) (area.width() - (2.0 * m_margin) -
                                      (m_spacing * (m_numOfSatellitesInRow - 1))) / (double) m_numOfSatellitesInRow;
    double barGraphBackground_height = (double) (area.height() - (2.0 * m_margin) -
                                       (2.0 * m_spacing)) / (double) m_numOfRows;

    if (index <= m_numOfSatellitesInRow) {
        barGraphBackground_x = area.x() + m_margin +
                               ((index - 1) * (barGraphBackground_width + m_spacing));
        barGraphBackground_y = area.y() + m_margin;
    } else {
        barGraphBackground_x = area.x() + m_margin +
                               (((index - 1) - m_numOfSatellitesInRow) * (barGraphBackground_width + m_spacing));
        barGraphBackground_y = area.y() + m_margin + (2.0 * m_spacing) + barGraphBackground_height;
    }
    QRectF barGraphBackgroundArea(barGraphBackground_x, barGraphBackground_y,
                                  barGraphBackground_width, barGraphBackground_height);

    painter.setPen(m_barGraphBorderColor);
    painter.setBrush(m_barGraphBackgroundColor);
    painter.drawRect(barGraphBackgroundArea);

    bool inUse = false;
    int signalStrength = 0;

    for (int iter = 0; iter < m_satellitesInView.count(); iter++) {
        if (m_satellitesInView.at(iter).prnNumber() == index) {
            signalStrength = m_satellitesInView.at(iter).signalStrength();
            break;
        }
    }
    for (int iter = 0; iter < m_satellitesInUse.count(); iter++) {
        if (m_satellitesInUse.at(iter).prnNumber() == index) {
            signalStrength = m_satellitesInUse.at(iter).signalStrength();
            inUse = true;
            break;
        }
    }

    if (signalStrength > 0) {
        if (inUse) {
            painter.setPen(m_barGraphInUseColor);
            painter.setBrush(m_barGraphInUseColor);
        } else {
            painter.setPen(m_barGraphInViewColor);
            painter.setBrush(m_barGraphInViewColor);
        }

        double level_width = barGraphBackground_width - 2.0;
        double level_height = (double) signalStrength * (barGraphBackground_height - 2.0) / 100.0;
        double level_x = barGraphBackground_x + 1.0;
        double level_y = barGraphBackground_y + barGraphBackground_height - level_height - 1.0;

        QRectF barGraphLevelArea(level_x, level_y, level_width, level_height);
        painter.drawRect(barGraphLevelArea);
    }

    /* Satellite's PRN must be painter over the level bar */
    QFont font = QApplication::font();
    font.setPixelSize(12);
    painter.setFont(font);
    painter.setPen(QApplication::palette().color(QPalette::Text));
    painter.drawText(barGraphBackgroundArea, Qt::AlignCenter, QString::number(index));
}

SatelliteSignalStrength::SatelliteSignalStrength(QWidget *parent) : QWidget(parent)
{
    m_barGraphBorderColor = QColor(0, 0, 0);
    m_barGraphBackgroundColor = QColor(74, 69, 66);
    m_barGraphInViewColor = QColor(153, 153, 153);
    m_barGraphInUseColor = QColor(51, 191, 51);
}
