/*
-------------------------------------------------------------------
SUDOKU SENSEI 1.03: a Sudoku Explainer Engine
Copyright (C) 2005  Martin Knoblauch

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA  02110-1301, USA.

Contact the author: comocomocomo AT users.sourceforge.net
Latest versions: http://sourceforge.net/projects/sudoku-sensei
-------------------------------------------------------------------

    sudoku-grids.c
*/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "sudoku-pr.h"

/*///////////////////////////////////////////////////////////////// */

t_board * ConstructBasicBoard (int order, char diag)
{
  int n, row, col, cell, block, group;
  t_board * b;
  t_group * g;

  if (order<2 || order>5)
    return NULL;

  n = order*order;          /* # of symbols (tipically 3*3) */

  b = AllocateBoardSkeleton (n*n,            /* Cells */
                             diag?3*n+2:3*n, /* Groups */
                             n,              /* Symbols */
                             n,              /* Width */
                             n,              /* Height */
                             order,    /* Block width */
                             order);   /* Block height */

  if (!b)
    return NULL;

  for (group=0; group<n; group++)         /* Groups {0..n-1} */
    b->pGr[group].type = GR_BLOCK;        /* are blocks */

  for (group=n; group<2*n; group++)       /* Groups {n..2n-1} */
    b->pGr[group].type = GR_ROW;          /* are rows */

  for (group=2*n; group<3*n; group++)     /* Groups {2n..3n-1} */
    b->pGr[group].type = GR_COL;          /* are columns */

  if (diag)                               /* Groups 3n and 3n+1 */
    b->pGr[3*n].type =                    /* are diagonals */
    b->pGr[3*n+1].type = GR_DIAG;

  for (row=cell=0; row<n; row++)          /* For every cell... */
    for (col=0; col<n; col++, cell++)     /* (track col and row) */
    {
      b->pCl[cell].y = row;               /* Coordinates */
      b->pCl[cell].x = col;

      b->pppCl[row][col] = b->pCl + cell; /* Pointer in map */

      block = row - row%order +
              col/order;                  /* Number of block */

      g = b->pGr + block;                 /* Add cell to */
      g->ppCl[g->nCl++] = b->pCl + cell;  /* block group */

      g = b->pGr + n + row;               /* Add cell to */
      g->ppCl[g->nCl++] = b->pCl + cell;  /* row group */

      g = b->pGr + 2*n + col;             /* Add cell to */
      g->ppCl[g->nCl++] = b->pCl + cell;  /* column goup */

      if (diag && row==col)               /* Add cell to */
      {                                   /* "backslash" */
        g = b->pGr + 3*n;                 /* diag. group */
        g->ppCl[g->nCl++] = b->pCl + cell;
      }

      if (diag && row==n-1-col)           /* Add cell to */
      {                                   /* "slash" */
        g = b->pGr + 3*n+1;               /* diag. group */
        g->ppCl[g->nCl++] = b->pCl + cell;
      }
    }

  if (!CompleteBoard(b))
  {
    DestroyBoard (b);
    return NULL;
  }

  return b;
}

/*///////////////////////////////////////////////////////////////// */

t_board * ConstructCustomBoard (
                             char wb,     /* Block width */
                             char hb,     /* Block height */
                             char diag,   /* TRUE == diagonals */
                             int nGrids,  /* Num. composed grids */
                             int * xg,    /* Coordinates of */
                             int * yg)    /* grids (in blocks) */
{
  int n, nCells, nGroups, zero, nGroupsPerGrid;
  int xmin, xmax, ymin, ymax, width, height;
  int i, j, k, x, y;
  int group, cell, row, col, block;
  t_board * b;
  t_group * g;

  n = wb * hb;         /* # of symbols */

  if (wb<1 || hb<1 || n<4 || n>30)
    return NULL;

  if (nGrids<1 || !xg || !yg)  /* No grids information... */
  {
    zero = 0;                  /* ...suppose just one grid */
    xg = yg = &zero;
    nGrids = 1;
  }                                        /* Every grid has: */
                                           /*   n rows */
  nGroupsPerGrid = diag ? 3*n+2 : 3*n;     /*   n columns */
  nGroups = nGrids * nGroupsPerGrid;       /*   n blocks */
                                           /*   [2 diagonals] */
  nCells = nGrids * n * n;                 /*   n*n cells */

  xmin = xmax = xg[0];
  ymin = ymax = yg[0];

  for (i=1; i<nGrids; i++)
  {
    if (xg[i]<xmin)
      xmin = xg[i];
    else if (xg[i]>xmax)
      xmax = xg[i];          /* Calculate board */
                             /* size */
    if (yg[i]<ymin)
      ymin = yg[i];
    else if (yg[i]>ymax)
      ymax = yg[i];

    for (x=xg[i]; x<xg[i]+hb; x++)
      for (y=yg[i]; y<yg[i]+wb; y++)
        for (j=0; j<i; j++)
          if (x >= xg[j]    &&    /* Discount cells of */
              x <  xg[j]+hb &&    /* overlapped blocks */
              y >= yg[j]    &&
              y <  yg[j]+wb   )
          {
            nCells -= hb * wb;
            break;
          }
  }

  width = (xmax-xmin+hb) * wb;     /* Board size */
  height = (ymax-ymin+wb) * hb;    /* in cells */

  b = AllocateBoardSkeleton (nCells,   /* Cells */
                             nGroups,  /* Groups */
                             n,        /* Symbols */
                             width,    /* Width */
                             height,   /* Height */
                             wb,       /* Block width */
                             hb);      /* Block height */

  if (!b)
    return NULL;

  for (i=group=0; i<nGrids; i++)     /* For every grid */
  {
    for (j=0; j<n; j++, group++)          /* Groups {0..n-1} */
      b->pGr[group].type = GR_BLOCK;      /* are blocks */

    for (j=0; j<n; j++, group++)          /* Groups {n..2n-1} */
      b->pGr[group].type = GR_ROW;        /* are rows */

    for (j=0; j<n; j++, group++)          /* Groups {2n..3n-1} */
      b->pGr[group].type = GR_COL;        /* are columns */

    if (diag)
    {                                     /* Groups 3n and 3n+1 */
      b->pGr[group++].type = GR_DIAG;     /* are diagonals */
      b->pGr[group++].type = GR_DIAG;
    }
  }

  for (y=cell=0; y<height; y++)   /* For every position */
    for (x=0; x<width; x++)       /* of the board */
    {
      for (i=j=0; i<nGrids; i++)     /* For every grid */
      {
        row = y - (yg[i]-ymin)*hb;   /* Pos. within the grid */
        col = x - (xg[i]-xmin)*wb;

        if (row>=0 && row<n &&  /* If the position belongs */
            col>=0 && col<n)    /* to the grid */
        {
          j = 1;                  /* ... then it is a cell */

          block = row - row%hb +
                  col/wb;         /* Number of block */

          k = i * nGroupsPerGrid;

          g = b->pGr + k + block;             /* Add cell to */
          g->ppCl[g->nCl++] = b->pCl + cell;  /* block group */

          g = b->pGr + k + n + row;           /* Add cell to */
          g->ppCl[g->nCl++] = b->pCl + cell;  /* row group */

          g = b->pGr + k + 2*n + col;         /* Add cell to */
          g->ppCl[g->nCl++] = b->pCl + cell;  /* column goup */

          if (diag && row==col)               /* Add cell to */
          {                                   /* "backslash" */
            g = b->pGr + k + 3*n;             /* diag. group */
            g->ppCl[g->nCl++] = b->pCl + cell;
          }

          if (diag && row==n-1-col)           /* Add cell to */
          {                                   /* "slash" */
            g = b->pGr + k + 3*n+1;           /* diag. group */
            g->ppCl[g->nCl++] = b->pCl + cell;
          }
        }
      }

      if (j)        /* If there was a cell in pos. (x,y) */
      {
        b->pCl[cell].y = y;             /* Coordinates */
        b->pCl[cell].x = x;

        b->pppCl[y][x] = b->pCl + cell; /* Pointer in map */

        cell ++;                        /* Next cell */
      }
    }

  if (!CompleteBoard(b))
  {
    DestroyBoard (b);
    return NULL;
  }

  return b;
}                             

/*///////////////////////////////////////////////////////////////// */

t_board * ConstructSamuraiBoard (
                             char wb,     /* Block width */
                             char hb,     /* Block height */
                             char diag)   /* TRUE == diagonals */
{
  int xg[5], yg[5];

  xg[4] = yg[4] = 0;

  xg[0] = - (hb-1);
  yg[0] = - (wb-1);

  xg[1] =    hb-1;
  yg[1] = - (wb-1);

  xg[2] = - (hb-1);
  yg[2] =    wb-1;

  xg[3] =    hb-1;
  yg[3] =    wb-1;

  return ConstructCustomBoard (wb, hb, diag, 5, xg, yg);
}

/*///////////////////////////////////////////////////////////////// */

