/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "settingsdialog.h"

#include <QGridLayout>
#include <QCheckBox>
#include <QDialogButtonBox>
#include <QLabel>
#include <QPushButton>
#include <QDebug>

#ifdef Q_WS_MAEMO_5
    #include <QtMaemo5/QMaemo5ListPickSelector>
    #include <QtMaemo5/QMaemo5ValueButton>
#else
    #include <QComboBox>
#endif

#include <QStandardItemModel>

SettingsDialog::SettingsDialog(QWidget *parent) :
    QDialog(parent)
{

    setWindowTitle("Get Me Wheels - " + tr("Settings"));
    QHBoxLayout *hLayout = new QHBoxLayout();
    setLayout(hLayout);

    QGridLayout *layout = new QGridLayout();
    hLayout->addLayout(layout);

    hLayout->addSpacerItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));
    m_buttonBox = new QDialogButtonBox(Qt::Vertical);
    m_okButton = m_buttonBox->addButton(QDialogButtonBox::Ok);
    m_buttonBox->addButton(QDialogButtonBox::Cancel);
    hLayout->addWidget(m_buttonBox);

    m_locationModel = new QStandardItemModel();
    m_mapProviderModel = new QStandardItemModel();

#ifdef Q_WS_MAEMO_5
    m_okButton->setEnabled(false);

    m_locationSelector = new QMaemo5ListPickSelector();
    m_locationSelector->setModel(m_locationModel);

    QMaemo5ValueButton *locationButton = new QMaemo5ValueButton(tr("Location"));
    locationButton->setValueLayout(QMaemo5ValueButton::ValueBesideText);
    locationButton->setPickSelector(m_locationSelector);
    connect(m_locationSelector, SIGNAL(selected(QString)), SLOT(checkForComplete()));
    layout->addWidget(locationButton);

    m_mapProviderSelector = new QMaemo5ListPickSelector();
    m_mapProviderSelector->setModel(m_mapProviderModel);

    QMaemo5ValueButton *mapProviderButton = new QMaemo5ValueButton(tr("Map Provider"));
    mapProviderButton->setValueLayout(QMaemo5ValueButton::ValueBesideText);
    mapProviderButton->setPickSelector(m_mapProviderSelector);
    connect(m_mapProviderSelector, SIGNAL(selected(QString)), SLOT(checkForComplete()));
    layout->addWidget(mapProviderButton);

    m_accountButton = new QMaemo5ValueButton(tr("Account"));
    m_accountButton->setValueText(tr("Setup..."));
    m_accountButton->setValueLayout(QMaemo5ValueButton::ValueBesideText);
    connect(m_accountButton, SIGNAL(clicked()), SIGNAL(setupAccount()));

    layout->addWidget(m_accountButton);


#else

    layout->addWidget(new QLabel(tr("Location")), 0, 0);
    m_locationBox = new QComboBox();
    m_locationBox->setModel(m_locationModel);
    layout->addWidget(m_locationBox, 0, 1);

    m_accountLabel = new QLabel(tr("Account: %1").arg(m_accountName));
    layout->addWidget(m_accountLabel);
    QPushButton *button = new QPushButton(m_accountName.isEmpty() ? "Setup..." : m_accountName);
    connect(button, SIGNAL(clicked()), SIGNAL(setupAccount()));
    layout->addWidget(button);

    layout->addWidget(new QLabel(tr("Map Provider")));
    m_mapProviderBox = new QComboBox();
    m_mapProviderBox->setModel(m_mapProviderModel);
    layout->addWidget(m_mapProviderBox);
#endif

    m_cbEnableRotation = new QCheckBox(tr("Enable rotation"));
    layout->addWidget(m_cbEnableRotation);

    connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
    connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
}

void SettingsDialog::setRotationEnabled(bool enabled){
    m_cbEnableRotation->setChecked(enabled);
}

bool SettingsDialog::rotationEnabled(){
    return m_cbEnableRotation->isChecked();
}

void SettingsDialog::setLocations(const QStringList &locations, const QString &currentSelection) {
    m_locationModel->clear();
    foreach(const QString &location, locations) {
        m_locationModel->appendRow(new QStandardItem(location));
        if(location == currentSelection) {
#ifdef Q_WS_MAEMO_5
            m_locationSelector->setCurrentIndex(m_locationModel->rowCount() - 1);
#else
            m_locationBox->setCurrentIndex(m_locationModel->rowCount() - 1);
#endif
        }
    }
}

QString SettingsDialog::location() {
#ifdef Q_WS_MAEMO_5
    qDebug() << "location is" << m_locationSelector->currentValueText();
    return m_locationSelector->currentValueText();
#else
    return m_locationBox->currentText();
#endif
}

void SettingsDialog::setMapProviders(const QStringList &providers, const QString &currentProvider) {
    m_mapProviderModel->clear();
    foreach(const QString &provider, providers) {
        m_mapProviderModel->appendRow(new QStandardItem(provider));
        if(provider == currentProvider) {
#ifdef Q_WS_MAEMO_5
            m_mapProviderSelector->setCurrentIndex(m_mapProviderModel->rowCount() - 1);
#else
            m_mapProviderBox->setCurrentIndex(m_mapProviderModel->rowCount() - 1);
#endif
        }
    }
}

QString SettingsDialog::mapProvider() {
#ifdef Q_WS_MAEMO_5
    return m_mapProviderSelector->currentValueText();
#else
    return m_mapProviderBox->currentText();
#endif
}

void SettingsDialog::setAccountName(const QString &name)
{
    qDebug() << "settings account name" << name;
    m_accountName = name;
#ifdef Q_WS_MAEMO_5
    m_accountButton->setValueText(m_accountName);
#else
    m_accountLabel->setText(tr("Account: %1").arg(m_accountName));
#endif
}

void SettingsDialog::checkForComplete()
{
#ifdef Q_WS_MAEMO_5
    if(m_mapProviderSelector->currentValueText().isEmpty() || m_locationSelector->currentValueText().isEmpty()) {
        m_okButton->setEnabled(false);
    } else {
        m_okButton->setEnabled(true);
    }
#endif
}
