/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "itemdialog.h"
#include "data/gmwvehicle.h"
#include "data/gmwparkingspot.h"

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QDebug>
#include <QPushButton>
#include <QStackedWidget>
#include <QMap>
#include <QLabel>
#include <QAction>

ItemDialog::ItemDialog(QWidget *parent, Qt::WindowFlags f) :
    QDialog(parent, f)
{
    setAttribute(Qt::WA_DeleteOnClose, true);
    setMaximumSize(parentWidget()->size());

    m_masterLayout = new QBoxLayout(QBoxLayout::TopToBottom);
    m_masterLayout->setMargin(0);

    m_stackWidget = new QStackedWidget(this);

    m_masterLayout->addWidget(m_stackWidget);

    setLayout(m_masterLayout);

    // Add the buttons for scrolling
    m_buttonLayout = new QBoxLayout(QBoxLayout::RightToLeft);

    m_pbNext = new QPushButton();
    m_pbNext->setIcon(QIcon(":/arrow-right.png"));
    m_pbNext->setEnabled(false);
    connect(m_pbNext, SIGNAL(clicked()), this, SLOT(nextPage()));
    m_buttonLayout->addWidget(m_pbNext);

#ifndef Q_OS_SYMBIAN
    m_pbGoTo = new QPushButton();
    m_pbGoTo->setIcon(QIcon(":/go-last.png"));
    connect(m_pbGoTo, SIGNAL(clicked()), this, SLOT(slotGoTo()));
    m_buttonLayout->addWidget(m_pbGoTo);
#else
    QAction* backButton = new QAction(tr("Back"), this);
    connect(backButton, SIGNAL(triggered()), SLOT(close()));
    backButton->setSoftKeyRole(QAction::NegativeSoftKey);

    QAction* gotoButton = new QAction(tr("Go To"), this);
    connect(gotoButton, SIGNAL(triggered()), SLOT(slotGoTo()));
    gotoButton->setSoftKeyRole(QAction::PositiveSoftKey);

    QList<QAction *> softKeys;
    softKeys.append(backButton);
    softKeys.append(gotoButton);

    /* set new soft keys */
    addActions(softKeys);

    setMaximumSize(parent->size());
#endif

    m_pbPrevious = new QPushButton();
    m_pbPrevious->setIcon(QIcon(":/arrow-left.png"));
    m_pbPrevious->setEnabled(false);
    connect(m_pbPrevious, SIGNAL(clicked()), this, SLOT(previousPage()));
    m_buttonLayout->addWidget(m_pbPrevious);
    m_masterLayout->addLayout(m_buttonLayout);    
}

ItemDialog::~ItemDialog() {
    qDebug() << "~ItemDialog";
}

QString ItemDialog::smileyName(GMWVehicle::State state) {
    switch(state) {
    case GMWVehicle::StateExcellent:
        return ":/face-smile-big.png";
    case GMWVehicle::StateGood:
        return ":/face-smile.png";
    case GMWVehicle::StateSatisfied:
        return ":/face-plain.png";
    case GMWVehicle::StateIssuesExist:
        return ":/face-sad.png";
    case GMWVehicle::StateUnacceptable:
        return ":/face-surprise.png";
    case GMWVehicle::StateUnknown:
        return ":/unknown.png";
    }
    return QString();
}

QString ItemDialog::statusText(GMWVehicle::State state) {
    switch(state) {
    case GMWVehicle::StateExcellent:
        return tr("Excellent");
    case GMWVehicle::StateGood:
        return tr("Good");
    case GMWVehicle::StateSatisfied:
        return tr("Satisfied");
    case GMWVehicle::StateIssuesExist:
        return tr("Issues Exist");
    case GMWVehicle::StateUnacceptable:
        return tr("Unacceptable");
    case GMWVehicle::StateUnknown:
        return tr("Unknown");
    }
    return QString();
}

void ItemDialog::addObject(GMWItem *object) {
    int index = m_stackWidget->insertWidget(m_stackWidget->count(), createPage(object));
    m_objectMap.insert(index, object);

    if(m_objectMap.count() > 1) {
        m_pbNext->setEnabled(true);
    }
}

QWidget *ItemDialog::createPage(GMWItem *object) {
    QWidget *page = new QWidget();
    page->setMaximumSize(parentWidget()->size());

    QHBoxLayout *layout = new QHBoxLayout();
    page->setLayout(layout);
    QVBoxLayout *vLayout1 = new QVBoxLayout();
    QVBoxLayout *vLayout2 = new QVBoxLayout();

    vLayout1->addWidget(new QLabel(tr("Name:")));
    QLabel *nameLabel = new QLabel(object->name());
    nameLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    vLayout2->addWidget(nameLabel);

    vLayout1->addWidget(new QLabel(tr("Address:")));
    vLayout2->addWidget(new QLabel(object->address()));

    vLayout1->addWidget(new QLabel(tr("Distance:")));

    QHBoxLayout *distanceLayout = new QHBoxLayout();

    QLabel *azimuthLabel = new QLabel();
    azimuthLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
    if (!object->azimuthImage().isNull()) {
        azimuthLabel->setPixmap(object->azimuthImage());
    } else {
        azimuthLabel->setVisible(false);
    }
    m_azimuthLabelMap.insert(object, azimuthLabel);
    distanceLayout->addWidget(azimuthLabel);

    QLabel *distanceLabel = new QLabel(object->distanceString());
    m_distanceLabelMap.insert(object, distanceLabel);
    distanceLayout->addWidget(distanceLabel);

    vLayout2->addLayout(distanceLayout);

    switch(object->objectType()){
    case GMWItem::TypeGasStation:
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
        break;
    case GMWItem::TypeParkingLot: {
        GMWParkingSpot *parking = dynamic_cast<GMWParkingSpot*>(object);
        QHBoxLayout *parkLayout = new QHBoxLayout();
        vLayout1->addWidget(new QLabel(tr("Capacity:")));
        QLabel *parkIcon = new QLabel();
        parkIcon->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        if(parking->capacityUsed() < parking->capacityTotal()){
            parkIcon->setPixmap(QPixmap(":/parking_free"));
        } else {
            parkIcon->setPixmap(QPixmap(":/parking_full"));
        }
        parkLayout->addWidget(parkIcon);
        parkLayout->addWidget(new QLabel(
                QString::number(parking->capacityTotal() - parking->capacityUsed())
                + '/' + QString::number(parking->capacityTotal())
                + " (" + tr("free/total") + ')'));
        vLayout2->addLayout(parkLayout);
        vLayout1->addWidget(new QLabel(this));
        vLayout1->addWidget(new QLabel(this));
        }
        break;
    case GMWItem::TypeVehicle: {
        GMWVehicle *car = dynamic_cast<GMWVehicle*>(object);

        vLayout1->addWidget(new QLabel(tr("Fuel level:")));
        QHBoxLayout *fuelLayout = new QHBoxLayout();
        QLabel *fuelLabel = new QLabel();
        if(car->fuelLevel() < 10){
            fuelLabel->setPixmap(QPixmap(":/fuel_red.png"));
        } else if (car->fuelLevel() < 25) {
            fuelLabel->setPixmap(QPixmap(":/fuel_orange.png"));
        } else {
            fuelLabel->setPixmap(QPixmap(":/fuel_green.png"));
        }
        fuelLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        fuelLayout->addWidget(fuelLabel);
        fuelLayout->addWidget(new QLabel(QString::number(car->fuelLevel()) + "%"));
        vLayout2->addLayout(fuelLayout);

        vLayout1->addWidget(new QLabel(tr("Interior state: ")));

        QHBoxLayout *statusLayout1 = new QHBoxLayout();
        QLabel *interiorLabel = new QLabel();
        interiorLabel->setPixmap(QPixmap(smileyName(car->interiorState())));
        interiorLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        statusLayout1->addWidget(interiorLabel);
        statusLayout1->addWidget(new QLabel(statusText(car->interiorState())));
        vLayout2->addLayout(statusLayout1);

        vLayout1->addWidget(new QLabel(tr("Exterior state: ")));

        QHBoxLayout *statusLayout2 = new QHBoxLayout();
        QLabel *exteriorLabel = new QLabel();
        exteriorLabel->setPixmap(QPixmap(smileyName(car->exteriorState())));
        exteriorLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
        statusLayout2->addWidget(exteriorLabel);
        statusLayout2->addWidget(new QLabel(statusText(car->exteriorState())));
        vLayout2->addLayout(statusLayout2);
        }
        break;

    case GMWItem::TypeUnknown:
        break;
    }

    vLayout1->addStretch(0);
    vLayout2->addStretch(0);

    layout->addLayout(vLayout1);
    layout->addLayout(vLayout2);
    return page;
}

void ItemDialog::nextPage(){
    if(m_stackWidget->currentIndex() < (m_stackWidget->count() - 1)) {
        m_stackWidget->setCurrentIndex(m_stackWidget->currentIndex() + 1);

        updateWindowTitle(m_objectMap[m_stackWidget->currentIndex()]);
    }
    if(m_stackWidget->currentIndex() == m_stackWidget->count() - 1){
        m_pbNext->setEnabled(false);
    } else {
        m_pbNext->setEnabled(true);
    }
    m_pbPrevious->setEnabled(true);
}

void ItemDialog::previousPage() {
    if(m_stackWidget->currentIndex() > 0) {
        m_stackWidget->setCurrentIndex(m_stackWidget->currentIndex() - 1);
        updateWindowTitle(m_objectMap[m_stackWidget->currentIndex()]);        
    }
    if(m_stackWidget->currentIndex() == 0){
        m_pbPrevious->setEnabled(false);
    } else {
        m_pbPrevious->setEnabled(true);
    }
    m_pbNext->setEnabled(true);
}

void ItemDialog::updateWindowTitle(GMWItem *object) {
    switch(m_objectMap[m_stackWidget->currentIndex()]->objectType()){
    case GMWItem::TypeGasStation:
        setWindowTitle(tr("Gas Station"));
        break;
    case GMWItem::TypeParkingLot:
        setWindowTitle(tr("Parking Spot"));
        break;
    case GMWItem::TypeVehicle:
        setWindowTitle(tr("Vehicle"));
        break;
    case GMWItem::TypeUnknown:
        setWindowTitle(tr("Unknown"));
        break;
    }
    setWindowIcon(QIcon(object->image()));
}

void ItemDialog::slotGoTo() {
    close();
    emit zoomTo(m_objectMap[m_stackWidget->currentIndex()]->location());
}

void ItemDialog::goToPage(int page) {
    if(m_stackWidget->count() > page) {
        m_stackWidget->setCurrentIndex(page);
    }
    if(page > 0) {
        m_pbPrevious->setEnabled(true);
    } else {
        m_pbPrevious->setEnabled(false);
    }
    if(page < m_stackWidget->count() - 1) {
        m_pbNext->setEnabled(true);
    } else {
        m_pbNext->setEnabled(false);
    }
}

void ItemDialog::currentPositionChanged()
{
    foreach (GMWItem *obj, m_objectMap) {
        m_distanceLabelMap[obj]->setText(obj->distanceString());
        m_azimuthLabelMap[obj]->setPixmap(obj->azimuthImage());
        m_azimuthLabelMap[obj]->setVisible(true);
    }
}
