/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include <QtCore/QDebug>

#include <QtGui/QResizeEvent>

#include "gmwmap.h"

GMWMap::GMWMap(GMWItemSortFilterProxyModel *model, QWidget *parent):
    QWidget(parent),
    m_model(model),
    m_tracking(false)
{
    // connect model to virtual functions
    connect(m_model, SIGNAL(rowsInserted(const QModelIndex &, int, int)), this, SLOT(rowsInserted(const QModelIndex &, int, int)));
    connect(m_model, SIGNAL(rowsAboutToBeRemoved(const QModelIndex &, int, int)), this, SLOT(rowsAboutToBeRemoved(const QModelIndex &, int, int)));

    m_layout = new QBoxLayout(QBoxLayout::BottomToTop);
    m_layout->setContentsMargins(0,0,0,0);
    setLayout(m_layout);

    // Buttons
    m_pbZoomIn = new QPushButton();
    m_pbZoomIn->setIcon(QIcon(":/zoom-in.png"));
    m_pbZoomIn->setIconSize(QSize(32,32));
    m_pbZoomIn->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    connect(m_pbZoomIn, SIGNAL(clicked()), SLOT(zoomIn()));

    m_pbZoomOut = new QPushButton();
    m_pbZoomOut->setIcon(QIcon(":/zoom-out.png"));
    m_pbZoomOut->setIconSize(QSize(32,32));
    m_pbZoomOut->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    connect(m_pbZoomOut, SIGNAL(clicked()), SLOT(zoomOut()));

    m_pbGoToPos = new QPushButton();
    m_pbGoToPos->setIcon(QIcon(QPixmap(":/currentLoc.png")));
    m_pbGoToPos->setIconSize(QSize(32,32));
    m_pbGoToPos->setEnabled(false);
    m_pbGoToPos->setCheckable(true);
    m_pbGoToPos->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    connect(m_pbGoToPos, SIGNAL(clicked(bool)), SLOT(setTracking(bool)));

    m_pbShowList = new QPushButton();
    m_pbShowList->setIcon(QIcon(":/list"));
    m_pbShowList->setIconSize(QSize(32,32));
    m_pbShowList->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    connect(m_pbShowList, SIGNAL(clicked()), SIGNAL(showList()));

    m_buttons = new QWidget();
    m_buttonLayout = new QBoxLayout(QBoxLayout::LeftToRight);
    m_buttonLayout->setContentsMargins(0,0,0,0);
    m_buttons->setLayout(m_buttonLayout);
    m_buttons->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);

    m_buttonLayout->addWidget(m_pbZoomIn);
    m_buttonLayout->addWidget(m_pbZoomOut);
    m_buttonLayout->addWidget(m_pbGoToPos);
    m_buttonLayout->addWidget(m_pbShowList);

    layout()->addWidget(m_buttons);
}

void GMWMap::resizeEvent(QResizeEvent *event)
{
    if (event->size().width() > event->size().height()) { // Landscape
        qDebug() << "Landscape";
        m_layout->setDirection(QBoxLayout::RightToLeft);
        m_buttonLayout->setDirection(QBoxLayout::TopToBottom);
        m_buttons->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    } else { // Portrait
        qDebug() << "Portrait";
        m_layout->setDirection(QBoxLayout::BottomToTop);
        m_buttonLayout->setDirection(QBoxLayout::LeftToRight);
        m_buttons->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    }
}

void GMWMap::positionUpdated(const QGeoPositionInfo &info)
{
    m_currentPosition = info;
    m_pbGoToPos->setEnabled(true);
    if (m_tracking) {
        moveTo(info.coordinate());
    }
}

void GMWMap::enableTracking()
{
    m_tracking = true;
    m_pbGoToPos->setChecked(true);
    moveTo(m_currentPosition.coordinate());
}

void GMWMap::disableTracking()
{
    m_pbGoToPos->setChecked(false);
    m_tracking = false;
}

void GMWMap::setTracking(bool tracking)
{
    tracking ? enableTracking() : disableTracking();
}
